//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
//
#include "com_microsoft_cognitiveservices_speech_transcription_Meeting.h"
#include "jni_utils.h"
#include "speechapi_c_meeting.h"

const size_t maxCharCount = 1024;

/*
 * Class:     com_microsoft_cognitiveservices_speech_transcription_Meeting
 * Method:    createMeetingFromConfig
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/IntRef;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_createMeetingFromConfig
  (JNIEnv *env, jclass, jobject meetingHandle, jobject speechConfigHandle, jstring meetingId)
{
    if (meetingId == NULL)
    {
        return (jlong)SPXERR_INVALID_ARG;
    }
    SPXMEETINGHANDLE meetHandle = SPXHANDLE_INVALID;
    jlong speechConfig = GetObjectHandle(env, speechConfigHandle);
    const char* id = GetStringUTFChars(env, meetingId);
    if (id == NULL)
    {
        return (jlong)SPXERR_OUT_OF_MEMORY;
    }
    SPXHR hr = meeting_create_from_config(&meetHandle, (SPXSPEECHCONFIGHANDLE)speechConfig, id);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, meetingHandle, (jlong)meetHandle);
    }
    ReleaseStringUTFChars(env, meetingId, id);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_transcription_Meeting
 * Method:    getMeetingId
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/StringRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_getMeetingId
  (JNIEnv *env, jobject , jobject meetingHandle, jobject meetingIdStr)
{
    char sz[maxCharCount + 1] = {};
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    SPXHR hr = meeting_get_meeting_id((SPXMEETINGHANDLE)meetHandle, sz, maxCharCount);
    if (SPX_SUCCEEDED(hr))
    {
        hr = SetStringObjectHandle(env, meetingIdStr, sz);
    }
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_transcription_Meeting
 * Method:    addParticipant
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_addParticipant
  (JNIEnv *env, jobject , jobject meetingHandle, jobject participantHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    jlong partHandle = GetObjectHandle(env, participantHandle);
    SPXHR hr = meeting_update_participant((SPXMEETINGHANDLE)meetHandle, true, (SPXPARTICIPANTHANDLE)partHandle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_transcription_Meeting
 * Method:    addParticipantByUser
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_addParticipantByUser
  (JNIEnv *env, jobject , jobject meetingHandle, jobject userHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    jlong usrHandle = GetObjectHandle(env, userHandle);
    SPXHR hr = meeting_update_participant_by_user((SPXMEETINGHANDLE)meetHandle, true, (SPXUSERHANDLE)usrHandle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_transcription_Meeting
 * Method:    removeParticipant
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_removeParticipant
  (JNIEnv *env, jobject , jobject meetingHandle, jobject participantHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    jlong partHandle = GetObjectHandle(env, participantHandle);
    SPXHR hr = meeting_update_participant((SPXMEETINGHANDLE)meetHandle, false, (SPXPARTICIPANTHANDLE)partHandle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_transcription_Meeting
 * Method:    removeParticipantByUser
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_removeParticipantByUser
  (JNIEnv *env, jobject , jobject meetingHandle, jobject userHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    jlong usrHandle = GetObjectHandle(env, userHandle);
    SPXHR hr = meeting_update_participant_by_user((SPXMEETINGHANDLE)meetHandle, false, (SPXUSERHANDLE)usrHandle);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_transcription_Meeting
 * Method:    removeParticipantByUserId
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Ljava/lang/String;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_removeParticipantByUserId
  (JNIEnv *env, jobject , jobject meetingHandle, jstring userId)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    const char* id = GetStringUTFChars(env, userId);
    SPXHR hr = meeting_update_participant_by_user_id((SPXMEETINGHANDLE)meetHandle, false, id);
    ReleaseStringUTFChars(env, userId, id);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_transcription_Meeting
 * Method:    endMeeting
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_endMeeting
  (JNIEnv *env, jobject , jobject meetingHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    SPXHR hr = meeting_end_meeting((SPXMEETINGHANDLE)meetHandle);
    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_startMeeting
  (JNIEnv *env, jobject , jobject meetingHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    SPXHR hr = meeting_start_meeting((SPXMEETINGHANDLE)meetHandle);
    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_deleteMeeting
  (JNIEnv *env, jobject , jobject meetingHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    SPXHR hr = meeting_delete_meeting((SPXMEETINGHANDLE)meetHandle);
    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_lockMeeting
  (JNIEnv *env, jobject , jobject meetingHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    SPXHR hr = meeting_lock_meeting((SPXMEETINGHANDLE)meetHandle);
    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_unlockMeeting
  (JNIEnv *env, jobject , jobject meetingHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    SPXHR hr = meeting_unlock_meeting((SPXMEETINGHANDLE)meetHandle);
    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_muteAll
  (JNIEnv *env, jobject , jobject meetingHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    SPXHR hr = meeting_mute_all_participants((SPXMEETINGHANDLE)meetHandle);
    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_unmuteAll
  (JNIEnv *env, jobject , jobject meetingHandle)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    SPXHR hr = meeting_unmute_all_participants((SPXMEETINGHANDLE)meetHandle);
    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_muteParticipant
  (JNIEnv *env, jobject , jobject meetingHandle, jstring userId)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    const char* id = GetStringUTFChars(env, userId);
    SPXHR hr = meeting_mute_participant((SPXMEETINGHANDLE)meetHandle, id);
    ReleaseStringUTFChars(env, userId, id);
    return (jlong)hr;
}

JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_unmuteParticipant
  (JNIEnv *env, jobject , jobject meetingHandle, jstring userId)
{
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    const char* id = GetStringUTFChars(env, userId);
    SPXHR hr = meeting_unmute_participant((SPXMEETINGHANDLE)meetHandle, id);
    ReleaseStringUTFChars(env, userId, id);
    return (jlong)hr;
}

/*
 * Class:     com_microsoft_cognitiveservices_speech_transcription_Meeting
 * Method:    getPropertyBag
 * Signature: (Lcom/microsoft/cognitiveservices/speech/util/SafeHandle;Lcom/microsoft/cognitiveservices/speech/util/IntRef;)J
 */
JNIEXPORT jlong JNICALL Java_com_microsoft_cognitiveservices_speech_transcription_Meeting_getPropertyBag
  (JNIEnv *env, jobject , jobject meetingHandle, jobject propertyHandle)
{
    SPXPROPERTYBAGHANDLE propHandle = SPXHANDLE_INVALID;
    jlong meetHandle = GetObjectHandle(env, meetingHandle);
    SPXHR hr = meeting_get_property_bag((SPXMEETINGHANDLE)meetHandle, &propHandle);
    if (SPX_SUCCEEDED(hr))
    {
        SetObjectHandle(env, propertyHandle, (jlong)propHandle);
    }
    return (jlong)hr;
}
