//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.util;

import java.lang.AutoCloseable;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.Contracts;

/*! \cond INTERNAL */

/**
 * Internal SafeHandle class
 * Note: close() must be called in order to release underlying resources held by the object.
 */

public class SafeHandle implements AutoCloseable {

    /**
     * A safe handle that implements AutoCloseable
     * @param val The value.
     * @param type the type
     */
    public SafeHandle(long val, SafeHandleType type) {
        //System.out.println("Java SafeHandle constructor, value is: " + String.valueOf(val) + ", HandleType is: " + String.valueOf(type));
        value = val;
        handleType = type;
    }

    /*! \cond PROTECTED */

    @Override
    /**
     * finalizer for the safe handle.
     */
    protected void finalize() throws Throwable {
        this.close();
    }

    /*! \endcond */

    /**
     * Sets the long value
     * @param newValue Long value to set.
     */
    public void setValue(long newValue) {
        //System.out.println("Java SafeHandle setValue to " + String.valueOf(newValue) + ", HandleType is: " + String.valueOf(handleType));
        this.value = newValue;
    }

    /**
     * Gets the long value
     * @return long value
     */
    public long getValue() {
        return this.value;
    }

     /**
     * Dispose of associated resources.
     * Note: close() must be called in order to release underlying resources held by the object.
     */
    @Override
    public void close() {
        //System.out.println("Java SafeHandle close, value is: " + String.valueOf(value) + ", HandleType is: " + String.valueOf(handleType));
        if (value != 0) {
            if (handleType == SafeHandleType.Event) {
                releaseEventHandle(value);
            }
            else if (handleType == SafeHandleType.Recognizer) {
                releaseRecognizerHandle(value);
            }
            else if (handleType == SafeHandleType.SpeechConfig) {
                releaseSpeechConfigHandle(value);
            }
            else if (handleType == SafeHandleType.PropertyCollection) {
                releasePropertyHandle(value);
            }
            else if (handleType == SafeHandleType.RecognitionResult) {
                releaseRecognizerResultHandle(value);
            }
            else if (handleType == SafeHandleType.AudioConfig) {
                releaseAudioConfigHandle(value);
            }
            else if (handleType == SafeHandleType.Connection) {
                releaseConnectionHandle(value);
            }
            else if (handleType == SafeHandleType.ConnectionMessage) {
                releaseConnectionMessageHandle(value);
            }
            else if (handleType == SafeHandleType.ConnectionMessageEvent) {
                releaseConnectionMessageEventHandle(value);
            }
            else if (handleType == SafeHandleType.AudioInputStream) {
                releaseAudioInputStreamHandle(value);
            }
            else if (handleType == SafeHandleType.AudioStreamFormat) {
                releaseAudioStreamFormatHandle(value);
            }
            else if (handleType == SafeHandleType.KeywordModel) {
                releaseKeywordModelHandle(value);
            }
            else if (handleType == SafeHandleType.LanguageUnderstandingModel) {
                releaseLanguageUnderstandingModelHandle(value);
            }
            else if (handleType == SafeHandleType.IntentTrigger) {
                releaseIntentTriggerHandle(value);
            }
            else if (handleType == SafeHandleType.User) {
                releaseUserHandle(value);
            }
            else if (handleType == SafeHandleType.Participant) {
                releaseParticipantHandle(value);
            }
            else if (handleType == SafeHandleType.Conversation) {
                releaseConversationHandle(value);
            }
            else if (handleType == SafeHandleType.Meeting) {
                releaseMeetingHandle(value);
            }            
            else if (handleType == SafeHandleType.DialogServiceConnector) {
                releaseDialogServiceConnectorHandle(value);
            }
            else if (handleType == SafeHandleType.TurnStatusReceivedEvent) {
                releaseTurnStatusReceivedEventHandle(value);
            }
            else if (handleType == SafeHandleType.ActivityReceivedEvent) {
                releaseActivityReceivedEventHandle(value);
            }
            else if (handleType == SafeHandleType.AudioOutputStream) {
                releaseAudioOutputStreamHandle(value);
            }
            else if (handleType == SafeHandleType.AutoDetectSourceLanguageConfig) {
                releaseAutoDetectSourceLanguageConfigHandle(value);
            }
            else if (handleType == SafeHandleType.SourceLanguageConfig) {
                releaseSourceLanguageConfigHandle(value);
            }
            else if (handleType == SafeHandleType.SynthesisEvent) {
                releaseSynthesisEventHandle(value);
            }
            else if (handleType == SafeHandleType.SynthesisResult) {
                releaseSynthesisResultHandle(value);
            }
            else if (handleType == SafeHandleType.VoiceInfo) {
                releaseVoiceInfoHandle(value);
            }
            else if (handleType == SafeHandleType.AudioDataStream) {
                releaseAudioDataStreamHandle(value);
            }
            else if (handleType == SafeHandleType.Synthesizer) {
                releaseSynthesizerHandle(value);
            }
            else if (handleType == SafeHandleType.Grammar) {
                releaseGrammarHandle(value);
            }
            else if (handleType == SafeHandleType.TranslationSynthesis) {
                releaseTranslationSynthesisHandle(value);
            }
            else if (handleType == SafeHandleType.ConversationTranslator) {
                releaseConversationTranslatorHandle(value);
            }
            else if (handleType == SafeHandleType.KeywordRecognizer) {
                releaseKeywordRecognizerHandle(value);
            }
            else if (handleType == SafeHandleType.PronunciationAssessmentConfig) {
                releasePronunciationAssessmentConfig(value);
            }
            else if (handleType == SafeHandleType.SpeakerRecognizer) {
                releaseSpeakerRecognizerHandle(value);
            }
            else if (handleType == SafeHandleType.SpeakerIdentificationModel) {
                releaseSpeakerIdentificationModelHandle(value);
            }
            else if (handleType == SafeHandleType.SpeakerRecognitionResult) {
                releaseSpeakerRecognitionResultHandle(value);
            }
            else if (handleType == SafeHandleType.SpeakerVerificationModel) {
                releaseSpeakerVerificationModelHandle(value);
            }
            else if (handleType == SafeHandleType.VoiceProfileResult) {
                releaseVoiceProfileResultHandle(value);
            }
            else if (handleType == SafeHandleType.VoiceProfileEnrollmentResult) {
                releaseVoiceProfileEnrollmentResultHandle(value);
            }
            else if (handleType == SafeHandleType.VoiceProfilePhraseResult) {
                releaseVoiceProfilePhraseResultHandle(value);
            }
            else if (handleType == SafeHandleType.VoiceProfileClient) {
                releaseVoiceProfileClientHandle(value);
            }
            else if (handleType == SafeHandleType.VoiceProfile) {
                releaseVoiceProfileHandle(value);
            }
            else if (handleType == SafeHandleType.AudioProcessingOptions) {
                releaseAudioProcessingOptionsHandle(value);
            }
            else if (handleType == SafeHandleType.SpeechRecognitionModel) {
                releaseSpeechRecognitionModelHandle(value);
            }
            else if (handleType == SafeHandleType.SpeechTranslationModel) {
                releaseSpeechTranslationModelHandle(value);
            }
            else if (handleType == SafeHandleType.JSONVALUE) {
                releaseJsonValueHandle(value);
            }
            else if (handleType == SafeHandleType.JSONBUILDER) {
                releaseJsonBuilderHandle(value);
            }
            else {
                System.out.println("Java SafeHandle close, invalid handle value: " + String.valueOf(value));
            }
            value = 0;
            handleType = SafeHandleType.UnInitialized;
        }
    }

    private final native long releaseActivityReceivedEventHandle(long handle);
    private final native long releaseAudioConfigHandle(long handle);
    private final native long releaseAudioDataStreamHandle(long handle);
    private final native long releaseAudioInputStreamHandle(long handle);
    private final native long releaseAudioOutputStreamHandle(long handle);
    private final native long releaseAudioProcessingOptionsHandle(long handle);
    private final native long releaseAudioStreamFormatHandle(long handle);
    private final native long releaseAutoDetectSourceLanguageConfigHandle(long handle);
    private final native long releaseConnectionHandle(long handle);
    private final native long releaseConnectionMessageEventHandle(long handle);
    private final native long releaseConnectionMessageHandle(long handle);
    private final native long releaseConversationHandle(long handle);
    private final native long releaseMeetingHandle(long handle);
    private final native long releaseConversationTranslatorHandle(long handle);
    private final native long releaseDialogServiceConnectorHandle(long handle);
    private final native long releaseEventHandle(long handle);
    private final native long releaseGrammarHandle(long handle);
    private final native long releaseIntentTriggerHandle(long handle);
    private final native long releaseJsonBuilderHandle(long handle);
    private final native long releaseJsonValueHandle(long handle);
    private final native long releaseKeywordModelHandle(long handle);
    private final native long releaseKeywordRecognizerHandle(long handle);
    private final native long releaseLanguageUnderstandingModelHandle(long handle);
    private final native long releaseParticipantHandle(long handle);
    private final native long releasePronunciationAssessmentConfig(long handle);
    private final native long releasePropertyHandle(long handle);
    private final native long releaseRecognizerHandle(long handle);
    private final native long releaseRecognizerResultHandle(long handle);
    private final native long releaseSourceLanguageConfigHandle(long handle);
    private final native long releaseSpeakerIdentificationModelHandle(long handle);
    private final native long releaseSpeakerRecognitionResultHandle(long handle);
    private final native long releaseSpeakerRecognizerHandle(long handle);
    private final native long releaseSpeakerVerificationModelHandle(long handle);
    private final native long releaseSpeechConfigHandle(long handle);
    private final native long releaseSpeechRecognitionModelHandle(long handle);
    private final native long releaseSpeechTranslationModelHandle(long handle);
    private final native long releaseSynthesisEventHandle(long handle);
    private final native long releaseSynthesisResultHandle(long handle);
    private final native long releaseSynthesizerHandle(long handle);
    private final native long releaseTranslationSynthesisHandle(long handle);
    private final native long releaseTurnStatusReceivedEventHandle(long handle);
    private final native long releaseUserHandle(long handle);
    private final native long releaseVoiceInfoHandle(long handle);
    private final native long releaseVoiceProfileClientHandle(long handle);
    private final native long releaseVoiceProfileEnrollmentResultHandle(long handle);
    private final native long releaseVoiceProfileHandle(long handle);
    private final native long releaseVoiceProfilePhraseResultHandle(long handle);
    private final native long releaseVoiceProfileResultHandle(long handle);

    private long value = 0;
    private SafeHandleType handleType = SafeHandleType.UnInitialized;
}

/*! \endcond */
