//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//

package com.microsoft.cognitiveservices.speech.util;

import com.microsoft.cognitiveservices.speech.util.JsonBuilderJNI;
import com.microsoft.cognitiveservices.speech.util.JsonValueJNI;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;

/*! \cond INTERNAL */
/**
 * The type Json builder.
 */

public class JsonBuilder implements AutoCloseable {
    
    /**
     * Instantiates a new Json builder.
     */
    public JsonBuilder() {
        JsonBuilder newBuilder = JsonBuilderJNI.createBuilder();
        this.builderHandle = newBuilder.builderHandle;
        this.root = newBuilder.root;
    }

    /**
     * Instantiates a new Json builder.
     *
     * @param handle the handle
     */
    public JsonBuilder(SafeHandle handle, int root) {
       builderHandle = handle;
       this.root = root;
    }

    /**
     * Adds an item to the specified ajv item
     *
     * @param item the specified ajv item to add to
     * @param index the index to add the new item at
     * @param find the json key to associate with the new item
     * @return the ajv item added
     */
    public int addItem(int item, int index, String find) {
        return JsonBuilderJNI.addItem(builderHandle, item, index, find);
    }

    /**
     * Sets the json of the specified ajv item
     *
     * @param item the specified ajv item to set the value of
     * @param json the json value to associate with the item
     * @return HR indicating success or failure
     */
    public int setJson(int item, String json) {
        return JsonBuilderJNI.setItem(builderHandle, item, json, JsonValueJNI.ValueKind.OBJECT, null, false, 0, 0);
    }

    /**
     * Sets the string of the specified ajv item
     *
     * @param item the specified ajv item to set the value of
     * @param str the string value to associate with the item
     * @return HR indicating success or failure
     */
    public int setString(int item, String str) {
        return JsonBuilderJNI.setItem(builderHandle, item, null, JsonValueJNI.ValueKind.STRING, str, false, 0, 0);
    }

    /**
     * Sets the boolean of the specified ajv item
     *
     * @param item the specified ajv item to set the value of
     * @param value the boolean value to associate with the item
     * @return HR indicating success or failure
     */
    public int setBoolean(int item, boolean value) {
        return JsonBuilderJNI.setItem(builderHandle, item, null, JsonValueJNI.ValueKind.BOOLEAN, null, value, 0, 0);
    }

    /**
     * Sets the integer of the specified ajv item
     *
     * @param item the specified ajv item to set the value of
     * @param integer the integer value to associate with the item
     * @return HR indicating success or failure
     */
    public int setInteger(int item, int integer) {
        return JsonBuilderJNI.setItem(builderHandle, item, null, JsonValueJNI.ValueKind.NUMBER, null, false, integer, 0);
    }

    /**
     * Sets the double of the specified ajv item
     *
     * @param item the specified ajv item to set the value of
     * @param value the double value to associate with the item
     * @return HR indicating success or failure
     */
    public int setDouble(int item, double value) {
        return JsonBuilderJNI.setItem(builderHandle, item, null, JsonValueJNI.ValueKind.NUMBER, null, false, 0, value);
    }

    /**
     * Returns the string representation of the json.
     */
    @Override
    public String toString()
    {
        return JsonBuilderJNI.asJsonCopy(builderHandle, root);
    }

    /***
     * Close
     * @throws Exception will throw if private handle.close fails.
     */
    @Override
    public void close() throws Exception {
        if(builderHandle != null) {
            builderHandle.close();
            builderHandle = null;
        }
    }

    public int root;
    private SafeHandle builderHandle;
}

/*! \endcond */
