//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech.audio;

import com.microsoft.cognitiveservices.speech.util.Contracts;

/**
 * Represents the geometry of a microphone array.
 */
public class MicrophoneArrayGeometry {

    /**
     * Type of microphone array.
     */
    private MicrophoneArrayType microphoneArrayType;

    /**
     * Start angle for beamforming in degrees.
     */
    private int beamformingStartAngle;

    /**
     * End angle for beamforming in degrees.
     */
    private int beamformingEndAngle;

    /**
     * Coordinates of microphones in the microphone array.
     */
    private MicrophoneCoordinates[] microphoneCoordinates;

    /**
     * Creates an object of MicrophoneArrayGeometry.
     * Beamforming start angle is set to zero.
     * Beamforming end angle is set to 180 degrees if microphoneArrayType is Linear, otherwise it is set to 360 degrees.
     * @param microphoneArrayType Type of microphone array.
     * @param microphoneCoordinates Coordinates of microphones in the microphone array.
     */
    public MicrophoneArrayGeometry(MicrophoneArrayType microphoneArrayType, MicrophoneCoordinates[] microphoneCoordinates) {
        Contracts.throwIfNull(microphoneCoordinates, "microphoneCoordinates");
        this.microphoneArrayType = microphoneArrayType;
        beamformingStartAngle = 0;
        beamformingEndAngle = (microphoneArrayType == MicrophoneArrayType.Linear) ? 180 : 360;
        this.microphoneCoordinates = new MicrophoneCoordinates[microphoneCoordinates.length];
        for (int i = 0; i < microphoneCoordinates.length; i++) {
            this.microphoneCoordinates[i] = new MicrophoneCoordinates(microphoneCoordinates[i]);
        }
    }

    /**
     * Creates an object of MicrophoneArrayGeometry.
     * @param microphoneArrayType Type of microphone array.
     * @param beamformingStartAngle Start angle for beamforming in degrees.
     * @param beamformingEndAngle End angle for beamforming in degrees.
     * @param microphoneCoordinates Coordinates of microphones in the microphone array.
     */
    public MicrophoneArrayGeometry(MicrophoneArrayType microphoneArrayType, int beamformingStartAngle, int beamformingEndAngle, MicrophoneCoordinates[] microphoneCoordinates) {
        Contracts.throwIfNull(microphoneCoordinates, "microphoneCoordinates");
        if (beamformingStartAngle < 0) {
            throw new IllegalArgumentException("beamformingStartAngle cannot be negative");
        }
        if (beamformingEndAngle < 0) {
            throw new IllegalArgumentException("beamformingEndAngle cannot be negative");
        }
        this.microphoneArrayType = microphoneArrayType;
        this.beamformingStartAngle = beamformingStartAngle;
        this.beamformingEndAngle = beamformingEndAngle;
        this.microphoneCoordinates = new MicrophoneCoordinates[microphoneCoordinates.length];
        for (int i = 0; i < microphoneCoordinates.length; i++) {
            this.microphoneCoordinates[i] = new MicrophoneCoordinates(microphoneCoordinates[i]);
        }
    }

    /**
     * Returns the type of microphone array.
     * @return Type of microphone array.
     */
    public MicrophoneArrayType getMicrophoneArrayType() {
        return microphoneArrayType;
    }

    /**
     * Returns the start angle for beamforming in degrees.
     * @return Start angle for beamforming in degrees.
     */
    public int getBeamformingStartAngle() {
        return beamformingStartAngle;
    }

    /**
     * Returns the end angle for beamforming in degrees.
     * @return End angle for beamforming in degrees.
     */
    public int getBeamformingEndAngle() {
        return beamformingEndAngle;
    }

    /**
     * Returns the coordinates of microphones in the microphone array.
     * @return Coordinates of microphones in the microphone array.
     */
    public MicrophoneCoordinates[] getMicrophoneCoordinates() {
        MicrophoneCoordinates[] microphoneCoordinates = new MicrophoneCoordinates[this.microphoneCoordinates.length];
        for (int i = 0; i < microphoneCoordinates.length; i++) {
            microphoneCoordinates[i] = new MicrophoneCoordinates(this.microphoneCoordinates[i]);
        }
        return microphoneCoordinates;
    }

}
