//
// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE.md file in the project root for full license information.
//
package com.microsoft.cognitiveservices.speech;

import java.lang.AutoCloseable;

import com.microsoft.cognitiveservices.speech.util.Contracts;
import com.microsoft.cognitiveservices.speech.util.SafeHandle;
import com.microsoft.cognitiveservices.speech.util.SafeHandleType;
import com.microsoft.cognitiveservices.speech.util.IntRef;
import com.microsoft.cognitiveservices.speech.util.StringRef;

/**
 * Defines contents of speech synthesis word boundary event.
 * Added in version 1.7.0
 */
public class SpeechSynthesisWordBoundaryEventArgs {

    /*! \cond INTERNAL */

    SpeechSynthesisWordBoundaryEventArgs(long eventArgs) {
        Contracts.throwIfNull(eventArgs, "eventArgs");

        SafeHandle eventHandle = new SafeHandle(eventArgs, SafeHandleType.SynthesisEvent);

        StringRef stringRef = new StringRef("");
        Contracts.throwIfFail(getResultId(eventHandle, stringRef));
        this.resultId = stringRef.getValue();

        IntRef audioOffsetRef = new IntRef(0);
        IntRef durationRef = new IntRef(0);
        IntRef textOffsetRef = new IntRef(0);
        IntRef wordLengthRef = new IntRef(0);
        IntRef boundaryTypeRef = new IntRef(0);
        Contracts.throwIfFail(
                getWordBoundaryEventValues(eventHandle, audioOffsetRef, durationRef,
                        textOffsetRef, wordLengthRef, boundaryTypeRef));
        this.audioOffset = audioOffsetRef.getValue();
        this.duration = durationRef.getValue();
        this.textOffset = textOffsetRef.getValue();
        if (this.textOffset > Integer.MAX_VALUE) {
            this.textOffset = -1;
        }
        this.wordLength = wordLengthRef.getValue();
        this.boundaryType = SpeechSynthesisBoundaryType.values()[(int)boundaryTypeRef.getValue()];
        this.text = getTextFromHandle(eventHandle);
        eventHandle.close();
    }

    /*! \endcond */


    /**
     * Gets the unique ID of the result.
     * Added in version 1.25.0
     * @return Specifies unique ID of the result.
     */
    public String getResultId() {
        return this.resultId;
    }

    /**
     * Specifies current word's binary offset in output audio, by ticks (100ns).
     * @return Current word's binary offset in output audio, by ticks (100ns).
     */
    public long getAudioOffset() {
        return this.audioOffset;
    }

    /**
     * Specifies the audio duration of current event, by ticks (100ns).
     * Added in version 1.21.0
     * @return Audio duration of current event, by ticks (100ns).
     */
    public long getDuration() {
        return this.duration;
    }

    /**
     * Specifies current word's text offset in input text, by characters.
     * @return Current word's text offset in input text, by characters.
     */
    public long getTextOffset() {
        return this.textOffset;
    }

    /**
     * Specifies current word's length, by characters.
     * @return Current word's length, by characters.
     */
    public long getWordLength() {
        return this.wordLength;
    }

    /**
     * Specifies the text.
     * Added in version 1.21.0
     * @return The text.
     */
    public String getText() {
        return this.text;
    }

    /**
     * Specifies the boundary type.
     * Added in version 1.21.0
     * @return The boundary type.
     */
    public SpeechSynthesisBoundaryType getBoundaryType() {
        return this.boundaryType;
    }

    private final native long getResultId(SafeHandle eventHandle, StringRef resultIdRef);
    private final native long getWordBoundaryEventValues(SafeHandle eventHandle, IntRef audioOffset, IntRef duration,
            IntRef textOffset, IntRef wordLength, IntRef boundaryType);
    private final native String getTextFromHandle(SafeHandle eventHandle);

    private String resultId;
    private long audioOffset;
    private long duration;
    private long textOffset;
    private long wordLength;
    private String text;
    private SpeechSynthesisBoundaryType boundaryType;
}
