/*
 * Decompiled with CFR 0.152.
 */
package com.microsoft.azure.sdk.iot.device.transport.mqtt;

import com.microsoft.azure.sdk.iot.device.ClientConfiguration;
import com.microsoft.azure.sdk.iot.device.IotHubMessageResult;
import com.microsoft.azure.sdk.iot.device.IotHubStatusCode;
import com.microsoft.azure.sdk.iot.device.Message;
import com.microsoft.azure.sdk.iot.device.MessageType;
import com.microsoft.azure.sdk.iot.device.ProxySettings;
import com.microsoft.azure.sdk.iot.device.transport.HttpProxySocketFactory;
import com.microsoft.azure.sdk.iot.device.transport.IotHubConnectionStatus;
import com.microsoft.azure.sdk.iot.device.transport.IotHubListener;
import com.microsoft.azure.sdk.iot.device.transport.IotHubTransportConnection;
import com.microsoft.azure.sdk.iot.device.transport.IotHubTransportMessage;
import com.microsoft.azure.sdk.iot.device.transport.TransportException;
import com.microsoft.azure.sdk.iot.device.transport.mqtt.MqttDirectMethod;
import com.microsoft.azure.sdk.iot.device.transport.mqtt.MqttMessageListener;
import com.microsoft.azure.sdk.iot.device.transport.mqtt.MqttMessaging;
import com.microsoft.azure.sdk.iot.device.transport.mqtt.MqttTwin;
import com.microsoft.azure.sdk.iot.device.transport.mqtt.Socks5SocketFactory;
import com.microsoft.azure.sdk.iot.device.transport.mqtt.exceptions.PahoExceptionTranslator;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.Proxy;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import javax.net.SocketFactory;
import javax.net.ssl.SSLContext;
import org.apache.commons.lang3.tuple.Pair;
import org.eclipse.paho.client.mqttv3.MqttAsyncClient;
import org.eclipse.paho.client.mqttv3.MqttCallback;
import org.eclipse.paho.client.mqttv3.MqttClientPersistence;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;
import org.eclipse.paho.client.mqttv3.MqttMessage;
import org.eclipse.paho.client.mqttv3.persist.MemoryPersistence;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MqttIotHubConnection
implements IotHubTransportConnection,
MqttMessageListener {
    private static final Logger log = LoggerFactory.getLogger(MqttIotHubConnection.class);
    private static final String WS_SSL_PREFIX = "wss://";
    private static final String WEBSOCKET_RAW_PATH = "/$iothub/websocket";
    private static final String NO_CLIENT_CERT_QUERY_STRING = "?iothub-no-client-cert=true";
    private static final String SSL_PREFIX = "ssl://";
    private static final String SSL_PORT_SUFFIX = ":8883";
    private static final int MQTT_VERSION = 4;
    private static final boolean SET_CLEAN_SESSION = false;
    private static final String MODEL_ID = "model-id";
    private String connectionId;
    private String webSocketQueryString;
    private final Object mqttConnectionStateLock = new Object();
    private final ClientConfiguration config;
    private IotHubConnectionStatus state = IotHubConnectionStatus.DISCONNECTED;
    private IotHubListener listener;
    private final String clientId;
    private final String serverUri;
    private final MqttMessaging deviceMessaging;
    private final MqttTwin deviceTwin;
    private final MqttDirectMethod directMethod;
    private final Map<IotHubTransportMessage, Integer> receivedMessagesToAcknowledge = new ConcurrentHashMap<IotHubTransportMessage, Integer>();

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public MqttIotHubConnection(ClientConfiguration config) throws TransportException {
        String serviceParams;
        String clientUserAgentIdentifier;
        SSLContext sslContext;
        if (config == null) {
            throw new IllegalArgumentException("The ClientConfiguration cannot be null.");
        }
        if (config.getIotHubHostname() == null || config.getIotHubHostname().length() == 0) {
            throw new IllegalArgumentException("hostName cannot be null or empty.");
        }
        if (config.getDeviceId() == null || config.getDeviceId().length() == 0) {
            throw new IllegalArgumentException("deviceID cannot be null or empty.");
        }
        this.config = config;
        try {
            sslContext = this.config.getAuthenticationProvider().getSSLContext();
        }
        catch (IOException e) {
            throw new TransportException("Failed to get SSLContext", e);
        }
        if (this.config.getAuthenticationType() == ClientConfiguration.AuthType.SAS_TOKEN) {
            log.trace("MQTT connection will use sas token based auth");
            this.webSocketQueryString = NO_CLIENT_CERT_QUERY_STRING;
        } else if (this.config.getAuthenticationType() == ClientConfiguration.AuthType.X509_CERTIFICATE) {
            log.trace("MQTT connection will use X509 certificate based auth");
        }
        String userAgentString = this.config.getProductInfo().getUserAgentString();
        try {
            clientUserAgentIdentifier = "DeviceClientType=" + URLEncoder.encode(userAgentString, StandardCharsets.UTF_8.name()).replaceAll("\\+", "%20");
        }
        catch (UnsupportedEncodingException e) {
            throw new TransportException("Failed to URLEncode the user agent string", e);
        }
        String deviceId = this.config.getDeviceId();
        String moduleId = this.config.getModuleId();
        this.clientId = moduleId != null && !moduleId.isEmpty() ? deviceId + "/" + moduleId : deviceId;
        String modelId = this.config.getModelId();
        if (modelId == null || modelId.isEmpty()) {
            serviceParams = "2020-09-30";
        } else {
            try {
                serviceParams = "2020-09-30&model-id=" + URLEncoder.encode(modelId, StandardCharsets.UTF_8.name()).replaceAll("\\+", "%20");
            }
            catch (UnsupportedEncodingException e) {
                throw new TransportException("Failed to URLEncode the modelId string", e);
            }
        }
        String iotHubUserName = this.config.getIotHubHostname() + "/" + this.clientId + "/?api-version=" + serviceParams + "&" + clientUserAgentIdentifier;
        String host = this.config.getGatewayHostname();
        if (host == null || host.isEmpty()) {
            host = this.config.getIotHubHostname();
        }
        this.serverUri = this.config.isUsingWebsocket() ? (this.webSocketQueryString == null ? WS_SSL_PREFIX + host + WEBSOCKET_RAW_PATH : WS_SSL_PREFIX + host + WEBSOCKET_RAW_PATH + this.webSocketQueryString) : SSL_PREFIX + host + SSL_PORT_SUFFIX;
        MqttConnectOptions connectOptions = new MqttConnectOptions();
        connectOptions.setKeepAliveInterval(config.getKeepAliveInterval());
        connectOptions.setCleanSession(false);
        connectOptions.setMqttVersion(4);
        connectOptions.setUserName(iotHubUserName);
        connectOptions.setMaxInflight(65000);
        ProxySettings proxySettings = config.getProxySettings();
        if (proxySettings != null) {
            if (proxySettings.getProxy().type() == Proxy.Type.SOCKS) {
                try {
                    connectOptions.setSocketFactory((SocketFactory)new Socks5SocketFactory(proxySettings.getHostname(), proxySettings.getPort()));
                }
                catch (UnknownHostException e) {
                    throw new TransportException("Failed to build the Socks5SocketFactory", e);
                }
            } else {
                if (proxySettings.getProxy().type() != Proxy.Type.HTTP) throw new IllegalArgumentException("Proxy settings must be configured to use either SOCKS or HTTP");
                connectOptions.setSocketFactory((SocketFactory)new HttpProxySocketFactory(sslContext.getSocketFactory(), proxySettings));
            }
        } else {
            connectOptions.setSocketFactory((SocketFactory)sslContext.getSocketFactory());
        }
        ConcurrentHashMap<Integer, Message> unacknowledgedSentMessages = new ConcurrentHashMap<Integer, Message>();
        ConcurrentLinkedQueue<Pair<String, MqttMessage>> receivedMessages = new ConcurrentLinkedQueue<Pair<String, MqttMessage>>();
        this.deviceMessaging = new MqttMessaging(deviceId, this, moduleId, this.config.getGatewayHostname() != null && !this.config.getGatewayHostname().isEmpty(), connectOptions, unacknowledgedSentMessages, receivedMessages);
        this.directMethod = new MqttDirectMethod(deviceId, connectOptions, unacknowledgedSentMessages, receivedMessages);
        this.deviceTwin = new MqttTwin(deviceId, connectOptions, unacknowledgedSentMessages, receivedMessages);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void open() throws TransportException {
        Object object = this.mqttConnectionStateLock;
        synchronized (object) {
            this.connectionId = UUID.randomUUID().toString();
            this.deviceMessaging.setConnectionId(this.connectionId);
            this.deviceTwin.setConnectionId(this.connectionId);
            this.directMethod.setConnectionId(this.connectionId);
            if (this.state == IotHubConnectionStatus.CONNECTED) {
                return;
            }
            log.debug("Opening MQTT connection...");
            if (this.config.getSasTokenAuthentication() != null) {
                try {
                    log.trace("Setting password for MQTT connection since it is a SAS token authenticated connection");
                    this.deviceMessaging.updatePassword(this.config.getSasTokenAuthentication().getSasToken());
                }
                catch (IOException e) {
                    throw new TransportException("Failed to open the MQTT connection because a SAS token could not be retrieved", e);
                }
            }
            MqttAsyncClient mqttAsyncClient = this.buildMqttAsyncClient(this.serverUri, this.clientId);
            mqttAsyncClient.setCallback((MqttCallback)this.deviceMessaging);
            this.deviceMessaging.setMqttAsyncClient(mqttAsyncClient);
            this.deviceTwin.setMqttAsyncClient(mqttAsyncClient);
            this.directMethod.setMqttAsyncClient(mqttAsyncClient);
            this.deviceMessaging.start();
            this.state = IotHubConnectionStatus.CONNECTED;
            log.debug("MQTT connection opened successfully");
            this.listener.onConnectionEstablished(this.connectionId);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        Object object = this.mqttConnectionStateLock;
        synchronized (object) {
            if (this.state == IotHubConnectionStatus.DISCONNECTED) {
                return;
            }
            log.debug("Closing MQTT connection");
            this.directMethod.stop();
            this.deviceTwin.stop();
            this.deviceMessaging.stop();
            this.state = IotHubConnectionStatus.DISCONNECTED;
            log.debug("Successfully closed MQTT connection");
        }
    }

    @Override
    public void setListener(IotHubListener listener) throws IllegalArgumentException {
        if (listener == null) {
            throw new IllegalArgumentException("listener cannot be null");
        }
        this.listener = listener;
        this.deviceMessaging.setListener(listener);
    }

    @Override
    public IotHubStatusCode sendMessage(Message message) throws TransportException {
        if (message == null || message.getBytes() == null || message.getMessageType() != MessageType.DEVICE_TWIN && message.getMessageType() != MessageType.DEVICE_METHODS && message.getBytes().length == 0) {
            return IotHubStatusCode.BAD_FORMAT;
        }
        if (this.state == IotHubConnectionStatus.DISCONNECTED) {
            throw new IllegalStateException("Cannot send event using a closed MQTT connection");
        }
        IotHubStatusCode result = IotHubStatusCode.OK;
        if (message.getMessageType() == MessageType.DEVICE_METHODS) {
            this.directMethod.start();
            log.trace("Sending MQTT device method message ({})", (Object)message);
            this.directMethod.send((IotHubTransportMessage)message);
        } else if (message.getMessageType() == MessageType.DEVICE_TWIN) {
            this.deviceTwin.start();
            log.trace("Sending MQTT device twin message ({})", (Object)message);
            this.deviceTwin.send((IotHubTransportMessage)message);
        } else {
            log.trace("Sending MQTT device telemetry message ({})", (Object)message);
            this.deviceMessaging.send(message);
        }
        return result;
    }

    @Override
    public boolean sendMessageResult(IotHubTransportMessage message, IotHubMessageResult result) throws TransportException {
        if (message == null || result == null) {
            throw new TransportException(new IllegalArgumentException("message and result must be non-null"));
        }
        if (message.getQualityOfService() == 0) {
            return true;
        }
        log.trace("Checking if MQTT layer can acknowledge the received message ({})", (Object)message);
        if (!this.receivedMessagesToAcknowledge.containsKey(message)) {
            TransportException e = new TransportException(new IllegalArgumentException("Provided message cannot be acknowledged because it was already acknowledged or was never received from service"));
            log.error("Mqtt layer could not acknowledge received message because it has no mapping to an outstanding mqtt message id ({})", (Object)message, (Object)e);
            throw e;
        }
        int messageId = this.receivedMessagesToAcknowledge.get(message);
        log.trace("Sending MQTT ACK for a received message ({})", (Object)message);
        if (message.getMessageType() == MessageType.DEVICE_METHODS) {
            this.directMethod.start();
            this.directMethod.sendMessageAcknowledgement(messageId);
        } else if (message.getMessageType() == MessageType.DEVICE_TWIN) {
            this.deviceTwin.start();
            this.deviceTwin.sendMessageAcknowledgement(messageId);
        } else {
            this.deviceMessaging.sendMessageAcknowledgement(messageId);
        }
        log.trace("MQTT ACK was sent for a received message so it has been removed from the messages to acknowledge list ({})", (Object)message);
        this.receivedMessagesToAcknowledge.remove(message);
        return true;
    }

    @Override
    public String getConnectionId() {
        return this.connectionId;
    }

    @Override
    public void onMessageArrived(int messageId) {
        IotHubTransportMessage transportMessage = this.directMethod.receive();
        if (transportMessage != null) {
            log.trace("Received MQTT device method message ({})", (Object)transportMessage);
        } else {
            transportMessage = this.deviceTwin.receive();
            if (transportMessage != null) {
                log.trace("Received MQTT device twin message ({})", (Object)transportMessage);
            } else {
                transportMessage = this.deviceMessaging.receive();
                if (transportMessage != null) {
                    log.trace("Received MQTT device messaging message ({})", (Object)transportMessage);
                }
            }
        }
        if (transportMessage == null) {
            this.listener.onMessageReceived(null, new TransportException("Message sent from service could not be parsed"));
            log.warn("Received message that could not be parsed. That message has been ignored.");
        } else {
            if (transportMessage.getQualityOfService() == 0) {
                log.trace("MQTT received message with QoS 0 so it has not been added to the messages to acknowledge list ({})", (Object)transportMessage);
            } else {
                log.trace("MQTT received message so it has been added to the messages to acknowledge list ({})", (Object)transportMessage);
                this.receivedMessagesToAcknowledge.put(transportMessage, messageId);
            }
            switch (transportMessage.getMessageType()) {
                case DEVICE_TWIN: {
                    transportMessage.setMessageCallback(this.config.getDeviceTwinMessageCallback());
                    transportMessage.setMessageCallbackContext(this.config.getDeviceTwinMessageContext());
                    break;
                }
                case DEVICE_METHODS: {
                    transportMessage.setMessageCallback(this.config.getDirectMethodsMessageCallback());
                    transportMessage.setMessageCallbackContext(this.config.getDirectMethodsMessageContext());
                    break;
                }
                case DEVICE_TELEMETRY: {
                    transportMessage.setMessageCallback(this.config.getDeviceTelemetryMessageCallback(transportMessage.getInputName()));
                    transportMessage.setMessageCallbackContext(this.config.getDeviceTelemetryMessageContext(transportMessage.getInputName()));
                    break;
                }
            }
            this.listener.onMessageReceived(transportMessage, null);
        }
    }

    private MqttAsyncClient buildMqttAsyncClient(String serverUri, String clientId) throws TransportException {
        MqttAsyncClient mqttAsyncClient;
        try {
            mqttAsyncClient = new MqttAsyncClient(serverUri, clientId, (MqttClientPersistence)new MemoryPersistence());
        }
        catch (MqttException e) {
            throw PahoExceptionTranslator.convertToMqttException(e, "Failed to create mqtt client");
        }
        mqttAsyncClient.setManualAcks(true);
        return mqttAsyncClient;
    }
}

