/**
 *
 * Copyright (c) Microsoft and contributors.  All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// Warning: This code was generated by a tool.
//
// Changes to this file may cause incorrect behavior and will be lost if the
// code is regenerated.

package com.microsoft.azure.keyvault;

import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;

import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.StringEntity;

import com.microsoft.windowsazure.core.ServiceOperations;
import com.microsoft.windowsazure.core.pipeline.apache.CustomHttpDelete;
import com.microsoft.windowsazure.core.utils.CollectionStringBuilder;
import com.microsoft.windowsazure.core.utils.StreamUtils;
import com.microsoft.windowsazure.exception.ServiceException;
import com.microsoft.windowsazure.tracing.CloudTracing;

/**
 * Cryptographic and management operations for keys in a vault
 */
class KeyOperationsImpl implements ServiceOperations<KeyVaultInternalClientImpl>, KeyOperations {
    /**
     * Initializes a new instance of the KeyOperationsImpl class.
     *
     * @param client
     *            Reference to the service client.
     */
    KeyOperationsImpl(KeyVaultInternalClientImpl client) {
        this.client = client;
    }

    private KeyVaultInternalClientImpl client;

    /**
     * Gets a reference to the
     * microsoft.azure.keyvault.internal.KeyVaultInternalClientImpl.
     *
     * @return The Client value.
     */
    @Override
    public KeyVaultInternalClientImpl getClient() {
        return client;
    }

    /**
     * Requests that a backup of the specified key be downloaded to the client.
     *
     * @param keyIdentifier
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> backupAsync(final String keyIdentifier) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return backup(keyIdentifier);
            }
        });
    }

    /**
     * Requests that a backup of the specified key be downloaded to the client.
     *
     * @param keyIdentifier
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent backup(String keyIdentifier) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            CloudTracing.enter(invocationId, this, "backupAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/backup";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPost httpRequest = new HttpPost(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.create(httpRequest, null, httpResponse, httpResponse.getEntity(), "Json");
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Creates a new, named, key in the specified vault.
     *
     * @param vault
     *            Required.
     * @param keyName
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> createAsync(final String vault, final String keyName, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return create(vault, keyName, keyOpRequest);
            }
        });
    }

    /**
     * Creates a new, named, key in the specified vault.
     *
     * @param vault
     *            Required.
     * @param keyName
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent create(String vault, String keyName, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (vault == null) {
            throw new NullPointerException("vault");
        }
        if (keyName == null) {
            throw new NullPointerException("keyName");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("vault", vault);
            tracingParameters.put("keyName", keyName);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "createAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + vault;
        url = url + "/keys/";
        url = url + URLEncoder.encode(keyName, "UTF-8");
        url = url + "/create";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPost httpRequest = new HttpPost(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.create(httpRequest, requestContent, httpResponse, httpResponse.getEntity(), "Json");
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Decrypts a single block of data.
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> decryptDataAsync(final String keyIdentifier, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return decryptData(keyIdentifier, keyOpRequest);
            }
        });
    }

    /**
     * Decrypts a single block of data.
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent decryptData(String keyIdentifier, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "decryptDataAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/decrypt";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPost httpRequest = new HttpPost(url);

        // Set Headers
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.create(httpRequest, requestContent, httpResponse, httpResponse.getEntity(), "Json");
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Delete the specified key
     *
     * @param vault
     *            Required.
     * @param keyName
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> deleteKeyAsync(final String vault, final String keyName) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return deleteKey(vault, keyName);
            }
        });
    }

    /**
     * Delete the specified key
     *
     * @param vault
     *            Required.
     * @param keyName
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent deleteKey(String vault, String keyName) throws IOException, ServiceException {
        // Validate
        if (vault == null) {
            throw new NullPointerException("vault");
        }
        if (keyName == null) {
            throw new NullPointerException("keyName");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("vault", vault);
            tracingParameters.put("keyName", keyName);
            CloudTracing.enter(invocationId, this, "deleteKeyAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + vault;
        url = url + "/keys/";
        url = url + URLEncoder.encode(keyName, "UTF-8");
        url = url + "/delete";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        CustomHttpDelete httpRequest = new CustomHttpDelete(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.create(httpRequest, null, httpResponse, httpResponse.getEntity(), "Json");
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Encrypts a single block of data.
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> encryptDataAsync(final String keyIdentifier, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return encryptData(keyIdentifier, keyOpRequest);
            }
        });
    }

    /**
     * Encrypts a single block of data.
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent encryptData(String keyIdentifier, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "encryptDataAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/encrypt";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPost httpRequest = new HttpPost(url);

        // Set Headers
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.createFromXml(httpRequest, requestContent, httpResponse, httpResponse.getEntity());
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Retrieves the public portion of a key plus its attributes
     *
     * @param keyIdentifier
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> getAsync(final String keyIdentifier) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return get(keyIdentifier);
            }
        });
    }

    /**
     * Retrieves the public portion of a key plus its attributes
     *
     * @param keyIdentifier
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent get(String keyIdentifier) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            CloudTracing.enter(invocationId, this, "getAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpGet httpRequest = new HttpGet(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.create(httpRequest, null, httpResponse, httpResponse.getEntity(), "Json");
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Imports a key into the specified vault
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> importMethodAsync(final String keyIdentifier, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return importMethod(keyIdentifier, keyOpRequest);
            }
        });
    }

    /**
     * Imports a key into the specified vault
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent importMethod(String keyIdentifier, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "importMethodAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/import";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPut httpRequest = new HttpPut(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.createFromXml(httpRequest, requestContent, httpResponse, httpResponse.getEntity());
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * List the keys in the vault
     *
     * @param vault
     *            Required.
     * @param top
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> listAsync(final String vault, final Integer top) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return list(vault, top);
            }
        });
    }

    /**
     * List the keys in the vault
     *
     * @param vault
     *            Required.
     * @param top
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent list(String vault, Integer top) throws IOException, ServiceException {
        // Validate
        if (vault == null) {
            throw new NullPointerException("vault");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("vault", vault);
            tracingParameters.put("top", top);
            CloudTracing.enter(invocationId, this, "listAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + vault;
        url = url + "/keys/";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (top != null) {
            queryParameters.add("maxresults=" + URLEncoder.encode(Integer.toString(top), "UTF-8"));
        }
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpGet httpRequest = new HttpGet(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.create(httpRequest, null, httpResponse, httpResponse.getEntity(), "Json");
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * List the next page of keys in the vault
     *
     * @param nextLink
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> listNextAsync(final String nextLink) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return listNext(nextLink);
            }
        });
    }

    /**
     * List the next page of keys in the vault
     *
     * @param nextLink
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent listNext(String nextLink) throws IOException, ServiceException {
        // Validate
        if (nextLink == null) {
            throw new NullPointerException("nextLink");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("nextLink", nextLink);
            CloudTracing.enter(invocationId, this, "listNextAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + nextLink;
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpGet httpRequest = new HttpGet(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.create(httpRequest, null, httpResponse, httpResponse.getEntity(), "Json");
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * List the versions of a key
     *
     * @param vault
     *            Required.
     * @param keyName
     *            Required.
     * @param top
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> listVersionsAsync(final String vault, final String keyName, final Integer top) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return listVersions(vault, keyName, top);
            }
        });
    }

    /**
     * List the versions of a key
     *
     * @param vault
     *            Required.
     * @param keyName
     *            Required.
     * @param top
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent listVersions(String vault, String keyName, Integer top) throws IOException, ServiceException {
        // Validate
        if (vault == null) {
            throw new NullPointerException("vault");
        }
        if (keyName == null) {
            throw new NullPointerException("keyName");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("vault", vault);
            tracingParameters.put("keyName", keyName);
            tracingParameters.put("top", top);
            CloudTracing.enter(invocationId, this, "listVersionsAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + vault;
        url = url + "/keys/";
        url = url + URLEncoder.encode(keyName, "UTF-8");
        url = url + "/versions";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (top != null) {
            queryParameters.add("maxresults=" + URLEncoder.encode(Integer.toString(top), "UTF-8"));
        }
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpGet httpRequest = new HttpGet(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.create(httpRequest, null, httpResponse, httpResponse.getEntity(), "Json");
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * List the next page of versions of a key
     *
     * @param nextLink
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> listVersionsNextAsync(final String nextLink) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return listVersionsNext(nextLink);
            }
        });
    }

    /**
     * List the next page of versions of a key
     *
     * @param nextLink
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent listVersionsNext(String nextLink) throws IOException, ServiceException {
        // Validate
        if (nextLink == null) {
            throw new NullPointerException("nextLink");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("nextLink", nextLink);
            CloudTracing.enter(invocationId, this, "listVersionsNextAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + nextLink;
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpGet httpRequest = new HttpGet(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.create(httpRequest, null, httpResponse, httpResponse.getEntity(), "Json");
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Restores the backup key in to a vault
     *
     * @param vault
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> restoreAsync(final String vault, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return restore(vault, keyOpRequest);
            }
        });
    }

    /**
     * Restores the backup key in to a vault
     *
     * @param vault
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent restore(String vault, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (vault == null) {
            throw new NullPointerException("vault");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("vault", vault);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "restoreAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + vault;
        url = url + "/keys/restore";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPost httpRequest = new HttpPost(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.createFromXml(httpRequest, requestContent, httpResponse, httpResponse.getEntity());
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Creates a signature from a digest using the specified key in the vault
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> signAsync(final String keyIdentifier, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return sign(keyIdentifier, keyOpRequest);
            }
        });
    }

    /**
     * Creates a signature from a digest using the specified key in the vault
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent sign(String keyIdentifier, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "signAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/sign";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPost httpRequest = new HttpPost(url);

        // Set Headers
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.createFromXml(httpRequest, requestContent, httpResponse, httpResponse.getEntity());
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Unwraps a symmetric key using the specified key in the vault
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> unwrapKeyAsync(final String keyIdentifier, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return unwrapKey(keyIdentifier, keyOpRequest);
            }
        });
    }

    /**
     * Unwraps a symmetric key using the specified key in the vault
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent unwrapKey(String keyIdentifier, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "unwrapKeyAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/unwrapkey";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPost httpRequest = new HttpPost(url);

        // Set Headers
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.createFromXml(httpRequest, requestContent, httpResponse, httpResponse.getEntity());
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Updates the Key Attributes associated with the specified key
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> updateAsync(final String keyIdentifier, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return update(keyIdentifier, keyOpRequest);
            }
        });
    }

    /**
     * Updates the Key Attributes associated with the specified key
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent update(String keyIdentifier, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "updateAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/update";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPatch httpRequest = new HttpPatch(url);

        // Set Headers
        httpRequest.setHeader("Accept", "application/json");
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.createFromXml(httpRequest, requestContent, httpResponse, httpResponse.getEntity());
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Verifies a signature using the specified key
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> verifyAsync(final String keyIdentifier, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return verify(keyIdentifier, keyOpRequest);
            }
        });
    }

    /**
     * Verifies a signature using the specified key
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent verify(String keyIdentifier, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "verifyAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/verify";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPost httpRequest = new HttpPost(url);

        // Set Headers
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.createFromXml(httpRequest, requestContent, httpResponse, httpResponse.getEntity());
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }

    /**
     * Wraps a symmetric key using the specified key
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @return Represents the response to a key operation request.
     */
    @Override
    public Future<KeyOpResponseMessageWithRawJsonContent> wrapKeyAsync(final String keyIdentifier, final KeyOpRequestMessageWithRawJsonContent keyOpRequest) {
        return getClient().getExecutorService().submit(new Callable<KeyOpResponseMessageWithRawJsonContent>() {
            @Override
            public KeyOpResponseMessageWithRawJsonContent call() throws Exception {
                return wrapKey(keyIdentifier, keyOpRequest);
            }
        });
    }

    /**
     * Wraps a symmetric key using the specified key
     *
     * @param keyIdentifier
     *            Required.
     * @param keyOpRequest
     *            Required.
     * @throws IOException
     *             Signals that an I/O exception of some sort has occurred. This
     *             class is the general class of exceptions produced by failed
     *             or interrupted I/O operations.
     * @throws ServiceException
     *             Thrown if an unexpected response is found.
     * @return Represents the response to a key operation request.
     */
    @Override
    public KeyOpResponseMessageWithRawJsonContent wrapKey(String keyIdentifier, KeyOpRequestMessageWithRawJsonContent keyOpRequest) throws IOException, ServiceException {
        // Validate
        if (keyIdentifier == null) {
            throw new NullPointerException("keyIdentifier");
        }
        if (keyOpRequest == null) {
            throw new NullPointerException("keyOpRequest");
        }

        // Tracing
        boolean shouldTrace = CloudTracing.getIsEnabled();
        String invocationId = null;
        if (shouldTrace) {
            invocationId = Long.toString(CloudTracing.getNextInvocationId());
            HashMap<String, Object> tracingParameters = new HashMap<String, Object>();
            tracingParameters.put("keyIdentifier", keyIdentifier);
            tracingParameters.put("keyOpRequest", keyOpRequest);
            CloudTracing.enter(invocationId, this, "wrapKeyAsync", tracingParameters);
        }

        // Construct URL
        String url = "";
        url = url + keyIdentifier;
        url = url + "/wrapkey";
        ArrayList<String> queryParameters = new ArrayList<String>();
        queryParameters.add("api-version=" + getClient().getApiVersion());
        if (queryParameters.size() > 0) {
            url = url + "?" + CollectionStringBuilder.join(queryParameters, "&");
        }
        url = url.replace(" ", "%20");

        // Create HTTP transport objects
        HttpPost httpRequest = new HttpPost(url);

        // Set Headers
        httpRequest.setHeader("client-request-id", UUID.randomUUID().toString());
        httpRequest.setHeader("Content-Type", "application/json");

        // Serialize Request
        String requestContent = keyOpRequest.getRawJsonRequest();
        StringEntity entity = new StringEntity(requestContent);
        httpRequest.setEntity(entity);
        httpRequest.setHeader("Content-Type", "application/json");

        // Send Request
        HttpResponse httpResponse = null;
        try {
            if (shouldTrace) {
                CloudTracing.sendRequest(invocationId, httpRequest);
            }
            httpResponse = getClient().getHttpClient().execute(httpRequest);
            if (shouldTrace) {
                CloudTracing.receiveResponse(invocationId, httpResponse);
            }
            int statusCode = httpResponse.getStatusLine().getStatusCode();
            if (statusCode >= HttpStatus.SC_BAD_REQUEST) {
                ServiceException ex = ServiceException.createFromXml(httpRequest, requestContent, httpResponse, httpResponse.getEntity());
                if (shouldTrace) {
                    CloudTracing.error(invocationId, ex);
                }
                throw ex;
            }

            // Create Result
            KeyOpResponseMessageWithRawJsonContent result = null;
            // Deserialize Response
            InputStream responseContent = httpResponse.getEntity().getContent();
            result = new KeyOpResponseMessageWithRawJsonContent();
            result.setKeyOpResponse(StreamUtils.toString(responseContent));

            result.setStatusCode(statusCode);

            if (shouldTrace) {
                CloudTracing.exit(invocationId, result);
            }
            return result;
        } finally {
            if (httpResponse != null && httpResponse.getEntity() != null) {
                httpResponse.getEntity().getContent().close();
            }
        }
    }
}
