package com.microsoft.azure.documentdb;

import org.json.JSONObject;

import com.microsoft.azure.documentdb.internal.Constants;

/**
 * Represents an offer version 2 in the Azure Cosmos DB database service.
 */
public class OfferV2 extends Offer {

    /**
     * Initialize an new instance of the OfferV2 object.
     *
     * @param offerThroughput the throughput value for this offer.
     * @param offerEnableRUPerMinuteThroughput if RU Per Minute throughput is enabled
     */
    public OfferV2(int offerThroughput, boolean offerEnableRUPerMinuteThroughput) {
        this.setOfferVersion(Constants.Properties.OFFER_VERSION_V2);
        this.setOfferType("");
        JSONObject content = new JSONObject();
        content.put(Constants.Properties.OFFER_THROUGHPUT, offerThroughput);
        content.put(Constants.Properties.OFFER_IS_RU_PER_MINUTE_THROUGHPUT_ENABLED, offerEnableRUPerMinuteThroughput);
        this.setContent(content);
    }

    /**
     * Initialize an new instance of the OfferV2 object.
     *
     * @param offerThroughput the throughput value for this offer.
     */
    public OfferV2(int offerThroughput) {
        this(offerThroughput, false);
    }

    /**
     * Initialize an new instance of the OfferV2 object, copy the base
     * properties from another Offer object and set the throughput value.
     *
     * @param otherOffer the Offer object whose base properties are to be copied.
     */
    public OfferV2(Offer otherOffer) {
        super(otherOffer);

        this.setOfferVersion(Constants.Properties.OFFER_VERSION_V2);
        this.setOfferType("");

        JSONObject content = this.getContent();
        if (content == null) {
            content = new JSONObject();
            this.setContent(content);
        }
    }

    /**
     * Gets the offer throughput for this offer.
     *
     * @return the offer throughput.
     */
    public int getOfferThroughput() {
        return this.getContent().getInt(Constants.Properties.OFFER_THROUGHPUT);
    }

    /**
     * Sets the offer throughput for this offer.
     *
     * @param throughput the throughput of this offer.
     */
    public void setOfferThroughput(int throughput) {
        this.getContent().put(Constants.Properties.OFFER_THROUGHPUT, throughput);
    }

    /**
     * Sets Request Units(RU)/Minute throughput enabled/disabled for collection in the Azure Cosmos DB database service.
     * 
     * @param offerEnableRUPerMinuteThroughput used for enabling or disabling option.
     */
    public void setOfferEnableRUPerMinuteThroughput(boolean offerEnableRUPerMinuteThroughput) {
         this.getContent().put(Constants.Properties.OFFER_IS_RU_PER_MINUTE_THROUGHPUT_ENABLED, offerEnableRUPerMinuteThroughput);
    }

    /**
     * Gets Request Units(RU)/Minute throughput is enabled/disabled for collection in the Azure Cosmos DB database service.
     * @return true if RUPerMin is enabled; otherwise false.
     */
    public boolean getOfferEnableRUPerMinuteThroughput() {
         return this.getContent().optBoolean(Constants.Properties.OFFER_IS_RU_PER_MINUTE_THROUGHPUT_ENABLED, false);
    }
}
