package com.mercadopago.mptracker;

import android.content.Context;

import com.mercadopago.mptracker.listeners.TracksListener;
import com.mercadopago.mptracker.trackers.MPTrackerService;

import java.util.HashMap;
import java.util.Map;

import static android.text.TextUtils.isEmpty;


/**
 * Created by mromar on 3/31/16.
 */
public class MPTracker {

    private static MPTracker mMPTrackerInstance;

    private TracksListener mTracksListener;

    private String mFlavor;
    private String mPublicKey;
    private String mSdkVersion;
    private String mSiteId;
    private Context mContext;

    private static final String SDK_PLATFORM = "Android";
    private static final String SDK_TYPE = "native";

    private static final String NO_SCREEN = "NO_SCREEN";
    private static final String DEFAULT_SITE = "MLA";

    private Boolean trackerInitialized = false;

    protected MPTracker() {
    }

    synchronized public static MPTracker getInstance() {
        if (mMPTrackerInstance == null) {
            mMPTrackerInstance = new MPTracker();
        }
        return mMPTrackerInstance;
    }

    public void setTracksListener(TracksListener tracksListener) {
        this.mTracksListener = tracksListener;
    }

    private void trackScreenLaunchedListener(String screenName) {
        if (this.mTracksListener != null) {
            this.mTracksListener.onScreenLaunched(screenName);
        }
    }

    private void trackEventPerformedListener(Map<String, String> eventMap) {
        if (this.mTracksListener != null) {
            this.mTracksListener.onEventPerformed(eventMap);
        }
    }

    /**
     * @param screenName      The name of the screen
     * @param action          The action that generate an event. Cannot be {@code null}. if it's
     *                        null, the track will not be sended.
     * @param paymentId       The payment id of a payment method off. Cannot be {@code null}. if
     *                        it's null, the track will not be sended.
     * @param paymentMethodId The payment method id. Cannot be {@code null}. if it's null, the track
     *                        will not be sended.
     * @param status          The payment status. Cannot be {@code null}. if it's null, the track
     *                        will not be sended.
     * @param statusDetail    The payment status detail. Cannot be {@code null}. if it's null, the
     *                        track will not be sended.
     * @param typeId          The payment type id. It have to be a card type.
     * @param installments    The installments quantity that the payment be done.
     * @param issuerId        The bank that emit the card.
     */
    public void trackPayment(String screenName, String action, Long paymentId, String paymentMethodId, String status, String statusDetail, String typeId, Integer installments, Integer issuerId) {
        if (trackerInitialized) {

            if (!isCardPaymentType(typeId)) {
                MPTrackerService.getInstance().trackPaymentId(paymentId, mFlavor, SDK_PLATFORM, SDK_TYPE, mPublicKey, mSdkVersion, mSiteId, mContext);
            }

            if (!isEmpty(action)) {
                //GATracker.getInstance().trackEvent(getPath(screenName), action, paymentMethodId, status, statusDetail, typeId, installments, issuerId);
            }

            // NEW 1.2.0: TracksListener
            Map<String, String> eventMap = new HashMap<>();
            eventMap.put("screen_name", screenName);
            eventMap.put("payment_id", "" + paymentId);
            eventMap.put("status", status);
            eventMap.put("status_detail", statusDetail);
            eventMap.put("type_id", typeId);
            eventMap.put("payment_method", paymentMethodId);
            eventMap.put("installments", "" + installments);
            eventMap.put("issuer_id", "" + issuerId);
            trackEventPerformedListener(eventMap);
        }
    }

    /**
     * @param token The card token id of a payment. Cannot be {@code null}. if it's null, the track
     *              will not be sended.
     */
    public void trackToken(String token) {
        if (trackerInitialized && !isEmpty(token)) {
            MPTrackerService.getInstance().trackToken(token, mFlavor, SDK_PLATFORM, SDK_TYPE, mPublicKey, mSdkVersion, mSiteId, mContext);
        }
    }

    /**
     * @param screenName The screen where an Google Analytic event happen.
     * @param action     The action that generate an event. Cannot be {@code null}. if it's null,
     *                   the track will not be sended.
     * @param flavor     The flavor that the merchant has integrated. Cannot be {@code null}. if
     *                   it's null, the track will not be sended.
     * @param publicKey  The public key of the merchant. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param siteId     The country where the preference come. Cannot be {@code null}. if it's
     *                   null, the track will not be sended.
     * @param sdkVersion The Mercado Pago sdk version. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param context    Reference to Android Context. Cannot be {@code null}.
     */
    public void trackEvent(String screenName, String action, String flavor, String publicKey, String siteId, String sdkVersion, Context context) {
        initTracker(flavor, publicKey, siteId, sdkVersion, context);

        if (trackerInitialized) {
            if (!isEmpty(action)) {
                //GATracker.getInstance().trackEvent(getPath(screenName), action);
            }

            // NEW 1.2.0: TracksListener
            Map<String, String> eventMap = new HashMap<>();
            eventMap.put("screen_name", screenName);
            eventMap.put("action", action);
            trackEventPerformedListener(eventMap);

        }

    }

    /**
     * @param screenName The screen where an Google Analytic event happen.
     * @param action     The action that generate an event. Cannot be {@code null}. if it's null,
     *                   the track will not be sended.
     * @param flavor     The flavor that the merchant has integrated. Cannot be {@code null}. if
     *                   it's null, the track will not be sended.
     * @param publicKey  The public key of the merchant. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param sdkVersion The Mercado Pago sdk version. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param context    Reference to Android Context. Cannot be {@code null}.
     */
    public void trackEvent(String screenName, String action, String flavor, String publicKey, String sdkVersion, Context context) {
        initTracker(flavor, publicKey, getSiteId(), sdkVersion, context);

        if (trackerInitialized) {
            if (!isEmpty(action)) {
                //GATracker.getInstance().trackEvent(getPath(screenName), action);
            }

            // NEW 1.2.0: TracksListener
            Map<String, String> eventMap = new HashMap<>();
            eventMap.put("screen_name", screenName);
            eventMap.put("action", action);
            trackEventPerformedListener(eventMap);

        }

    }

    /**
     * @param screenName The screen where an Google Analytic event happen.
     * @param action     The action that generate an event. Cannot be {@code null}. if it's null,
     *                   the track will not be sended.
     * @param flavor     The flavor that the merchant has integrated. Cannot be {@code null}. if
     *                   it's null, the track will not be sended.
     * @param publicKey  The public key of the merchant. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param siteId     The country where the preference come. Cannot be {@code null}. if it's
     *                   null, the track will not be sended.
     * @param sdkVersion The Mercado Pago sdk version. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param context    Reference to Android Context. Cannot be {@code null}.
     */
    public void trackEvent(String screenName, String action, String result, String flavor, String publicKey, String siteId, String sdkVersion, Context context) {
        initTracker(flavor, publicKey, siteId, sdkVersion, context);

        if (trackerInitialized) {
            if (!isEmpty(action) && !isEmpty(result)) {
                //GATracker.getInstance().trackEvent(getPath(screenName), action, result);
            }

            // NEW 1.2.0: TracksListener
            Map<String, String> eventMap = new HashMap<>();
            eventMap.put("screen_name", screenName);
            eventMap.put("result", result);
            eventMap.put("action", action);
            trackEventPerformedListener(eventMap);

        }

    }

    /**
     * @param screenName The screen where an Google Analytic event happen.
     * @param action     The action that generate an event. Cannot be {@code null}. if it's null,
     *                   the track will not be sended.
     * @param result     The result indicate if an action is success or fauile. Cannot be {@code
     *                   null}. if it's null, the track will not be sended.
     * @param flavor     The flavor that the merchant has integrated. Cannot be {@code null}. if
     *                   it's null, the track will not be sended.
     */
    public void trackEvent(String screenName, String action, String result, String flavor) {
        initTracker(flavor, mPublicKey, mSiteId, mSdkVersion, mContext);

        if (trackerInitialized) {
            if (!isEmpty(action) && !isEmpty(result)) {
                //GATracker.getInstance().trackEvent(getPath(screenName), action, result);
            }

            // NEW 1.2.0: TracksListener
            Map<String, String> eventMap = new HashMap<>();
            eventMap.put("screen_name", screenName);
            eventMap.put("result", result);
            eventMap.put("action", action);
            trackEventPerformedListener(eventMap);

        }
    }

    /**
     * @param name       The screen where an Google Analytic event happen.
     * @param flavor     The flavor that the merchant has integrated. Cannot be {@code null}. if
     *                   it's null, the track will not be sended.
     * @param publicKey  The public key of the merchant. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param siteId     The country where the preference come. Cannot be {@code null}. if it's
     *                   null, the track will not be sended.
     * @param sdkVersion The Mercado Pago sdk version. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param context    Reference to Android Context. Cannot be {@code null}.
     */
    public void trackScreen(String name, String flavor, String publicKey, String siteId, String sdkVersion, Context context) {
        initTracker(flavor, publicKey, siteId, sdkVersion, context);

        if (trackerInitialized) {
            if (areScreenParametersValid(name, context)) {
                //GATracker.getInstance().trackScreen(getPath(name));
            }

            // NEW 1.2.0: TracksListener
            trackScreenLaunchedListener(name);
        }
    }

    /**
     * @param name       The screen where an Google Analytic event happen.
     * @param flavor     The flavor that the merchant has integrated. Cannot be {@code null}. if
     *                   it's null, the track will not be sended.
     * @param publicKey  The public key of the merchant. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param sdkVersion The Mercado Pago sdk version. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param context    Reference to Android Context. Cannot be {@code null}.
     */
    public void trackScreen(String name, String flavor, String publicKey, String sdkVersion, Context context) {
        initTracker(flavor, publicKey, getSiteId(), sdkVersion, context);

        if (trackerInitialized) {
            if (areScreenParametersValid(name, context)) {
                //GATracker.getInstance().trackScreen(getPath(name));
            }

            // NEW 1.2.0: TracksListener
            trackScreenLaunchedListener(name);
        }

    }

    /**
     * @param flavor     The flavor that the merchant has integrated. Cannot be {@code null}. if
     *                   it's null, the track will not be sended.
     * @param publicKey  The public key of the merchant. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param siteId     The country where the preference come. Cannot be {@code null}. if it's
     *                   null, the track will not be sended.
     * @param sdkVersion The Mercado Pago sdk version. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param context    Reference to Android Context. Cannot be {@code null}.
     */
    private void initTracker(String flavor, String publicKey, String siteId, String sdkVersion, Context context) {
        if (!isTrackerInitialized()) {
            if (areInitParametersValid(flavor, publicKey, siteId, sdkVersion, context)) {
                trackerInitialized = true;

                this.mFlavor = flavor;
                this.mPublicKey = publicKey;
                this.mSiteId = siteId;
                this.mSdkVersion = sdkVersion;
                this.mContext = context;

                //GATracker.getInstance().trackNewSession(flavor, publicKey, siteId, sdkVersion, SDK_TYPE, context);
            }
        }
    }

    /**
     * Validate all init parameters
     *
     * @param flavor     The flavor that the merchant has integrated. Cannot be {@code null}. if
     *                   it's null, the track will not be sended.
     * @param publicKey  The public key of the merchant. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param siteId     The country where the preference come. Cannot be {@code null}. if it's
     *                   null, the track will not be sended.
     * @param sdkVersion The Mercado Pago sdk version. Cannot be {@code null}. if it's null, the
     *                   track will not be sended.
     * @param context    Reference to Android Context. Cannot be {@code null}.
     * @return True if all parameters are valid. False if any parameter is invalid
     */
    private boolean areInitParametersValid(String flavor, String publicKey, String siteId, String sdkVersion, Context context) {
        return !isEmpty(flavor) && !isEmpty(publicKey) && !isEmpty(sdkVersion) && !isEmpty(siteId) && context != null;
    }

    /**
     * Validate all screen parameters
     *
     * @param name    The screen where an Google Analytic event happen.
     * @param context Reference to Android Context. Cannot be {@code null}.
     * @return True if all parameters are valid. False if any parameter is invalid
     */
    private boolean areScreenParametersValid(String name, Context context) {
        return !isEmpty(name) && context != null;
    }

    /**
     * Check if MPTracker is initialized
     *
     * @return True if is initialized. False if is not initialize.
     */
    private boolean isTrackerInitialized() {
        return this.mFlavor != null && this.mPublicKey != null && this.mSdkVersion != null && this.mSiteId != null && this.mContext != null;
    }

    /**
     * Generate a custom name for Google Analytics tracks
     *
     * @param name The name of a screen or an event
     * @return The name with the flavor concatenared
     */
    private String getPath(String name) {
        StringBuilder path = new StringBuilder();

        if (isEmpty(name)) {
            path.append("F" + mFlavor + "/" + NO_SCREEN);
            return path.toString();
        } else {
            path.append("F" + mFlavor + "/" + name);
            return path.toString();
        }
    }

    /**
     * Get the set site
     *
     * @return The site that set the first track. if it is null return a site by default
     */
    private String getSiteId() {
        return mSiteId == null ? DEFAULT_SITE : mSiteId;
    }

    /**
     * Indicate if a payment was done by card or not
     *
     * @param paymentTypeId The payment type id of the payment tracked
     * @return True if it is a card payment. False if not a card payment.
     */
    private Boolean isCardPaymentType(String paymentTypeId) {
        return paymentTypeId.equals("credit_card") || paymentTypeId.equals("debit_card") || paymentTypeId.equals("prepaid_card");
    }

    /**
     * Enable the test mode. If it is enable, this tracker will not track to Google Analytics and
     * will track to the beta version of MPService
     */
    public void enableTestMode() {
        MPTrackerService.getInstance().enableTestMode();
    }
}




