package com.melo.android.mvp.ui.fragment;

import android.app.Fragment;
import android.content.Context;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Toast;

import com.melo.android.mvp.R;
import com.melo.android.mvp.business.IPresenter;
import com.melo.android.mvp.event.IEvent;
import com.melo.android.mvp.ui.loading.Loading;

import org.greenrobot.eventbus.EventBus;

import java.util.HashMap;
import java.util.Map;

public abstract class MFragment<P extends IPresenter> extends Fragment implements IView {

    P presenter;
    private Map<Integer, Loading> mapLoadings;

    abstract int getLayoutId();
    abstract P createPresenter();
    abstract Integer[] getLoadingsLayoutsIds();

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);

        if (presenter != null) return;

        presenter = createPresenter();
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, Bundle savedInstanceState) {
        mapLoadings = new HashMap<>();

        if (isPresenterEvent() && !EventBus.getDefault().isRegistered(presenter))
            ((IEvent) presenter).register();

        presenter.setView(this);

        return inflater.inflate(getLayoutId(), container, false);
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);

        createLoadingOptions(view);
        registerLoadings();
    }

    @Override
    public void onStart() {
        super.onStart();
        presenter.onStart();
    }

    @Override
    public void onStop() {
        presenter.onStop();
        super.onStop();
    }

    @Override
    public void onDestroyView() {
        if (isPresenterEvent()) ((IEvent) presenter).unregister();

        unregisterLoadings();

        presenter.setView(null);

        super.onDestroyView();
    }

    private void createLoadingOptions(View view) {
        createLoadingIfExists(view, R.id.lyt_loading_main);

        Integer[] idsLoadings = getLoadingsLayoutsIds();

        for (Integer idLoading: idsLoadings) {
            createLoadingIfExists(view, idLoading);
        }
    }

    private void createLoadingIfExists(View view, Integer idLoading) {
        if (view.findViewById(idLoading) == null) return;
        if (mapLoadings.containsKey(idLoading)) return;

        mapLoadings.put(idLoading, new Loading(view, idLoading));
    }

    @Override
    public void showToast(final String message) {
        Toast.makeText(getActivity(), message, Toast.LENGTH_SHORT).show();
    }

    @Override
    public void showToast(final int message) {
        Toast.makeText(getActivity(), message, Toast.LENGTH_SHORT).show();
    }

    private boolean isPresenterEvent() {
        return (IEvent.class.isAssignableFrom(presenter.getClass()));
    }

    private void registerLoadings() {
        for (Loading loading: mapLoadings.values())
            loading.register();
    }

    private void unregisterLoadings() {
        for (Loading loading: mapLoadings.values())
            loading.unregister();
    }
}
