package com.meizu.cloud.pushsdk.util;

import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;

import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.PushManager;
import com.meizu.cloud.pushsdk.constants.PushConstants;
import com.meizu.cloud.pushsdk.handler.impl.model.PlatformMessage;
import com.meizu.cloud.pushsdk.notification.MPushMessage;
import com.meizu.cloud.pushsdk.pushtracer.QuickTracker;
import com.meizu.cloud.pushsdk.pushtracer.constant.Parameters;
import com.meizu.cloud.pushsdk.pushtracer.event.PushEvent;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

/**
 * Created by liaojinlong on 15-6-29.
 *
 * ======================================================
 * |       事件名称       |                 含义           |
 * ======================================================
 * |  receive_push_event |   收到消息,包括通知栏消息和透传消息|
 * --------------------------------------------------------
 * |  show_push_message  |   通知栏展示即现在的曝光          |
 * --------------------------------------------------------
 * |  delete_push_message  |   删除通知栏                  |
 * --------------------------------------------------------
 * |  click_push_message  |   点击通知栏消息                |
 * --------------------------------------------------------
 *
 * 错误信息埋点
 * ======================================================
 * |       事件名称       |                 含义           |
 * ======================================================
 * | notification_service_message  |   消息错误说明        |
 * --------------------------------------------------------
 *
 * 上报时增加PushSDK版本号信息
 */
public class UxIPUtils {
    private static final String TAG = "UxIPUtils";




    public static void init(Context context){
         //UsageStatsProxy.getInstance(context,true);
    }

    public static void notificationEvent(Context context,String pushInfo,int pushInfoType,String taskId,String imei){
        if(!TextUtils.isEmpty(taskId)){
            onRecordMessageFlow(context,context.getPackageName(),imei,
                    taskId,PushManager.TAG,pushInfo,pushInfoType);
        }
    }

    public static void notificationEvent(Context context,Intent intent,String pushInfo,int pushInfoType){
        notificationEvent(context,intent,PushManager.TAG,pushInfo,pushInfoType);
    }
    public static void notificationEvent(Context context,Intent intent,String pushsdkVersion,String pushInfo,int pushInfoType){
        if(!TextUtils.isEmpty(getTaskId(intent))){
            UxIPUtils.onRecordMessageFlow(context, context.getPackageName(), intent.getStringExtra(PushConstants.MZ_PUSH_MESSAGE_STATISTICS_IMEI_KEY),
                    getTaskId(intent), pushsdkVersion,pushInfo,pushInfoType);
        }
    }

    public static String getTaskId(Intent intent){
        String taskId = intent.getStringExtra(PushConstants.EXTRA_APP_PUSH_TASK_ID);
        if(TextUtils.isEmpty(taskId)){
            try {
                MPushMessage mPushMessage = (MPushMessage) intent.getSerializableExtra(PushConstants.MZ_PUSH_PRIVATE_MESSAGE);
                if(mPushMessage != null){
                    taskId = mPushMessage.getTaskId();
                }
            } catch (Exception e){
                taskId = "no push platform task";
                DebugLogger.e(TAG,"paese MessageV2 error "+e.getMessage());
            }

        }
        return taskId;
    }

    public static void onRecordMessageFlow(Context context, String packageName, String deviceId, String taskId, String pushsdkVersion,String pushInfo,int pushInfoType){
        Map<String,String> propertiesMap = new HashMap<>();
        propertiesMap.put("taskId",taskId);
        propertiesMap.put("deviceId", deviceId);
        propertiesMap.put("timestamp",String.valueOf(System.currentTimeMillis()/1000));
        propertiesMap.put("package_name",packageName);
        propertiesMap.put("pushsdk_version", pushsdkVersion);
        propertiesMap.put("push_info",pushInfo);
        propertiesMap.put("push_info_type",String.valueOf(pushInfoType));
        onLogEvent(context, false,"notification_service_message", propertiesMap);
    }

    /**
     * 供桌面调用使用,上报桌面通知的展示事件
     * @param context
     * @param platformExtra PushSDk通过透传过来的platformExtra字段
     * @param packageName  上报事件的包名
     * */
    public static void onShowPushMessageEvent(Context context,String packageName,String platformExtra){
        if(!TextUtils.isEmpty(platformExtra)){
            try {
                JSONObject platformExtraJsonObj = new JSONObject(platformExtra);
                String taskId = null;
                String seqId = null;
                String pushTimestamp = null;
                String deviceId = null;
                if(platformExtraJsonObj.has(PlatformMessage.PLATFORM_TASK_ID)){
                    taskId = platformExtraJsonObj.getString(PlatformMessage.PLATFORM_TASK_ID);
                }
                if(platformExtraJsonObj.has(PlatformMessage.PLATFORM_SEQ_ID)){
                    seqId = platformExtraJsonObj.getString(PlatformMessage.PLATFORM_SEQ_ID);
                }
                if(platformExtraJsonObj.has(PlatformMessage.PLATFORM_PUSH_TIMESTAMP)){
                    pushTimestamp = platformExtraJsonObj.getString(PlatformMessage.PLATFORM_PUSH_TIMESTAMP);
                }
                if(platformExtraJsonObj.has(PlatformMessage.PLATFORM_DEVICE_ID)){
                    deviceId = platformExtraJsonObj.getString(PlatformMessage.PLATFORM_DEVICE_ID);
                }
                onShowPushMessageEvent(context,packageName,deviceId,taskId,seqId,pushTimestamp);
            } catch (Exception e ){
                DebugLogger.e(TAG,"the platformExtra parse error");
            }
        } else {
            DebugLogger.e(TAG,"the platformExtra is empty");
        }
    }

    /**
     * 弹通知栏event:show_push_message
     * */
    public static void onShowPushMessageEvent(Context context,String packageName,String deviceId,String taskId,String seqId,String pushTimestamp){
        onLogEvent(context,true,packageName,deviceId,taskId,seqId,Parameters.SHOW_PUSH_MESSAGE,pushTimestamp);
    }


    /**
     * 删除通知栏event:delete_push_message
     * **/
    public static void onDeletePushMessageEvent(Context context,String packageName,String deviceId,String taskId,String seqId,String pushTimestamp){
        onLogEvent(context,true,packageName,deviceId,taskId,seqId,Parameters.DELETE_PUSH_MESSAGE,pushTimestamp);
    }


    /**
     * 这个方法同下面的onReceiveThroughMessage 表示同一个意思，都是消息达到Receiver后埋点
     * */
    public static void onReceivePushMessageEvent(Context context,String packageName,String deviceId,String taskId,String seqId,String pushTimestamp){
        onLogEvent(context,false,packageName,deviceId,taskId,seqId,Parameters.RECEIVE_PUSH_EVNET,pushTimestamp);
    }

    /**
     * 第三方应用收到透传消息
     * 第三方接入方在魅族手机由于使用的pushService是cloud的,因此获取imei必须采用通用的方式
     * */
    public static void onReceiveThroughMessage(Context context,String packageName,String deviceId,String taskId,String seqId,String pushTimestamp){
        onLogEvent(context,true,packageName,deviceId,taskId,seqId,Parameters.RECEIVE_PUSH_EVNET,pushTimestamp);
    }



    public static void onClickPushMessageEvent(Context context,String packageName,String deviceId,String taskId,String seqId,String pushTimestamp){
        onLogEvent(context,true,packageName,deviceId,taskId,seqId,Parameters.CLICK_PUSH_MESSAGE,pushTimestamp);
    }

    public static void onInvalidPushMessage(Context context,String packageName,String deviceId,String taskId,String seqId,String pushTimestamp){
        onLogEvent(context,true,packageName,deviceId,taskId,seqId,Parameters.INVALID_PUSH_MESSAGE,pushTimestamp);
    }

    public static void onShowInBoxPushMessage(Context context,String packageName,String deviceId,String taskId,String seqId,String pushTimestamp){
        onLogEvent(context,false,packageName,deviceId,taskId,seqId,Parameters.SHOWINBOX_PUSH_MESSAGE,pushTimestamp);
    }

    public static void onNoShowPushMessage(Context context,String packageName,String deviceId,String taskId,String seqId,String pushTimestamp){
        onLogEvent(context,false,packageName,deviceId,taskId,seqId,Parameters.NOSHOW_PUSH_MESSAGE,pushTimestamp);
    }

    public static void onReceiveServerMessage(Context context,String packageName,String deviceId,String taskId,String seqId,String pushTimestamp){
        onLogEvent(context,false,packageName,deviceId,taskId,seqId,Parameters.RECEIVER_SERVER_MESSAGE,pushTimestamp);
    }


    public static void onLogEvent(Context context,boolean emitNow,String packageName,String deviceId,String taskId,String seqId,String eventName,String pushTimestamp){
        Map<String,String> propertiesMap = new HashMap<>();
        propertiesMap.put("taskId",taskId);
        propertiesMap.put("deviceId", deviceId);
        propertiesMap.put("timestamp",TextUtils.isEmpty(pushTimestamp) ? String.valueOf(System.currentTimeMillis()/1000) : pushTimestamp);
        propertiesMap.put("package_name",packageName);
        propertiesMap.put("pushsdk_version", PushManager.TAG);
        if(!TextUtils.isEmpty(seqId)){
            propertiesMap.put("seq_id",seqId);
        }
        onLogEvent(context,emitNow, eventName, propertiesMap);
    }

    /**
     * @param eventName
     *               event name for usage
     *        Uxip realtimeLog for notification_service_message
     * */
    public static void onLogEvent(Context context,boolean emitNow,String eventName,Map<String,String> propertiesMap){
        DebugLogger.e(TAG, "onLogEvent eventName [" + eventName + "] properties = " + propertiesMap);
        if("notification_service_message".equals(eventName)){
            //UsageStatsProxy.getInstance(context,true).onLogRealtime(eventName,propertiesMap);
        } else {
            QuickTracker.getAndroidTrackerClassic(context, null).track(PushEvent.builder()
                    .eventName(eventName)
                    .timestamp(Long.valueOf(propertiesMap.get("timestamp")))
                    .eventCreateTime(String.valueOf(System.currentTimeMillis() / 1000))
                    .deviceId(propertiesMap.get("deviceId"))
                    .packageName(propertiesMap.get("package_name"))
                    .pushsdkVersion(propertiesMap.get("pushsdk_version"))
                    .taskId(propertiesMap.get("taskId"))
                    .seqId(TextUtils.isEmpty(propertiesMap.get("seq_id")) ? "" : propertiesMap.get("seq_id"))
                    .messageSeq(String.valueOf(PushPreferencesUtils.getMessageSeqInCrease(context, propertiesMap.get("package_name"))))
                    .build(),emitNow);
        }
    }

}
