package com.meizu.cloud.pushsdk.pushtracer.storage;

import com.meizu.cloud.pushsdk.pushtracer.dataload.DataLoad;
import com.meizu.cloud.pushsdk.pushtracer.dataload.TrackerDataload;
import com.meizu.cloud.pushsdk.pushtracer.emitter.EmittableEvents;
import com.meizu.cloud.pushsdk.pushtracer.utils.Logger;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicLong;

/**
 *
 */
public class MemoryStore implements Store {
    private static final String TAG = "MemoryStore";
    private int sendLimit;
    private AtomicLong atomicLong = new AtomicLong(0);
    private Map<Long, byte[]> storeMap = new ConcurrentHashMap<>();
    private List<Long> dataKeyList = new CopyOnWriteArrayList<>();

    public MemoryStore(int sendLimit) {
        this.sendLimit = sendLimit;
    }

    @Override
    public void add(DataLoad dataLoad) {
        insertEvent(dataLoad);
    }

    @Override
    public boolean isOpen() {
        return true;
    }

    @Override
    public void close() {
        storeMap.clear();
        atomicLong.set(0);
        dataKeyList.clear();
    }


    @Override
    public boolean removeEvent(long id) {
        return dataKeyList.remove(id) && storeMap.remove(id) != null;
    }

    @Override
    public boolean removeAllEvents() {
        storeMap.clear();
        dataKeyList.clear();
        return true;
    }

    @Override
    public long getSize() {
        return dataKeyList.size();
    }

    @Override
    public EmittableEvents getEmittableEvents() {
        LinkedList<Long> eventIds = new LinkedList<>();
        ArrayList<DataLoad> events = new ArrayList<>();

        // FIFO Pattern for sending events
        int keySize = (int) getSize();
        int largeEmiteSize = (keySize > sendLimit ? sendLimit : keySize);
        for(int i=0; i< largeEmiteSize; i++){
            Object key = dataKeyList.get(i);
            if(key != null){
                TrackerDataload payload = new TrackerDataload();
                Object mapObj = EventStore.deserializer(storeMap.get(key));
                Map<String, Object> payloadMap = (Map<String, Object>)mapObj;
                payload.addMap(payloadMap);
                Logger.i(TAG," current key "+key+" payload "+payload);
                // Store the eventId
                eventIds.add((Long) key);
                // Add the payload to the list
                events.add(payload);
            }
        }
        return new EmittableEvents(events, eventIds);
    }



    /**
     * Inserts a payload into the database
     *
     * @param dataLoad The event dataLoad to
     *                be stored
     * @return a boolean stating if the insert
     * was a success or not
     */
    @SuppressWarnings("unchecked")
    public long insertEvent(DataLoad dataLoad){
        byte[] bytes = EventStore.serialize(dataLoad.getMap());
        long key = atomicLong.getAndIncrement();
        dataKeyList.add(key);
        storeMap.put(key,bytes);
        return key;
    }


    /**
     * Returns a Map containing the event
     * payload values, the table row ID and
     * the date it was created.
     *
     * @param id the row id of the event to get
     * @return event metadata
     */
    public Map<String, Object> getEvent(long id){
        byte[] eventByte = storeMap.get(id);
        if(eventByte != null){
            Object event = EventStore.deserializer(eventByte);
            return (Map<String, Object>) event;
        } else {
            return null;
        }

    }


}
