package com.meizu.cloud.pushsdk.platform.pushstrategy;

import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;

import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.networking.common.ANResponse;
import com.meizu.cloud.pushsdk.networking.error.ANError;
import com.meizu.cloud.pushsdk.platform.PlatformMessageSender;
import com.meizu.cloud.pushsdk.platform.api.PushAPI;
import com.meizu.cloud.pushsdk.platform.message.PushSwitchStatus;
import com.meizu.cloud.pushsdk.util.PushPreferencesUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;

/**
 * Created by liaojinlong on 17-4-17.
 * 开关状态存储方式
 */
public class SwitchStatusStrategy extends Strategy<PushSwitchStatus>{
    private String pushId;
    private int switchType = 0;
    boolean switcher;
    public static final int CHECK_PUSH = 2;
    public static final int SWITCH_ALL = 3;
    public static final int SWITCH_NOTIFICATION = 0;
    public static final int SWITCH_THROUGH_MESSAGE = 1;
    private Map<String,Boolean> pushStatusMap = new HashMap<>();

    public SwitchStatusStrategy(Context context, String appId, String appKey, PushAPI pushAPI, ScheduledExecutorService executorService) {
        super(context, appId, appKey, pushAPI, executorService);
    }

    public SwitchStatusStrategy(Context context, String appId, String appKey, String pushId,PushAPI pushAPI, ScheduledExecutorService executorService) {
        this(context, appId, appKey, pushAPI, executorService);
        this.pushId = pushId;
    }

    public SwitchStatusStrategy(Context context,PushAPI pushAPI, ScheduledExecutorService executorService) {
        this(context, null, null,null, pushAPI, executorService);
    }

    public SwitchStatusStrategy(Context context,PushAPI pushAPI, ScheduledExecutorService executorService,boolean enableRPC){
        this(context,pushAPI,executorService);
        this.enableRPC = enableRPC;
    }

    public void setSwitcher(boolean switcher){
        this.switcher = switcher;
    }

    public void setSwitchType(int switchType){
        this.switchType = switchType;
    }

    public void setPushId(String pushId){
        this.pushId = pushId;
    }

    @Override
    protected boolean matchCondition() {
        return !TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey) && !TextUtils.isEmpty(pushId);
    }

    @Override
    protected PushSwitchStatus feedBackErrorResponse() {
        PushSwitchStatus pushSwitchStatus = new PushSwitchStatus();
        pushSwitchStatus.setCode(FEEDBACK_PARAMETER_ERROR_CODE);
        if(TextUtils.isEmpty(appId)){
            pushSwitchStatus.setMessage("appId not empty");
        } else if(TextUtils.isEmpty(appKey)){
            pushSwitchStatus.setMessage("appKey not empty");
        } else if(TextUtils.isEmpty(pushId)){
            pushSwitchStatus.setMessage("pushId not empty");
        }
        return pushSwitchStatus;
    }

    @Override
    protected Intent sendRpcRequest() {
        Intent intent = null;
        if(switchType != CHECK_PUSH){
            intent = new Intent();
            intent.putExtra(APP_ID,appId);
            intent.putExtra(APP_KEY,appKey);
            intent.putExtra(STRATEGY_PACKAGE_NANME,context.getPackageName());
            intent.putExtra(PUSH_ID,pushId);
            intent.putExtra(STRATEGY_TYPE,strategyType());
            intent.putExtra(STRATEGY_CHILD_TYPE,switchType);
            intent.putExtra(STRATEGY_PARAMS, switcher ? "1" : "0");
        }
        return intent;
    }

    /**
     * 首次订阅成功,通知栏和透传消息开关全部打开
     * */
    @Override
    protected PushSwitchStatus netWorkRequest() {
        PushSwitchStatus pushSwitchStatus = new PushSwitchStatus();
        pushSwitchStatus.setPushId(pushId);
        pushSwitchStatus.setCode(SUCCESS_CODE);
        pushSwitchStatus.setMessage("");
        ANResponse<String> anResponse = null;
        switch (switchType){
            case CHECK_PUSH:
                 pushSwitchStatus.setSwitchNotificationMessage(notificationSwitch());
                 pushSwitchStatus.setSwitchThroughMessage(throughMessageSwitch());
                 //anResponse = pushAPI.checkPush(appId, appKey, pushId);
                break;
            case SWITCH_ALL:
                if(notificationSwitch() == switcher && throughMessageSwitch() == switcher && !isSyncPushStatus()){
                    pushSwitchStatus.setSwitchNotificationMessage(switcher);
                    pushSwitchStatus.setSwitchThroughMessage(switcher);
                } else {
                    changeSyncPushStatus(true);
                    switchAll(switcher);
                    anResponse = pushAPI.switchPush(appId,appKey,pushId,switcher);
                }
                break;
            case SWITCH_NOTIFICATION:
                if(notificationSwitch() == switcher && !isSyncPushStatus()){
                    pushSwitchStatus.setSwitchNotificationMessage(switcher);
                    pushSwitchStatus.setSwitchThroughMessage(throughMessageSwitch());
                } else {
                    changeSyncPushStatus(true);
                    switchNotification(switcher);
                    anResponse = pushAPI.switchPush(appId,appKey,pushId,switchType,switcher);
                }
                break;
            case SWITCH_THROUGH_MESSAGE:
                if(throughMessageSwitch() == switcher && !isSyncPushStatus()){
                    pushSwitchStatus.setSwitchNotificationMessage(notificationSwitch());
                    pushSwitchStatus.setSwitchThroughMessage(switcher);
                } else {
                    changeSyncPushStatus(true);
                    switchThroughMessage(switcher);
                    anResponse = pushAPI.switchPush(appId,appKey,pushId,switchType,switcher);
                }
                break;
        }

        if(anResponse != null){
            if(anResponse.isSuccess()){
                pushSwitchStatus = new PushSwitchStatus(anResponse.getResult());
                DebugLogger.e(TAG, "network pushSwitchStatus " + pushSwitchStatus);
                if(SUCCESS_CODE.equals(pushSwitchStatus.getCode())){
                     changeSyncPushStatus(false);
                }
            } else {
                ANError error = anResponse.getError();
                if (error.getResponse() != null) {
                    DebugLogger.e(TAG, "status code=" + error.getErrorCode() + " data=" + error.getResponse());
                }
                pushSwitchStatus.setCode(String.valueOf(error.getErrorCode()));
                pushSwitchStatus.setMessage(error.getErrorBody());
                DebugLogger.e(TAG, "pushSwitchStatus " + pushSwitchStatus);
            }
        }

        return pushSwitchStatus;
    }

    @Override
    protected PushSwitchStatus localResponse() {
        PushSwitchStatus pushSwitchStatus = null;
        switch (switchType){
            case CHECK_PUSH:
                pushSwitchStatus = new PushSwitchStatus();
                pushSwitchStatus.setPushId(pushId);
                pushSwitchStatus.setCode(SUCCESS_CODE);
                pushSwitchStatus.setMessage("");
                pushSwitchStatus.setSwitchNotificationMessage(notificationSwitch());
                pushSwitchStatus.setSwitchThroughMessage(throughMessageSwitch());
                break;
            case SWITCH_ALL:
                switchAll(switcher);
                break;
            case SWITCH_NOTIFICATION:
                switchNotification(switcher);
                break;
            case SWITCH_THROUGH_MESSAGE:
                switchThroughMessage(switcher);
                break;
        }
        return pushSwitchStatus;
    }

    @Override
    protected void sendReceiverMessage(PushSwitchStatus message) {
        PlatformMessageSender.sendPushStatus(context, !TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName(), message);
    }

    @Override
    protected int strategyType() {
        return STRATEGY_SWITCH;
    }

    private void switchNotification(boolean switcher){
        PushPreferencesUtils.setNotificationMessageSwitchStatus(context, !TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName(), switcher);
    }

    private void switchThroughMessage(boolean switcher){
        PushPreferencesUtils.setThroughMessageSwitchStatus(context, !TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName(), switcher);
    }

    private void switchAll(boolean switcher){
        PushPreferencesUtils.setNotificationMessageSwitchStatus(context, !TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName(), switcher);
        PushPreferencesUtils.setThroughMessageSwitchStatus(context, !TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName(), switcher);
    }

    private boolean notificationSwitch(){
        return PushPreferencesUtils.getNotificationMessageSwitchStatus(context,!TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName());
    }

    private boolean throughMessageSwitch(){
        return PushPreferencesUtils.getThroughMessageSwitchStatus(context, !TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName());
    }

    private void changeSyncPushStatus(boolean flag){
        pushStatusMap.put(strategyPackageNanme+"_" +switchType,flag);
    }

    private boolean isSyncPushStatus(){
        Boolean pushStatus = pushStatusMap.get(strategyPackageNanme + "_" +switchType);
        return pushStatus != null ? pushStatus.booleanValue() : true;
    }
}
