package com.meizu.cloud.pushsdk.platform.pushstrategy;

import android.content.Context;
import android.content.Intent;
import android.text.TextUtils;
import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.networking.common.ANResponse;
import com.meizu.cloud.pushsdk.networking.error.ANError;
import com.meizu.cloud.pushsdk.platform.PlatformMessageSender;
import com.meizu.cloud.pushsdk.platform.api.PushAPI;
import com.meizu.cloud.pushsdk.platform.message.SubAliasStatus;
import com.meizu.cloud.pushsdk.util.PushPreferencesUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;

/**
 * Created by liaojinlong on 17-4-17.
 */
public class SubScribeAliasStrategy extends Strategy<SubAliasStatus>{
    private String pushId;
    private int subAliasType;
    private String alias;
    public static final int SUB_ALIAS = 0;
    public static final int UNSUB_ALIAS = 1;
    public static final int CHECK_ALIAS = 2;
    private Map<String,Boolean> aliasStatusMap = new HashMap<>();

    public SubScribeAliasStrategy(Context context, String appId, String appKey, PushAPI pushAPI, ScheduledExecutorService executorService) {
        super(context, appId, appKey, pushAPI, executorService);
    }

    public SubScribeAliasStrategy(Context context, String appId, String appKey, String pushId,PushAPI pushAPI, ScheduledExecutorService executorService) {
        this(context, appId, appKey, pushAPI, executorService);
        this.pushId = pushId;
    }

    public SubScribeAliasStrategy(Context context,PushAPI pushAPI, ScheduledExecutorService executorService) {
        this(context, null, null, null, pushAPI, executorService);
    }

    public SubScribeAliasStrategy(Context context,PushAPI pushAPI, ScheduledExecutorService executorService, boolean enableRPC){
        this(context,pushAPI,executorService);
        this.enableRPC = enableRPC;
    }

    public void setSubAliasType(int subAliasType){
        this.subAliasType = subAliasType;
    }

    public void setAlias(String alias){
        this.alias = alias;
    }

    public void setPushId(String pushId){
        this.pushId = pushId;
    }

    @Override
    protected boolean matchCondition() {
        return !TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey) && !TextUtils.isEmpty(pushId);
    }

    @Override
    protected SubAliasStatus feedBackErrorResponse() {
        SubAliasStatus subAliasStatus = new SubAliasStatus();
        subAliasStatus.setCode(FEEDBACK_PARAMETER_ERROR_CODE);
        if(TextUtils.isEmpty(appId)){
            subAliasStatus.setMessage("appId not empty");
        } else if(TextUtils.isEmpty(appKey)){
            subAliasStatus.setMessage("appKey not empty");
        } else if(TextUtils.isEmpty(pushId)){
            subAliasStatus.setMessage("pushId not empty");
        }
        return subAliasStatus;
    }

    @Override
    protected Intent sendRpcRequest() {
        Intent intent = null;
        if(subAliasType != CHECK_ALIAS){
            intent = new Intent();
            intent.putExtra(APP_ID,appId);
            intent.putExtra(APP_KEY,appKey);
            intent.putExtra(STRATEGY_PACKAGE_NANME,context.getPackageName());
            intent.putExtra(PUSH_ID,pushId);
            intent.putExtra(STRATEGY_TYPE,strategyType());
            intent.putExtra(STRATEGY_CHILD_TYPE,subAliasType);
            intent.putExtra(STRATEGY_PARAMS, alias);
        }

        return intent;
    }

    @Override
    protected SubAliasStatus netWorkRequest() {
        SubAliasStatus subAliasStatus = new SubAliasStatus();
        subAliasStatus.setPushId(pushId);
        subAliasStatus.setMessage("");
        ANResponse<String> anResponse = null;
        switch (subAliasType){
            case SUB_ALIAS:
                if(alias.equals(localAlias()) && !isSyncAliasStatus()){
                    subAliasStatus.setCode(SUCCESS_CODE);
                    subAliasStatus.setAlias(alias);
                } else {
                    changeSyncAliasStatus(true);
                    if(isCacheAlias()){
                        storeAlias(alias);
                    }
                    anResponse = pushAPI.subScribeAlias(appId,appKey,pushId,alias);
                }
                break;
            case UNSUB_ALIAS:
                if(TextUtils.isEmpty(localAlias()) && !isSyncAliasStatus()){
                    subAliasStatus.setCode(SUCCESS_CODE);
                    subAliasStatus.setAlias("");
                } else {
                    changeSyncAliasStatus(true);
                    if(isCacheAlias()){
                        storeAlias("");
                    }
                    anResponse = pushAPI.unSubScribeAlias(appId,appKey,pushId,alias);
                }
                break;
            case CHECK_ALIAS:
                subAliasStatus.setAlias(localAlias());
                subAliasStatus.setCode(SUCCESS_CODE);
                break;
        }
        if(anResponse != null){
            if(anResponse.isSuccess()){
                subAliasStatus = new SubAliasStatus(anResponse.getResult());
                DebugLogger.e(TAG, "network subAliasStatus " + subAliasStatus);
                if(SUCCESS_CODE.equals(subAliasStatus.getCode())){
                    changeSyncAliasStatus(false);
                }
            } else {
                ANError error = anResponse.getError();
                if (error.getResponse() != null) {
                    DebugLogger.e(TAG, "status code=" + error.getErrorCode() + " data=" + error.getResponse());
                }
                subAliasStatus.setCode(String.valueOf(error.getErrorCode()));
                subAliasStatus.setMessage(error.getErrorBody());
                DebugLogger.e(TAG, "subAliasStatus " + subAliasStatus);
            }
        }

        return subAliasStatus;
    }

    @Override
    protected SubAliasStatus localResponse() {
        SubAliasStatus subAliasStatus = null;
        switch (subAliasType){
            case CHECK_ALIAS:
                subAliasStatus = new SubAliasStatus();
                subAliasStatus.setCode(SUCCESS_CODE);
                subAliasStatus.setPushId(pushId);
                subAliasStatus.setAlias(localAlias());
                subAliasStatus.setMessage("check alias success");
                break;
        }
        return subAliasStatus;
    }

    @Override
    protected void sendReceiverMessage(SubAliasStatus message) {
        PlatformMessageSender.sendSubAlias(context, !TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName(), message);
    }

    @Override
    protected int strategyType() {
        return STRATEGY_SUBALIAS;
    }

    private void storeAlias(String alias){
        PushPreferencesUtils.setAlias(context, !TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName(), alias);
    }

    private String localAlias(){
        return PushPreferencesUtils.getAlias(context,!TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName());
    }

    private void changeSyncAliasStatus(boolean flag){
        aliasStatusMap.put(strategyPackageNanme + "_" + subAliasType, flag);
    }

    private boolean isSyncAliasStatus(){
        Boolean aliasStatus = aliasStatusMap.get(strategyPackageNanme + "_" + subAliasType);
        return aliasStatus != null ? aliasStatus.booleanValue() : true;
    }

    private boolean isCacheAlias(){
        return !isSupportRemoteInvoke && "com.meizu.cloud".equals(strategyPackageNanme);
    }
}
