package com.meizu.cloud.pushsdk.platform.pushstrategy;

import android.content.Context;
import android.content.Intent;
import android.content.pm.ResolveInfo;
import android.text.TextUtils;
import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.platform.api.PushAPI;
import com.meizu.cloud.pushsdk.platform.message.BasicPushStatus;
import com.meizu.cloud.pushsdk.util.MzSystemUtils;
import com.meizu.cloud.pushsdk.util.PushPreferencesUtils;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;

/**
 * Created by liaojinlong on 17-4-17.
 * 参数校验 ==>> 状态回复
 *   ||
 *   \/
 * 是否支持远程调用=/flase\=>>本地调用==>>状态回复
 *    ||
 *    \/
 * 错误状态回复
 */
public abstract class Strategy<T extends BasicPushStatus> {
    public static final String TAG ="Strategy";

    protected ScheduledExecutorService executorService;
    protected Context context;
    protected String appId;
    protected String appKey;
    protected String strategyPackageNanme;
    protected volatile String deviceId;
    protected PushAPI pushAPI;
    /**是否支持远程调用**/
    protected boolean isSupportRemoteInvoke = false;

    /**是否启用RPC调用*/
    protected boolean enableRPC = true;

    public static final int STRATEGY_REGISTER = 1 << 1;
    public static final int STRATEGY_SUBTAGS = 1 << 2;
    public static final int STRATEGY_SUBALIAS = 1 << 3;
    public static final int STRATEGY_SWITCH = 1 << 4;
    public static final int STRATEGY_UNREGISTER = 1 << 5;

    public static final String APP_ID = "app_id";
    public static final String APP_KEY = "app_key";
    public static final String PUSH_ID = "push_id";
    public static final String STRATEGY_PACKAGE_NANME = "strategy_package_name";
    public static final String STRATEGY_TYPE = "strategy_type";
    public static final String STRATEGY_CHILD_TYPE = "strategy_child_type";
    public static final String STRATEGY_PARAMS = "strategy_params";

    private String managerServicePackageName = null;

    public static final String DEVICE_ERROR_CODE = "20000";
    public static final String SUCCESS_CODE = "200";
    public static final String FEEDBACK_PARAMETER_ERROR_CODE = "20001";

    public Strategy(Context context,String appId, String appKey,PushAPI pushAPI,ScheduledExecutorService executorService){
        this.executorService = executorService;
        this.context = context;
        this.appId = appId;
        this.appKey = appKey;
        this.pushAPI = pushAPI;

    }

    public void setExecutorService(ScheduledExecutorService executorService){
        this.executorService = executorService;
    }

    public void setAppId(String appId){
        this.appId = appId;
    }

    public void setAppKey(String appKey){
        this.appKey = appKey;
    }

    public void setStrategyPackageNanme(String strategyPackageNanme){
        this.strategyPackageNanme = strategyPackageNanme;
    }

    /**
     * 设置是否支持远程调用
     * */
    public void setSupportRemoteInvoke(boolean isSupportRemoteInvoke){
        this.isSupportRemoteInvoke = isSupportRemoteInvoke;
    }

    /**
     * 检查参数是否非空
     * @return 如果有一个字段为空,
     * */
    protected abstract boolean matchCondition();

    /**
     * 回复错误状态
     * */
    protected abstract T feedBackErrorResponse();

    /**
     * 是否支持service远程调用
     * */
    protected boolean supportServiceRpc(){
        return  enableRPC && isSupportRemoteInvoke && !TextUtils.isEmpty(
                findService(context,"com.meizu.cloud.pushservice.action.PUSH_MANAGER_SERVICE")
        );
    }

    /**
     * 请消息发送给远程service
     * */
    protected abstract Intent sendRpcRequest();

    /**
     * 网络请求,同步返回
     * 运行在Cloud进程中
     * */
    protected abstract T netWorkRequest();

    /**
     * 组建虚拟成功消息，本地返回
     * 运行在本应用进程中
     * */
    protected abstract T localResponse();

    /**
     * 回调消息
     * */
    protected abstract void sendReceiverMessage(T message);

    /**
     * 如果不支持远程调用,则不管成功与失败都要返回
     * */
    private boolean supportAllResponse(){
        return enableRPC && !context.getPackageName().equals(managerServicePackageName);
    }

    /**
     * 是否为业务类型异常code
     * @param code
     * */
    private boolean isServiceCode(int code){
        return code >= 110000 && code <= 200000;
    }

    /**
     * 策略类型
     * */
    protected abstract int strategyType();

    protected boolean isRegisterStatus(){
        return STRATEGY_REGISTER == strategyType() || STRATEGY_UNREGISTER == strategyType();
    }

    public boolean process(){
        boolean flag = true;
        if(executorService != null){
            executorService.execute(new Runnable() {
                @Override
                public void run() {
                    processMainThread();
                }
            });
        } else {
          flag = processMainThread();
        }
        return flag;
    }

    public boolean processMainThread(){
        boolean flag = true;
        T basicPushStatus = null;
        if(!matchCondition()){
            DebugLogger.e(TAG,"Missing required parameters");
            basicPushStatus = feedBackErrorResponse();
            sendReceiverMessage(basicPushStatus);
        } else if(!supportServiceRpc()){
            basicPushStatus = netWorkRequest();
            DebugLogger.i(TAG,"real response status "+basicPushStatus);
            if(basicPushStatus != null){
                if(isRegisterStatus() && DEVICE_ERROR_CODE.equals(basicPushStatus.getCode())){
                    return true;
                }
                if(supportAllResponse()){
                    DebugLogger.e(TAG,"response all request in local app");
                    sendReceiverMessage(basicPushStatus);
                } else {
                    String code = basicPushStatus.getCode();
                    if(TextUtils.isEmpty(code)){
                        code = "0";
                    }
                    //所有成功全部返回,服务异常错误不返回,进行重试
                    if(SUCCESS_CODE.equals(basicPushStatus.getCode())){
                        sendReceiverMessage(basicPushStatus);
                    }
                    int serviceCode = Integer.valueOf(code);
                    if(isServiceCode(serviceCode)){
                        DebugLogger.e(TAG,"service error so notify pushManager invoker code="+serviceCode+" message "+basicPushStatus.getMessage());
                        sendReceiverMessage(basicPushStatus);
                    }
                }
            }

        } else {
            DebugLogger.i(TAG,"send message to remote service");
            //不再立即回复请求消息,如果用用开启远程调用,由推送服务返回
            if(!isRegisterStatus()){
                basicPushStatus = localResponse();
                if(basicPushStatus != null){
                    DebugLogger.e(TAG,"local response "+basicPushStatus);
                    sendReceiverMessage(basicPushStatus);
                }
            }
            //转发消息到managerService
            Intent intent = sendRpcRequest();
            if(intent != null){
                sendIntentMessage(intent);
            }
        }
        if(basicPushStatus != null){
            DebugLogger.e(TAG,"current status code "+basicPushStatus.getCode());
            flag = !isServerError(basicPushStatus);
        }
        return flag;
    }


    private boolean isServerError(T basicPushStatus){
        String code = basicPushStatus.getCode();
        int errorCode = Integer.valueOf(code);
        return (errorCode > 200 && errorCode < 600)|| (errorCode > 1000 && errorCode < 2000) || errorCode == 0;
    }


//    protected String getDeviceId(){
//        deviceId = PushPreferencesUtils.getDeviceId(context);
//        DebugLogger.i(TAG, "get deviceId from local preference " + deviceId);
//        if (TextUtils.isEmpty(deviceId)) {
//            deviceId = MzSystemUtils.getDeviceId(context);
//            if (!TextUtils.isEmpty(deviceId)) {
//                DebugLogger.i(TAG, "put deviceId " + deviceId + " to preference");
//                PushPreferencesUtils.putDeviceId(context, deviceId);
//            }
//        }
//        DebugLogger.e(TAG, "deviceId " + deviceId);
//        return deviceId;
//    }

    protected String getDeviceId(){
        if(TextUtils.isEmpty(deviceId)){
            deviceId = MzSystemUtils.getDeviceId(context);
            DebugLogger.e(TAG, "deviceId " + deviceId);
        }
        return deviceId;
    }


    protected String findService(Context paramContext, String action){
        String componentName = null;
        if(!TextUtils.isEmpty(action)){
            Intent localIntent = new Intent(action);
            List<ResolveInfo> localList = paramContext.getPackageManager().queryIntentServices(localIntent, 0);
            if(localList != null){
                for (ResolveInfo info : localList){
                    if("com.meizu.cloud".equals(info.serviceInfo.packageName)){
                        managerServicePackageName = info.serviceInfo.packageName;
                        componentName = info.serviceInfo.name;
                        break;
                    }
                }
                if(TextUtils.isEmpty(componentName) && localList.size() > 0){
                    managerServicePackageName = localList.get(0).serviceInfo.packageName;
                    componentName = localList.get(0).serviceInfo.name;
                }
            }
        }
        DebugLogger.i(TAG,"current process packageName "+managerServicePackageName);
        return componentName;
    }

    protected void sendIntentMessage(Intent intent){
        try {
            intent.setPackage(managerServicePackageName);
            intent.setAction("com.meizu.cloud.pushservice.action.PUSH_MANAGER_SERVICE");
            context.startService(intent);
        } catch (Exception e){
            DebugLogger.e(TAG,"start RemoteService error "+e.getMessage());
        }

    }
}
