package com.meizu.cloud.pushsdk.platform.pushstrategy;

import android.content.Context;
import android.content.Intent;
import android.os.Handler;
import android.os.Message;
import android.text.TextUtils;
import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.networking.common.ANResponse;
import com.meizu.cloud.pushsdk.networking.error.ANError;
import com.meizu.cloud.pushsdk.platform.PlatformMessageSender;
import com.meizu.cloud.pushsdk.platform.PushIdEncryptUtils;
import com.meizu.cloud.pushsdk.platform.api.PushAPI;
import com.meizu.cloud.pushsdk.platform.message.RegisterStatus;
import com.meizu.cloud.pushsdk.pushtracer.emitter.classic.Executor;
import com.meizu.cloud.pushsdk.util.PushPreferencesUtils;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Created by liaojinlong on 17-4-17.
 */
public class RegisterStatusStrategy extends Strategy<RegisterStatus>{
    protected Handler mainHandler;
    protected ScheduledExecutorService scheduledExecutorService;
    protected int deviceIdRetry = 0;


    public RegisterStatusStrategy(Context context, String appId, String appKey, PushAPI pushAPI, ScheduledExecutorService executorService) {
        super(context, appId, appKey, pushAPI, executorService);
    }

    public RegisterStatusStrategy(Context context,PushAPI pushAPI, ScheduledExecutorService executorService) {
        this(context, null, null, pushAPI, executorService);
        scheduledExecutorService = (ScheduledExecutorService) Executor.getExecutor();
        mainHandler = new Handler(context.getMainLooper()){
            @Override
            public void handleMessage(Message msg) {
                switch (msg.what){
                    case 0:
                        process();
                        break;
                }

            }
        };
    }

    public RegisterStatusStrategy(Context context,PushAPI pushAPI, ScheduledExecutorService executorService, boolean enableRPC){
        this(context,pushAPI,executorService);
        this.enableRPC = enableRPC;
    }

    @Override
    public boolean matchCondition() {
        return !TextUtils.isEmpty(appId) && !TextUtils.isEmpty(appKey);
    }

    @Override
    protected RegisterStatus feedBackErrorResponse() {
        RegisterStatus registerStatus = new RegisterStatus();
        registerStatus.setCode(FEEDBACK_PARAMETER_ERROR_CODE);
        if(TextUtils.isEmpty(appId)){
           registerStatus.setMessage("appId not empty");
        } else if(TextUtils.isEmpty(appKey)){
           registerStatus.setMessage("appKey not empty");
        }
        return registerStatus;
    }


    @Override
    public Intent sendRpcRequest() {
        Intent intent = new Intent();
        intent.putExtra(APP_ID,appId);
        intent.putExtra(APP_KEY,appKey);
        intent.putExtra(STRATEGY_PACKAGE_NANME,context.getPackageName());
        intent.putExtra(STRATEGY_TYPE, strategyType());
        return intent;
    }

    @Override
    protected RegisterStatus localResponse() {
        return null;
    }

    @Override
    public RegisterStatus netWorkRequest() {
        RegisterStatus registerStatus = new RegisterStatus();
        String pushId = PushPreferencesUtils.getPushId(context,strategyPackageNanme);
        int expireTime = PushPreferencesUtils.getPushIdExpireTime(context,strategyPackageNanme);
        if(!retryRegister(pushId,expireTime)){
            registerStatus.setCode(SUCCESS_CODE);
            registerStatus.setMessage("already register PushId,dont register frequently");
            registerStatus.setPushId(pushId);
            registerStatus.setExpireTime((int) (expireTime - System.currentTimeMillis() / 1000));
        } else {
            //remove pushId when it expired
            PushPreferencesUtils.putPushId(context, "",strategyPackageNanme);
            deviceId = getDeviceId();
            if(TextUtils.isEmpty(deviceId) && deviceIdRetry < 3){
                DebugLogger.i(TAG, "after " + deviceIdRetry * 10 + " seconds start register");
                executeAfterGetDeviceId(deviceIdRetry * 10);
                deviceIdRetry++;
                registerStatus.setCode(DEVICE_ERROR_CODE);
                registerStatus.setMessage("deviceId is empty");
            } else {
                deviceIdRetry = 0;
                ANResponse<String> registerStatusANResponse = pushAPI.register(appId,appKey,deviceId);
                if(registerStatusANResponse.isSuccess()){
                    registerStatus = new RegisterStatus(registerStatusANResponse.getResult());
                    DebugLogger.e(TAG, "registerStatus " + registerStatus);
                    if(!TextUtils.isEmpty(registerStatus.getPushId())){
                        PushPreferencesUtils.putPushId(context,registerStatus.getPushId(),strategyPackageNanme);
                        PushPreferencesUtils.putPushIdExpireTime(context, (int) (System.currentTimeMillis()/1000 + registerStatus.getExpireTime()),strategyPackageNanme);
                    }
                } else {
                    ANError error = registerStatusANResponse.getError();
                    if (error.getResponse() != null) {
                        DebugLogger.e(TAG, "status code=" + error.getErrorCode() + " data=" + error.getResponse());
                    }
                    registerStatus.setCode(String.valueOf(error.getErrorCode()));
                    registerStatus.setMessage(error.getErrorBody());
                    DebugLogger.e(TAG, "registerStatus " + registerStatus);
                }
            }
        }
        return registerStatus;
    }

    @Override
    public void sendReceiverMessage(RegisterStatus message) {
        PlatformMessageSender.sendRegisterStatus(context,!TextUtils.isEmpty(strategyPackageNanme) ? strategyPackageNanme : context.getPackageName(), message);
    }

    @Override
    protected int strategyType() {
        return STRATEGY_REGISTER;
    }

    /**
     * 获取deviceId
     * */
    protected void executeAfterGetDeviceId(final long delay){
        scheduledExecutorService.schedule(new Runnable() {
            @Override
            public void run() {
                getDeviceId();
                mainHandler.sendEmptyMessage(0);
            }
        }, delay, TimeUnit.SECONDS);
    }


    /**
     * 是否重新发起订阅请求
     * 0.设备校验
     * 1.pushId校验
     * 2.pushId过期事件校验
     * */
    protected boolean retryRegister(String pushId,int expireTime){
        boolean flag = true;
        if(!TextUtils.isEmpty(pushId)){
            String deviceId = getDeviceId();
            if(pushId.startsWith(deviceId) || PushIdEncryptUtils.decryptPushId(pushId).startsWith(deviceId)){
                if(System.currentTimeMillis()/1000 < expireTime){
                    flag = false;
                }
            }
        }
        return flag;
    }
}
