package com.meizu.cloud.pushsdk.platform.api;

import android.content.Context;
import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.networking.common.ANResponse;
import com.meizu.cloud.pushsdk.networking.error.ANError;
import com.meizu.cloud.pushsdk.networking.http.Response;
import com.meizu.cloud.pushsdk.networking.interfaces.OkHttpResponseAndStringRequestListener;
import com.meizu.cloud.pushsdk.platform.message.StrategyMessage;
import com.meizu.cloud.pushsdk.platform.pushstrategy.RegisterStatusStrategy;
import com.meizu.cloud.pushsdk.platform.pushstrategy.Strategy;
import com.meizu.cloud.pushsdk.platform.pushstrategy.SubScribeAliasStrategy;
import com.meizu.cloud.pushsdk.platform.pushstrategy.SubScribeTagStrategy;
import com.meizu.cloud.pushsdk.platform.pushstrategy.SwitchStatusStrategy;
import com.meizu.cloud.pushsdk.platform.pushstrategy.UnRegisterStatusStrategy;
import com.meizu.cloud.pushsdk.pushtracer.emitter.classic.Executor;

import java.io.File;
import java.util.concurrent.ScheduledExecutorService;


/**
 * Created by liaojinlong on 16-11-25.
 */
public class PushPlatformManager {
    private static final String TAG = "PushPlatformManager";
    private static PushPlatformManager mInstance;
    private ScheduledExecutorService executorService;
    private Context mContext;
    private PushAPI pushAPI;
    private RegisterStatusStrategy registerStatusStrategy;
    private UnRegisterStatusStrategy unRegisterStatusStrategy;
    private SwitchStatusStrategy switchStatusStrategy;
    private SubScribeTagStrategy subScribeTagStrategy;
    private SubScribeAliasStrategy subScribeAliasStrategy;

    public PushPlatformManager(Context context,boolean isAsync){
        this(context,isAsync,true);
    }

    public PushPlatformManager(Context context,boolean isAsync,boolean enableRPC){
        this.mContext = context.getApplicationContext();
        this.pushAPI = new PushAPI(mContext);
        if(isAsync){
            executorService = (ScheduledExecutorService) Executor.getExecutor();
        }
        registerStatusStrategy = new RegisterStatusStrategy(mContext,pushAPI,executorService,enableRPC);
        unRegisterStatusStrategy = new UnRegisterStatusStrategy(mContext,pushAPI,executorService,enableRPC);
        switchStatusStrategy = new SwitchStatusStrategy(mContext,pushAPI,executorService,enableRPC);
        subScribeTagStrategy = new SubScribeTagStrategy(mContext,pushAPI,executorService,enableRPC);
        subScribeAliasStrategy = new SubScribeAliasStrategy(mContext,pushAPI,executorService,enableRPC);
    }

    public static PushPlatformManager getInstance(Context context){
        if(mInstance == null){
            synchronized (PushPlatformManager.class){
                if(mInstance == null){
                    mInstance = new PushPlatformManager(context,true);
                }
            }
        }
        return mInstance;
    }

    public void enableRemoteInvoker(boolean isRemoteInvoker){
        registerStatusStrategy.setSupportRemoteInvoke(isRemoteInvoker);
        unRegisterStatusStrategy.setSupportRemoteInvoke(isRemoteInvoker);
        switchStatusStrategy.setSupportRemoteInvoke(isRemoteInvoker);
        subScribeAliasStrategy.setSupportRemoteInvoke(isRemoteInvoker);
        subScribeTagStrategy.setSupportRemoteInvoke(isRemoteInvoker);
    }

    /**
     * 分发消息
     * @param strategyMessage
     * */
    public boolean dispatcherStrategyMessage(StrategyMessage strategyMessage){
        boolean flag = true;
        if(strategyMessage != null){
            int strategyType = strategyMessage.getStrategyType();
            switch (strategyType){
                case Strategy.STRATEGY_REGISTER:
                    flag = register(strategyMessage.getAppId(),strategyMessage.getAppKey(),strategyMessage.getPackageName());
                    break;
                case Strategy.STRATEGY_UNREGISTER:
                    flag = unRegister(strategyMessage.getAppId(),strategyMessage.getAppKey(),strategyMessage.getPackageName());
                    break;
                case Strategy.STRATEGY_SUBTAGS:
                    if(SubScribeTagStrategy.SUB_TAGS == strategyMessage.getStrategyChildType()){
                        flag = subScribeTags(strategyMessage.getAppId(),strategyMessage.getAppKey(),strategyMessage.getPackageName(),strategyMessage.getPushId(), strategyMessage.getParams());
                    } else if(SubScribeTagStrategy.CHECK_SUB_TAGS == strategyMessage.getStrategyChildType()){
                        flag = checkSubScribeTags(strategyMessage.getAppId(),strategyMessage.getAppKey(),strategyMessage.getPackageName(),strategyMessage.getPushId());
                    } else if(SubScribeTagStrategy.UNSUB_TAGS == strategyMessage.getStrategyChildType()){
                        flag = unSubScribeTags(strategyMessage.getAppId(), strategyMessage.getAppKey(), strategyMessage.getPackageName(), strategyMessage.getPushId(), strategyMessage.getParams());
                    } else if(SubScribeTagStrategy.UNSUB_ALL_TAGS == strategyMessage.getStrategyChildType()){
                        flag = unSubScribeAllTags(strategyMessage.getAppId(),strategyMessage.getAppKey(),strategyMessage.getPackageName(),strategyMessage.getPushId());
                    }
                    break;
                case Strategy.STRATEGY_SWITCH:
                    if(SwitchStatusStrategy.SWITCH_NOTIFICATION == strategyMessage.getStrategyChildType()){
                        flag = switchPush(strategyMessage.getAppId(), strategyMessage.getAppKey(), strategyMessage.getPackageName(), strategyMessage.getPushId(),SwitchStatusStrategy.SWITCH_NOTIFICATION,
                                "1".equals(strategyMessage.getParams()) ? true : false);
                    } else if(SwitchStatusStrategy.SWITCH_THROUGH_MESSAGE == strategyMessage.getStrategyChildType()){
                        flag = switchPush(strategyMessage.getAppId(), strategyMessage.getAppKey(), strategyMessage.getPackageName(), strategyMessage.getPushId(),SwitchStatusStrategy.SWITCH_THROUGH_MESSAGE,
                                "1".equals(strategyMessage.getParams()) ? true : false);
                    } else if(SwitchStatusStrategy.SWITCH_ALL == strategyMessage.getStrategyChildType()){
                        flag = switchPush(strategyMessage.getAppId(), strategyMessage.getAppKey(), strategyMessage.getPackageName(), strategyMessage.getPushId(),
                                "1".equals(strategyMessage.getParams()) ? true : false);
                    } else if(SwitchStatusStrategy.CHECK_PUSH == strategyMessage.getStrategyChildType()){
                        flag = checkPush(strategyMessage.getAppId(), strategyMessage.getAppKey(), strategyMessage.getPackageName(), strategyMessage.getPushId());
                    }
                    break;
                case Strategy.STRATEGY_SUBALIAS:
                    if(SubScribeAliasStrategy.SUB_ALIAS == strategyMessage.getStrategyChildType()){
                        flag = subScribeAlias(strategyMessage.getAppId(), strategyMessage.getAppKey(), strategyMessage.getPackageName(), strategyMessage.getPushId(),strategyMessage.getParams());
                    } else if(SubScribeAliasStrategy.UNSUB_ALIAS == strategyMessage.getStrategyChildType()){
                        flag = unSubScribeAlias(strategyMessage.getAppId(), strategyMessage.getAppKey(), strategyMessage.getPackageName(), strategyMessage.getPushId(),strategyMessage.getParams());
                    } else if(SubScribeAliasStrategy.CHECK_ALIAS == strategyMessage.getStrategyChildType()) {
                        flag = checkSubScribeAlias(strategyMessage.getAppId(), strategyMessage.getAppKey(), strategyMessage.getPackageName(), strategyMessage.getPushId());
                    }
                    break;
            }
        }
        return flag;
    }

    /**
     * pushId 订阅 PushManger 接口使用
     * @param appId
     * @param appKey
     * */
    public boolean register(String appId,String appKey,String packageName){
        registerStatusStrategy.setAppId(appId);
        registerStatusStrategy.setAppKey(appKey);
        registerStatusStrategy.setStrategyPackageNanme(packageName);
        return registerStatusStrategy.process();
    }

    public boolean unRegister(String appId,String appKey,String packageName){
        unRegisterStatusStrategy.setAppId(appId);
        unRegisterStatusStrategy.setAppKey(appKey);
        unRegisterStatusStrategy.setStrategyPackageNanme(packageName);
        return unRegisterStatusStrategy.process();
    }


    public void unRegisterAdvance(final String packageName,String deviceId){
        pushAPI.unRegister(packageName, deviceId, new OkHttpResponseAndStringRequestListener() {
            @Override
            public void onResponse(Response okHttpResponse, String response) {
                DebugLogger.e(TAG, "unregisetr advance pakcage " + packageName + " result " + response);
            }

            @Override
            public void onError(ANError anError) {
                DebugLogger.e(TAG, "unregisetr advance pakcage " + packageName + " error " + anError.getErrorBody());
            }
        });
    }



    public boolean checkPush(String appId,String appKey,String packageName, String pushId){
        switchStatusStrategy.setAppId(appId);
        switchStatusStrategy.setAppKey(appKey);
        switchStatusStrategy.setStrategyPackageNanme(packageName);
        switchStatusStrategy.setPushId(pushId);
        switchStatusStrategy.setSwitchType(SwitchStatusStrategy.CHECK_PUSH);
        return switchStatusStrategy.process();
    }

    public boolean switchPush(String appId,String appKey,String packageName, String pushId, int msgType,boolean switcher){
        switchStatusStrategy.setAppId(appId);
        switchStatusStrategy.setAppKey(appKey);
        switchStatusStrategy.setStrategyPackageNanme(packageName);
        switchStatusStrategy.setPushId(pushId);
        switchStatusStrategy.setSwitchType(msgType);
        switchStatusStrategy.setSwitcher(switcher);
        return switchStatusStrategy.process();
    }

    public boolean switchPush(String appId,String appKey,String packageName, String pushId,boolean switcher){
        switchStatusStrategy.setAppId(appId);
        switchStatusStrategy.setAppKey(appKey);
        switchStatusStrategy.setStrategyPackageNanme(packageName);
        switchStatusStrategy.setPushId(pushId);
        switchStatusStrategy.setSwitchType(SwitchStatusStrategy.SWITCH_ALL);
        switchStatusStrategy.setSwitcher(switcher);
        return switchStatusStrategy.process();
    }

    public boolean subScribeTags(String appId,String appKey,String packageName,String pushId,String tags){
        subScribeTagStrategy.setAppId(appId);
        subScribeTagStrategy.setAppKey(appKey);
        subScribeTagStrategy.setStrategyPackageNanme(packageName);
        subScribeTagStrategy.setPushId(pushId);
        subScribeTagStrategy.setSubTagType(SubScribeTagStrategy.SUB_TAGS);
        subScribeTagStrategy.setSubTags(tags);
        return subScribeTagStrategy.process();
    }

    public boolean unSubScribeTags(String appId,String appKey,String packageName,String pushId,String tags){
        subScribeTagStrategy.setAppId(appId);
        subScribeTagStrategy.setAppKey(appKey);
        subScribeTagStrategy.setStrategyPackageNanme(packageName);
        subScribeTagStrategy.setPushId(pushId);
        subScribeTagStrategy.setSubTagType(SubScribeTagStrategy.UNSUB_TAGS);
        subScribeTagStrategy.setSubTags(tags);
        return subScribeTagStrategy.process();
    }

    public boolean unSubScribeAllTags(String appId,String appKey,String packageName,String pushId){
        subScribeTagStrategy.setAppId(appId);
        subScribeTagStrategy.setAppKey(appKey);
        subScribeTagStrategy.setStrategyPackageNanme(packageName);
        subScribeTagStrategy.setPushId(pushId);
        subScribeTagStrategy.setSubTagType(SubScribeTagStrategy.UNSUB_ALL_TAGS);
        return subScribeTagStrategy.process();
    }

    public boolean checkSubScribeTags(String appId,String appKey,String packageName,String pushId){
        subScribeTagStrategy.setAppId(appId);
        subScribeTagStrategy.setAppKey(appKey);
        subScribeTagStrategy.setStrategyPackageNanme(packageName);
        subScribeTagStrategy.setPushId(pushId);
        subScribeTagStrategy.setSubTagType(SubScribeTagStrategy.CHECK_SUB_TAGS);
        return subScribeTagStrategy.process();
    }

    public boolean subScribeAlias(String appId,String appKey,String packageName,String pushId,String alias){
        subScribeAliasStrategy.setAppId(appId);
        subScribeAliasStrategy.setAppKey(appKey);
        subScribeAliasStrategy.setStrategyPackageNanme(packageName);
        subScribeAliasStrategy.setPushId(pushId);
        subScribeAliasStrategy.setSubAliasType(SubScribeAliasStrategy.SUB_ALIAS);
        subScribeAliasStrategy.setAlias(alias);
        return subScribeAliasStrategy.process();
    }

    public boolean unSubScribeAlias(String appId,String appKey,String packageName,String pushId,String alias){
        subScribeAliasStrategy.setAppId(appId);
        subScribeAliasStrategy.setAppKey(appKey);
        subScribeAliasStrategy.setStrategyPackageNanme(packageName);
        subScribeAliasStrategy.setPushId(pushId);
        subScribeAliasStrategy.setSubAliasType(SubScribeAliasStrategy.UNSUB_ALIAS);
        subScribeAliasStrategy.setAlias(alias);
        return subScribeAliasStrategy.process();
    }

    public boolean checkSubScribeAlias(String appId,String appKey,String packageName,String pushId){
        subScribeAliasStrategy.setAppId(appId);
        subScribeAliasStrategy.setAppKey(appKey);
        subScribeAliasStrategy.setStrategyPackageNanme(packageName);
        subScribeAliasStrategy.setPushId(pushId);
        subScribeAliasStrategy.setSubAliasType(SubScribeAliasStrategy.CHECK_ALIAS);
        return subScribeAliasStrategy.process();
    }


    public ANResponse<String> uploadLogFile(String messageId,String deviceId,String errorMsg,File logFile){
        return pushAPI.uploadLogFile(messageId,deviceId,errorMsg,logFile);
    }
}
