package com.meizu.cloud.pushsdk.notification;

import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.KeyguardManager;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Icon;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.PowerManager;
import android.text.TextUtils;
import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushinternal.R;
import com.meizu.cloud.pushsdk.constants.PushConstants;
import com.meizu.cloud.pushsdk.handler.MessageV3;
import com.meizu.cloud.pushsdk.handler.impl.model.NotificationState;
import com.meizu.cloud.pushsdk.networking.AndroidNetworking;
import com.meizu.cloud.pushsdk.networking.common.ANRequest;
import com.meizu.cloud.pushsdk.networking.common.ANResponse;
import com.meizu.cloud.pushsdk.notification.model.AdvanceSetting;
import com.meizu.cloud.pushsdk.notification.util.NotificationUtils;
import com.meizu.cloud.pushsdk.notification.util.RProxy;
import com.meizu.cloud.pushsdk.util.MinSdkChecker;
import com.meizu.cloud.pushsdk.util.MzSystemUtils;
import com.meizu.cloud.pushsdk.util.PushPreferencesUtils;

import org.json.JSONObject;

/**
 * 支持Notification自定义
 */
public abstract class AbstractPushNotification implements PushNotification {
    protected static final String TAG = "AbstractPushNotification";
    protected Context context;
    protected PushNotificationBuilder pushNotificationBuilder;
    private NotificationManager notificationManager;
    protected Handler handler;
    protected AbstractPushNotification(Context context, PushNotificationBuilder pushNotificationBuilder) {
        this.pushNotificationBuilder = pushNotificationBuilder;
        this.context = context;
        handler = new Handler(context.getMainLooper());
        this.notificationManager = (NotificationManager)context.getSystemService(Context.NOTIFICATION_SERVICE);
    }

    /**
     * builder for expandable content such text and picture
     * this content can build with NotificationCompat builder
     * @param builder
     *             the notification builder
     * @param messageV3
     *              message which construct notification
     * */
    protected void buildExpandableContent(Notification.Builder builder, MessageV3 messageV3){}

    /**
     * for build notification contentview
     * @param notification
     * @param messageV3
     * */
    protected void buildContentView(Notification notification,MessageV3 messageV3){}

    /**
     * for build notification bigContentView
     * @param messageV3
     * @param notification
     * */
    protected void buildBigContentView(Notification notification,MessageV3 messageV3){}


    /**
     * notification builder set app icon
     * @param builder
     * @param messageV3
     * */
    protected void appIconSettingBuilder(Notification.Builder builder, MessageV3 messageV3){}

    /**
     * construct Notification
     * notification builder
     *                build notification expandable Content
     * notification contentView
     *                build notification contentView and bigContentView
     * */
    protected Notification construtNotificationFinal(MessageV3 messageV3,PendingIntent clickIntent, PendingIntent deleteIntent){
        Notification.Builder builder = new Notification.Builder(context);
        basicSettingBuilder(builder, messageV3, clickIntent, deleteIntent);
        advanceSettingBuilder(builder, messageV3);

        appIconSettingBuilder(builder, messageV3);
        buildExpandableContent(builder, messageV3);
        Notification notification;
        if (MinSdkChecker.isSupportNotificationBuild()) {
            notification = builder.build();
        } else {
            notification = builder.getNotification();
        }
        buildContentView(notification, messageV3);
        buildBigContentView(notification, messageV3);
        return notification;
    }

    /**
     * create click intent for click callback
     * 此回调需要跳转到目标应用内回调
     * @param messageV3
     * */
    protected PendingIntent createClickIntent(MessageV3 messageV3){
        Intent serviceClickIntent = new Intent();
        serviceClickIntent.setData(Uri.parse("custom://" + System.currentTimeMillis()));
        serviceClickIntent.putExtra(PushConstants.MZ_PUSH_PRIVATE_MESSAGE, messageV3);
        serviceClickIntent.putExtra(PushConstants.MZ_PUSH_MESSAGE_METHOD, PushConstants.MZ_PUSH_MESSAGE_METHOD_ACTION_PRIVATE);
        String messageReceiver = MzSystemUtils.findReceiver(context, PushConstants.MZ_PUSH_ON_MESSAGE_ACTION, messageV3.getUploadDataPackageName());
        serviceClickIntent.setClassName(messageV3.getUploadDataPackageName(), messageReceiver);
        serviceClickIntent.setAction(PushConstants.MZ_PUSH_ON_MESSAGE_ACTION);
        //不同弹出的通知栏Extra数据不能覆盖
        PendingIntent clickPendingIntent = PendingIntent.getBroadcast(context, 0, serviceClickIntent, PendingIntent.FLAG_ONE_SHOT);
        return clickPendingIntent;
    }

    /**
     * create delete Intent for delete notification
     * 此回调在通知栏发送应用内回调即可
     * @param messageV3
     * */
    protected PendingIntent createDeleteIntent(MessageV3 messageV3){
        Intent serviceDeleteIntent = new Intent();
        //区分intent传递不同的notification,防止由于数据相同导致intent覆盖
        serviceDeleteIntent.setData(Uri.parse("custom://" + System.currentTimeMillis()));
        serviceDeleteIntent.putExtra(PushConstants.MZ_PUSH_PRIVATE_MESSAGE, messageV3);
        serviceDeleteIntent.putExtra(PushConstants.MZ_PUSH_MESSAGE_METHOD, PushConstants.MZ_PUSH_MESSAGE_METHOD_ACTION_NOTIFICATION_DELETE);
        String messageReceiver = MzSystemUtils.findReceiver(context, PushConstants.MZ_PUSH_ON_MESSAGE_ACTION, messageV3.getPackageName());
        serviceDeleteIntent.setClassName(messageV3.getPackageName(), messageReceiver);
        serviceDeleteIntent.setAction(PushConstants.MZ_PUSH_ON_MESSAGE_ACTION);
        PendingIntent deletePendingIntent = PendingIntent.getBroadcast(context, 0, serviceDeleteIntent, PendingIntent.FLAG_ONE_SHOT);
        return deletePendingIntent;
    }

    /**
     * 创建notification 状态回调,用于通知栏数据上报统计
     * 此回调在通知栏发送应用内回调即可
     * @param messageV3
     * */
    protected PendingIntent createStateIntent(MessageV3 messageV3){
        Intent serviceStateIntent = new Intent();
        //区分intent传递不同的notification,防止由于数据相同导致intent覆盖
        serviceStateIntent.setData(Uri.parse("custom://" + System.currentTimeMillis()));
        serviceStateIntent.putExtra(PushConstants.MZ_PUSH_NOTIFICATION_STATE_MESSAGE, messageV3.getNotificationMessage());
        serviceStateIntent.putExtra(PushConstants.NOTIFICATION_EXTRA_TASK_ID,messageV3.getTaskId());
        serviceStateIntent.putExtra(PushConstants.NOTIFICATION_EXTRA_SEQ_ID,messageV3.getSeqId());
        serviceStateIntent.putExtra(PushConstants.NOTIFICATION_EXTRA_DEVICE_ID,messageV3.getDeviceId());
        serviceStateIntent.putExtra(PushConstants.NOTIFICATION_EXTRA_PUSH_TIMESTAMP,messageV3.getPushTimestamp());
        serviceStateIntent.putExtra(PushConstants.NOTIFICATION_EXTRA_SHOW_PACKAGE_NAME,messageV3.getUploadDataPackageName());
        serviceStateIntent.putExtra(PushConstants.MZ_PUSH_MESSAGE_METHOD, PushConstants.MZ_PUSH_MESSAGE_METHOD_ACTION_NOTIFICATION_STATE);
        String messageReceiver = MzSystemUtils.findReceiver(context, PushConstants.MZ_PUSH_ON_MESSAGE_ACTION, messageV3.getPackageName());
        serviceStateIntent.setClassName(messageV3.getPackageName(), messageReceiver);
        serviceStateIntent.setAction(PushConstants.MZ_PUSH_ON_MESSAGE_ACTION);
        PendingIntent deletePendingIntent = PendingIntent.getBroadcast(context, 0, serviceStateIntent, PendingIntent.FLAG_ONE_SHOT);
        return deletePendingIntent;
    }
    /**
     * basic setting include
     * title: the notification title
     * content: the notification main content
     * autoCancel: notification disappear when click it
     * smallIcon: the icon show in the status bar,the default icon is the application icon
     * clickIntent: action when click notification
     * deleteIntent: action when delete notification
     * */
    protected void basicSettingBuilder(Notification.Builder builder, MessageV3 messageV3, PendingIntent clickIntent, PendingIntent deleteIntent){
        builder.setContentTitle(messageV3.getTitle());
        builder.setContentText(messageV3.getContent());
        builder.setTicker(messageV3.getContent());
        builder.setAutoCancel(true);
        //set visible when screen lock
        if(MinSdkChecker.isSupportSendNotification()){
            builder.setVisibility(Notification.VISIBILITY_PUBLIC);
        }
        if(MinSdkChecker.isSupportSetDrawableSmallIcon()){
            Icon smallIcon = loadSmallIcon(messageV3.getUploadDataPackageName());
            if(smallIcon != null){
                builder.setSmallIcon(smallIcon);
            } else {
                DebugLogger.e(TAG, "cannot get " + messageV3.getUploadDataPackageName() + " smallIcon");
                builder.setSmallIcon(RProxy.stat_sys_third_app_notify(context));
            }
        } else {
            //不再默认设置应用大图标为状态栏图标,直接用第三方默认图标
            builder.setSmallIcon((pushNotificationBuilder != null && pushNotificationBuilder.getmStatusbarIcon() != 0) ?
                pushNotificationBuilder.getmStatusbarIcon() : RProxy.stat_sys_third_app_notify(context));
        }

        builder.setContentIntent(clickIntent);
        builder.setDeleteIntent(deleteIntent);
    }

    /**
     * default: light,vibrate,sound and all
     * onGoing: cannot delete notification if you do not click the notification
     * HeadUpNotification: show global notification
     * */
    protected void advanceSettingBuilder(Notification.Builder builder, MessageV3 messageV3){
        AdvanceSetting advanceSetting = messageV3.getmAdvanceSetting();
        if(advanceSetting != null){
            if(advanceSetting.getNotifyType() != null){
                boolean vibrate = advanceSetting.getNotifyType().isVibrate();
                boolean lights = advanceSetting.getNotifyType().isLights();
                boolean sound = advanceSetting.getNotifyType().isSound();
                if(vibrate || lights || sound){
                    int defaultType = 0;
                    if(vibrate){
                        defaultType |= Notification.DEFAULT_VIBRATE;
                    }
                    if(lights){
                        defaultType |= Notification.DEFAULT_LIGHTS;
                        //defaultType |= Notification.FLAG_SHOW_LIGHTS;
                        //builder.setLights(0xff00ff00,300,1000);
                    }
                    if(sound){
                        defaultType |= Notification.DEFAULT_SOUND;
                    }
                    DebugLogger.e(TAG,"current notification type is "+defaultType);
                    builder.setDefaults(defaultType);
                }

            }
            builder.setOngoing(!advanceSetting.isClearNotification());
            if(advanceSetting.isHeadUpNotification()){
                if(MinSdkChecker.isSupportNotificationBuild()){
                    builder.setPriority(Notification.PRIORITY_MAX);
                }
                //setFullScreenIntent when screen off it will auto click
                //builder.setFullScreenIntent(clickIntent, true);
                //boolean isScreenOn = isScreenOnAndUnlock();
                //DebugLogger.e(TAG,"current screen is "+isScreenOn);
                //builder.setOngoing(!isScreenOn && !advanceSetting.isClearNotification());
            }
        }
    }

    public Bitmap getBitmapFromURL(String src) {
        ANRequest request = AndroidNetworking.get(src).build();
        ANResponse<Bitmap> response = request.executeForBitmap();
        if(response.isSuccess() && response.getResult() != null){
            DebugLogger.i(TAG,"ANRequest On other Thread down load largeIcon "+src+ "image "+(response.getResult() != null? "success":"fail"));
            return response.getResult();
        } else {
            DebugLogger.i(TAG, "ANRequest On other Thread down load largeIcon " + src + "image fail");
            return null;
        }
    }

    /**
     * 根据报名获取应用状态栏小图标
     * @param despackageName 目标应用包名
     * */
    @TargetApi(Build.VERSION_CODES.M)
    private Icon loadSmallIcon(String despackageName) {
        Icon smallIcon = null;
        try {
            Resources res = context.getPackageManager().getResourcesForApplication(despackageName);
            int titleId = res.getIdentifier(PushConstants.MZ_PUSH_NOTIFICATION_SMALL_ICON, "drawable", despackageName);
            if(titleId != 0){
                DebugLogger.i(TAG, "get " + despackageName + " smallIcon success resId " + titleId);
                smallIcon = Icon.createWithResource(despackageName, titleId);
            }
        } catch (Exception e) {
            DebugLogger.e(TAG, "cannot load smallIcon form package " + despackageName + " Error message " + e.getMessage());
        }
        return smallIcon;

    }

    /**
     * 获取当前应用图标
     * */
    public Bitmap getAppIcon(Context context,String packageName) {
        Bitmap appIcon;
        try {
            PackageManager packageManager = context.getPackageManager();
            BitmapDrawable bitmapDrawable = (BitmapDrawable) packageManager.getApplicationIcon(packageName);
            appIcon = bitmapDrawable.getBitmap();
        } catch (PackageManager.NameNotFoundException e) {
            DebugLogger.i(TAG, "getappicon error " + e.getMessage());
            BitmapDrawable bitmapDrawable = (BitmapDrawable)context.getApplicationInfo().loadIcon(context.getPackageManager());
            appIcon = bitmapDrawable.getBitmap();
        }
        return appIcon;
    }

    /**
     * check current thread
     * */
    protected boolean isOnMainThread(){
        return Thread.currentThread() == context.getMainLooper().getThread();
    }

    /**
     * check screen status and screen lock status
     * */
    protected boolean isScreenOnAndUnlock(){
        PowerManager powerManager = (PowerManager) context.getSystemService(Context.POWER_SERVICE);
        if(Build.VERSION.SDK_INT < 20 ? powerManager.isScreenOn():powerManager.isInteractive()){
            KeyguardManager keyguardManager = (KeyguardManager) context.getSystemService(Context.KEYGUARD_SERVICE);
            if(!keyguardManager.inKeyguardRestrictedInputMode()){
                return true;
            }
        }
        return false;
    }

    /**
     * 获取Flyme绿色通道的配置
     * */
    protected String getFlymeGreenNotificationSetting(MessageV3 messageV3){
        String flymeNotificationSetting = null;
        try {
            if(!TextUtils.isEmpty(messageV3.getNotificationMessage())){
                JSONObject jsonObject = new JSONObject(messageV3.getNotificationMessage());
                flymeNotificationSetting = jsonObject.getJSONObject("data").getJSONObject("extra").getString("fns");
            }

        } catch (Exception e){
            DebugLogger.e(TAG,"parse flyme notifification setting error "+e.getMessage());
        }
        DebugLogger.i(TAG,"current notification setting is "+flymeNotificationSetting);
        return flymeNotificationSetting;
    }


    @Override
    @SuppressLint("NewApi")
    public void show(MessageV3 messageV3) {
        Notification notification = construtNotificationFinal(messageV3, createClickIntent(messageV3), createDeleteIntent(messageV3));
        //设置该应用为系统内部应用,用于支持自定义状态栏图标
        NotificationUtils.setInternalApp(notification,true);
        //设置监听通知栏状态回调
        NotificationUtils.setReplyIntent(notification, createStateIntent(messageV3));
        //设置通知栏源包名
        notification.extras.putString(PushConstants.EXTRA_ORIGINAL_NOTIFICATION_PACKAGE_NAME, messageV3.getUploadDataPackageName());
        //设置绿色通道信息
        notification.extras.putString(PushConstants.EXTRA_FLYME_GREEN_NOTIFICATION_SETTING,getFlymeGreenNotificationSetting(messageV3));
        //设置任务信息
        notification.extras.putString(PushConstants.NOTIFICATION_EXTRA_TASK_ID,messageV3.getTaskId());
        notification.extras.putString(PushConstants.NOTIFICATION_EXTRA_SEQ_ID,messageV3.getSeqId());
        notification.extras.putString(PushConstants.NOTIFICATION_EXTRA_DEVICE_ID,messageV3.getDeviceId());
        notification.extras.putString(PushConstants.NOTIFICATION_EXTRA_PUSH_TIMESTAMP, messageV3.getPushTimestamp());

        int notifyId = (int) System.currentTimeMillis();
        if(messageV3.isDiscard()){
            if(PushPreferencesUtils.getDiscardNotificationId(context, messageV3.getPackageName()) == 0){
                PushPreferencesUtils.putDiscardNotificationIdByPackageName(context,messageV3.getPackageName(),notifyId);
                DebugLogger.i(TAG,"no notification show so put notification id "+notifyId);
            }
            if(!TextUtils.isEmpty(messageV3.getTaskId())){
                if(PushPreferencesUtils.getDiscardNotificationTaskId(context,messageV3.getPackageName()) == 0){
                    PushPreferencesUtils.putDiscardNotificationTaskId(context,messageV3.getPackageName(),Integer.valueOf(messageV3.getTaskId()));
                } else {
                    if(Integer.valueOf(messageV3.getTaskId()) < PushPreferencesUtils.getDiscardNotificationTaskId(context,messageV3.getPackageName())){
                        DebugLogger.i(TAG, "current package " + messageV3.getPackageName() + " taskid " + messageV3.getTaskId() + " dont show notification");
                        return;
                    } else {
                        PushPreferencesUtils.putDiscardNotificationTaskId(context,messageV3.getPackageName(),Integer.valueOf(messageV3.getTaskId()));
                        notifyId = PushPreferencesUtils.getDiscardNotificationId(context,messageV3.getPackageName());
                    }
                }
            }
            DebugLogger.i(TAG, "current package " + messageV3.getPackageName() + " notificationId=" + notifyId + " taskId=" + messageV3.getTaskId());
        }
        notificationManager.notify(notifyId, notification);
        dismissFloatNotification(notificationManager,notifyId,messageV3);
    }

    /**
     * show head_notification and setOngoing dismiss float notification and show normal notification
     * */
    protected void dismissFloatNotification(final NotificationManager notificationManager, final int notificationId,MessageV3 messageV3){
        AdvanceSetting advanceSetting = messageV3.getmAdvanceSetting();
        if(advanceSetting != null){
            boolean isFloatNotification = advanceSetting.isHeadUpNotification();
            boolean isClearNotification = advanceSetting.isClearNotification();
            if(isFloatNotification && !isClearNotification){
                messageV3.getmAdvanceSetting().setHeadUpNotification(false);
                messageV3.getmAdvanceSetting().getNotifyType().setSound(false);
                messageV3.getmAdvanceSetting().getNotifyType().setVibrate(false);
                final Notification notification = construtNotificationFinal(messageV3,createClickIntent(messageV3),createDeleteIntent(messageV3));
                //refresh notification for clear head up notification
                handler.postDelayed(new Runnable() {
                    @Override
                    public void run() {
                        notificationManager.notify(notificationId,notification);
                    }
                },5000);
            }
        }
    }
}
