package com.meizu.cloud.pushsdk.networking.http;


import com.meizu.cloud.pushsdk.networking.okio.Buffer;
import com.meizu.cloud.pushsdk.networking.okio.BufferedSource;

import java.io.IOException;
import java.util.List;

/**
 * Created by liaojinlong on 16-12-21.
 */
public class Response {
    private final Request request;
    private final int code;
    private final String message;
    private final Headers headers;
    private final ResponseBody body;
    private Response networkResponse;
    private Response cacheResponse;
    private final Response priorResponse;


    private Response(Builder builder) {
        this.request = builder.request;
        this.code = builder.code;
        this.message = builder.message;
        this.headers = builder.headers.build();
        this.body = builder.body;
        this.networkResponse = builder.networkResponse;
        this.cacheResponse = builder.cacheResponse;
        this.priorResponse = builder.priorResponse;
    }

    /**
     * The wire-level request that initiated this HTTP response. This is not
     * necessarily the same request issued by the application:
     *
     * <ul>
     *     <li>It may be transformed by the HTTP client. For example, the client
     *         may copy headers like {@code Content-Length} from the request body.
     *     <li>It may be the request generated in response to an HTTP redirect or
     *         authentication challenge. In this case the request URL may be
     *         different than the initial request URL.
     * </ul>
     */
    public Request request() {
        return request;
    }


    /** Returns the HTTP status code. */
    public int code() {
        return code;
    }

    /**
     * Returns true if the code is in [200..300), which means the request was successfully received,
     * understood, and accepted.
     */
    public boolean isSuccessful() {
        return code >= 200 && code < 300;
    }

    /** Returns the HTTP status message or null if it is unknown. */
    public String message() {
        return message;
    }

    public List<String> headers(String name) {
        return headers.values(name);
    }

    public String header(String name) {
        return header(name, null);
    }

    public String header(String name, String defaultValue) {
        String result = headers.get(name);
        return result != null ? result : defaultValue;
    }

    public Headers headers() {
        return headers;
    }

    /**
     * Peeks up to {@code byteCount} bytes from the response body and returns them as a new response
     * body. If fewer than {@code byteCount} bytes are in the response body, the full response body is
     * returned. If more than {@code byteCount} bytes are in the response body, the returned value
     * will be truncated to {@code byteCount} bytes.
     *
     * <p>It is an error to call this method after the body has been consumed.
     *
     * <p><strong>Warning:</strong> this method loads the requested bytes into memory. Most
     * applications should set a modest limit on {@code byteCount}, such as 1 MiB.
     */
    public ResponseBody peekBody(long byteCount) throws IOException {
        BufferedSource source = body.source();
        source.request(byteCount);
        Buffer copy = source.buffer().clone();

        // There may be more than byteCount bytes in source.buffer(). If there is, return a prefix.
        Buffer result;
        if (copy.size() > byteCount) {
            result = new Buffer();
            result.write(copy, byteCount);
            copy.clear();
        } else {
            result = copy;
        }

        return ResponseBody.create(body.contentType(), result.size(), result);
    }

    public ResponseBody body() {
        return body;
    }

    public Builder newBuilder() {
        return new Builder(this);
    }

    /**
     * Returns the raw response received from the network. Will be null if this response didn't use
     * the network, such as when the response is fully cached. The body of the returned response
     * should not be read.
     */
    public Response networkResponse() {
        return networkResponse;
    }

    /**
     * Returns the raw response received from the cache. Will be null if this response didn't use the
     * cache. For conditional get requests the cache response and network response may both be
     * non-null. The body of the returned response should not be read.
     */
    public Response cacheResponse() {
        return cacheResponse;
    }

    /**
     * Returns the response for the HTTP redirect or authorization challenge that triggered this
     * response, or null if this response wasn't triggered by an automatic retry. The body of the
     * returned response should not be read because it has already been consumed by the redirecting
     * client.
     */
    public Response priorResponse() {
        return priorResponse;
    }



    @Override public String toString() {
        return "Response{protocol="
                + ", code="
                + code
                + ", message="
                + message
                + ", url="
                + request.url()
                + '}';
    }

    public static class Builder {
        private Request request;
        private int code = -1;
        private String message;
        private Headers.Builder headers;
        private ResponseBody body;
        private Response networkResponse;
        private Response cacheResponse;
        private Response priorResponse;

        public Builder() {
            headers = new Headers.Builder();
        }

        private Builder(Response response) {
            this.request = response.request;
            this.code = response.code;
            this.message = response.message;
            this.headers = response.headers.newBuilder();
            this.body = response.body;
            this.networkResponse = response.networkResponse;
            this.cacheResponse = response.cacheResponse;
            this.priorResponse = response.priorResponse;
        }

        public Builder request(Request request) {
            this.request = request;
            return this;
        }

        public Builder code(int code) {
            this.code = code;
            return this;
        }

        public Builder message(String message) {
            this.message = message;
            return this;
        }

        /**
         * Sets the header named {@code name} to {@code value}. If this request already has any headers
         * with that name, they are all replaced.
         */
        public Builder header(String name, String value) {
            headers.set(name, value);
            return this;
        }

        /**
         * Adds a header with {@code name} and {@code value}. Prefer this method for multiply-valued
         * headers like "Set-Cookie".
         */
        public Builder addHeader(String name, String value) {
            headers.add(name, value);
            return this;
        }

        public Builder removeHeader(String name) {
            headers.removeAll(name);
            return this;
        }

        /** Removes all headers on this builder and adds {@code headers}. */
        public Builder headers(Headers headers) {
            this.headers = headers.newBuilder();
            return this;
        }

        public Builder body(ResponseBody body) {
            this.body = body;
            return this;
        }

        public Builder networkResponse(Response networkResponse) {
            if (networkResponse != null) checkSupportResponse("networkResponse", networkResponse);
            this.networkResponse = networkResponse;
            return this;
        }

        public Builder cacheResponse(Response cacheResponse) {
            if (cacheResponse != null) checkSupportResponse("cacheResponse", cacheResponse);
            this.cacheResponse = cacheResponse;
            return this;
        }

        private void checkSupportResponse(String name, Response response) {
            if (response.body != null) {
                throw new IllegalArgumentException(name + ".body != null");
            } else if (response.networkResponse != null) {
                throw new IllegalArgumentException(name + ".networkResponse != null");
            } else if (response.cacheResponse != null) {
                throw new IllegalArgumentException(name + ".cacheResponse != null");
            } else if (response.priorResponse != null) {
                throw new IllegalArgumentException(name + ".priorResponse != null");
            }
        }

        public Builder priorResponse(Response priorResponse) {
            if (priorResponse != null) checkPriorResponse(priorResponse);
            this.priorResponse = priorResponse;
            return this;
        }

        private void checkPriorResponse(Response response) {
            if (response.body != null) {
                throw new IllegalArgumentException("priorResponse.body != null");
            }
        }

        public Response build() {
            if (request == null) throw new IllegalStateException("request == null");
            if (code < 0) throw new IllegalStateException("code < 0: " + code);
            return new Response(this);
        }
    }
}
