package com.meizu.cloud.pushsdk.networking.http;

import com.meizu.cloud.pushsdk.networking.common.ANLog;
import com.meizu.cloud.pushsdk.networking.common.Method;
import com.meizu.cloud.pushsdk.networking.okio.BufferedSink;
import com.meizu.cloud.pushsdk.networking.okio.BufferedSource;
import com.meizu.cloud.pushsdk.networking.okio.Okio;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;



/**
 * Created by liaojinlong on 16-12-21.
 */
public class HttpURLConnectionCall implements Call{
    Request originalRequest;
    public HttpURLConnectionCall(Request originalRequest) {
        this.originalRequest = originalRequest;
    }

    @Override
    public Request request() {
        return originalRequest;
    }

    @Override
    public Response execute() throws IOException {
        HttpURLConnection connection = openConnection(originalRequest);
        for (String headerName : originalRequest.headers().names()) {
            String value = originalRequest.header(headerName);
            ANLog.i("current header name "+headerName+" value "+value);
            connection.addRequestProperty(headerName, value);
        }
        setConnectionParametersForRequest(connection, originalRequest);
        int responseCode = connection.getResponseCode();
        String responseMessage = connection.getResponseMessage();

        Response response = new Response.Builder()
                .code(responseCode)
                .headers(originalRequest.headers())
                .message(responseMessage)
                .request(originalRequest)
                .body(createOkBody(connection))
                .build();

        return response;
    }

    /**
     * Creates an OkHttp Response.Body containing the supplied information.
     */
    private static ResponseBody createOkBody(final HttpURLConnection urlConnection) throws IOException {
        if (!urlConnection.getDoInput()) {
            return null;
        }

        final BufferedSource body = Okio.buffer(Okio.source(isSuccessfulSend(urlConnection.getResponseCode())
                ? urlConnection.getInputStream() : urlConnection.getErrorStream()));
        return new ResponseBody() {
            @Override public MediaType contentType() {
                String contentTypeHeader = urlConnection.getContentType();
                return contentTypeHeader == null ? null : MediaType.parse(contentTypeHeader);
            }

            @Override public long contentLength() {
                String s = urlConnection.getHeaderField("Content-Length");
                return stringToLong(s);
            }

            @Override public BufferedSource source() {
                return body;
            }
        };
    }

    protected static boolean isSuccessfulSend(int code) {
        return code >= 200 && code < 300;
    }

    private static long stringToLong(String s) {
        if (s == null) return -1;
        try {
            return Long.parseLong(s);
        } catch (NumberFormatException e) {
            return -1;
        }
    }

    @Override
    public void cancel() {

    }

    @Override
    public boolean isExecuted() {
        return false;
    }

    @Override
    public boolean isCanceled() {
        return false;
    }


    private HttpURLConnection openConnection(Request request) throws IOException {
        String url = request.url().toString();
        URL parsedUrl = new URL(url);
        HttpURLConnection connection = createConnection(parsedUrl);
        connection.setConnectTimeout(60 * 1000);
        connection.setReadTimeout(60 * 1000);
        connection.setUseCaches(false);
        connection.setDoInput(true);
        // use caller-provided custom SslSocketFactory, if any, for HTTPS
        if(request.isHttps() && url.startsWith("https://push.statics")){
            //((HttpsURLConnection)connection).setSSLSocketFactory(SSLCertificateValidation.getSSLSocketFactory());
            //((HttpsURLConnection)connection).setHostnameVerifier(SSLCertificateValidation.getHostnameVerifier());
        }
        return connection;
    }

    /**
     * Create an {@link HttpURLConnection} for the specified {@code url}.
     */
    protected HttpURLConnection createConnection(URL url) throws IOException {
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        // Workaround for the M release HttpURLConnection not observing the
        // HttpURLConnection.setFollowRedirects() property.
        // https://code.google.com/p/android/issues/detail?id=194495
        connection.setInstanceFollowRedirects(HttpURLConnection.getFollowRedirects());
        return connection;
    }


    static void setConnectionParametersForRequest(HttpURLConnection connection,
                                                  Request request) throws IOException {
        switch (request.getmethod()) {
            case Method.GET:
                // Not necessary to set the request method because connection defaults to GET but
                // being explicit here.
                connection.setRequestMethod("GET");
                break;
            case Method.DELETE:
                connection.setRequestMethod("DELETE");
                break;
            case Method.POST:
                connection.setRequestMethod("POST");
                addBodyIfExists(connection, request);
                break;
            case Method.PUT:
                connection.setRequestMethod("PUT");
                addBodyIfExists(connection, request);
                break;
            case Method.HEAD:
                connection.setRequestMethod("HEAD");
                break;
            case Method.PATCH:
                connection.setRequestMethod("PATCH");
                addBodyIfExists(connection, request);
                break;
            default:
                throw new IllegalStateException("Unknown method type.");
        }
    }
    private static void addBodyIfExists(HttpURLConnection connection, Request request)
            throws IOException {
        RequestBody body = request.body();
        if (body != null) {
            connection.setDoOutput(true);
            connection.addRequestProperty("Content-Type", body.contentType().toString());
            BufferedSink fromSink = Okio.buffer(Okio.sink(connection.getOutputStream()));
            body.writeTo(fromSink);
            fromSink.close();
        }
    }

}
