package com.meizu.cloud.pushsdk.handler.impl;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.os.SystemClock;
import android.text.TextUtils;

import com.meizu.cloud.pushsdk.NotificationService;
import com.meizu.cloud.pushsdk.handler.AbstractAppLogicListener;
import com.meizu.cloud.pushsdk.constants.PushConstants;
import com.meizu.cloud.pushsdk.handler.MessageV3;
import com.meizu.cloud.pushsdk.notification.flyme.ExpandablePicNotification;
import com.meizu.cloud.pushsdk.notification.flyme.ExpandableTextNotification;
import com.meizu.cloud.pushsdk.notification.PictureNotification;
import com.meizu.cloud.pushsdk.notification.PushNotification;
import com.meizu.cloud.pushsdk.notification.PushNotificationBuilder;
import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.notification.flyme.StandardNotification;
import com.meizu.cloud.pushsdk.notification.StandardNotificationV2;
import com.meizu.cloud.pushsdk.notification.android.AndroidExpandablePicNotification;
import com.meizu.cloud.pushsdk.notification.android.AndroidExpandableTextNotification;
import com.meizu.cloud.pushsdk.notification.android.AndroidStandardNotification;
import com.meizu.cloud.pushsdk.notification.android.AndroidVideoNotification;
import com.meizu.cloud.pushsdk.notification.model.styleenum.BaseStyleModel;
import com.meizu.cloud.pushsdk.notification.model.styleenum.InnerStyleLayout;
import com.meizu.cloud.pushsdk.util.MzSystemUtils;
import com.meizu.cloud.pushsdk.util.UxIPUtils;

import java.text.SimpleDateFormat;
import java.util.Date;


/**
 * 处理MzPushServiceV3 发送的通知栏Json消息
 * 处理低版本云服务透传的通知栏Json消息
 */
public class MessageV3Handler extends AbstractMessageHandler<MessageV3> {

    public MessageV3Handler(Context context, AbstractAppLogicListener abstractAppLogicListener) {
        super(context, abstractAppLogicListener);
    }

    @Override
    protected MessageV3 getMessage(Intent intent) {
        String pushMessage;
        if(PushConstants.MZ_PUSH_MESSAGE_METHOD_ACTION_NOTIFICATION_SHOW_V3.equals(getIntentMethod(intent))){
            pushMessage = intent.getStringExtra(PushConstants.MZ_PUSH_PRIVATE_MESSAGE);
        } else {
            //think cloudService through message as notification message
            pushMessage = intent.getStringExtra(PushConstants.EXTRA_PUSH_MESSAGE);
        }
        return MessageV3.parse(context().getPackageName(),
                getPushServiceDefaultPackageName(intent),
                getPushTimestamp(intent),
                getDeviceId(intent),
                getTaskId(intent),
                getSeqId(intent),
                pushMessage);
    }

    @Override
    protected void unsafeSend(MessageV3 message, PushNotification pushNotification) {
        if(pushNotification != null){
            pushNotification.show(message);
            appLogicListener().onNotificationArrived(context(),message.getTitle(),message.getContent(),selfDefineContentString(message.getWebUrl(),message.getParamsMap()));
        }
    }

    @Override
    protected PushNotification onCreateNotification(MessageV3 messageV3) {
        PushNotificationBuilder pushNotificationBuilder = new PushNotificationBuilder();
        appLogicListener().onUpdateNotificationBuilder(pushNotificationBuilder);
        PushNotification pushNotification = null;
        if(messageV3.getmNotificationStyle() != null){
            int baseStyle = messageV3.getmNotificationStyle().getBaseStyle();
            if(BaseStyleModel.FLYME.getCode() == baseStyle){
                int innerStyle = messageV3.getmNotificationStyle().getInnerStyle();
                if(InnerStyleLayout.EXPANDABLE_STANDARD.getCode() == innerStyle){
                    DebugLogger.i(TAG,"show Standard Notification with Expandable disable");
                    pushNotification = new StandardNotification(context(),pushNotificationBuilder);
                } else if(InnerStyleLayout.EXPANDABLE_TEXT.getCode() == innerStyle){
                    DebugLogger.i(TAG,"show Standard Notification with Expandable Text");
                    pushNotification = new ExpandableTextNotification(context(),pushNotificationBuilder);
                } else if(InnerStyleLayout.EXPANDABLE_PIC.getCode() == innerStyle){
                    DebugLogger.i(TAG,"show Standard Notification with Expandable Picture");
                    pushNotification = new ExpandablePicNotification(context(),pushNotificationBuilder);
                } else if(InnerStyleLayout.EXPANDABLE_VIDEO.getCode() == innerStyle){
                    DebugLogger.i(TAG, "show Flyme Video notification");
                    pushNotification = new AndroidVideoNotification(context(),pushNotificationBuilder);
                }
            } else if(BaseStyleModel.PURE_PICTURE.getCode() == baseStyle){
                pushNotification = new PictureNotification(context(),pushNotificationBuilder);
                DebugLogger.i(TAG,"show Pure Picture Notification");
            } else if(BaseStyleModel.ANDROID.getCode() == baseStyle){
                int innerStyle = messageV3.getmNotificationStyle().getInnerStyle();
                if(InnerStyleLayout.EXPANDABLE_STANDARD.getCode() == innerStyle){
                    DebugLogger.i(TAG,"show Android  Notification with Expandable disable");
                    pushNotification = new AndroidStandardNotification(context(),pushNotificationBuilder);
                } else if(InnerStyleLayout.EXPANDABLE_TEXT.getCode() == innerStyle){
                    DebugLogger.i(TAG,"show Android  Notification with Expandable Text");
                    pushNotification = new AndroidExpandableTextNotification(context(),pushNotificationBuilder);
                } else if(InnerStyleLayout.EXPANDABLE_PIC.getCode() == innerStyle){
                    DebugLogger.i(TAG,"show Android  Notification with Expandable Picture");
                    pushNotification = new AndroidExpandablePicNotification(context(),pushNotificationBuilder);
                }else if(InnerStyleLayout.EXPANDABLE_VIDEO.getCode() == innerStyle){
                    DebugLogger.i(TAG, "show Flyme Video notification");
                    pushNotification = new AndroidVideoNotification(context(),pushNotificationBuilder);
                }
            }
        }
        if(pushNotification == null){
            DebugLogger.e(TAG,"use standard v2 notification");
            pushNotification = new StandardNotificationV2(context(),pushNotificationBuilder);
        }
        return pushNotification;
    }

    @Override
    protected boolean canSendMessage(MessageV3 message) {
        boolean flag = true;
        String pk = message.getUriPackageName();
        if(!TextUtils.isEmpty(pk)){
            flag = MzSystemUtils.isPackageInstalled(context(),pk);
        }
        return flag;
    }

    @Override
    protected int scheduleNotificationStatus(MessageV3 message) {
        int flag = SCHEDULE_OFF;
        if(message.getmTimeDisplaySetting() != null){
            if(!message.getmTimeDisplaySetting().isTimeDisplay()){
                flag = SCHEDULE_OFF;
            } else if(System.currentTimeMillis() > Long.valueOf(message.getmTimeDisplaySetting().getEndShowTime())){
                flag = SCHEDULE_ON_EXPIRE;
                UxIPUtils.notificationEvent(context(),"schedule notification expire",PushConstants.EXPIRE_NOTIFICATION,message.getTaskId(),message.getDeviceId());
            } else if(System.currentTimeMillis() > Long.valueOf(message.getmTimeDisplaySetting().getStartShowTime())){
                flag = SCHEDULE_ON_TIME;
                UxIPUtils.notificationEvent(context(),"schedule notification on time",PushConstants.ONTIME_NOTIFICATION,message.getTaskId(),message.getDeviceId());
            } else {
                flag = SCHEDULE_ON_DELAY;
                UxIPUtils.notificationEvent(context(),"schedule notification delay",PushConstants.DELAY_NOTIFICATION,message.getTaskId(),message.getDeviceId());
            }
        }
        return flag;
    }

    @Override
    protected void scheduleShowNotification(MessageV3 message) {
        AlarmManager alarmManager = (AlarmManager) context().getSystemService(context().ALARM_SERVICE);
        Intent intent = new Intent(context(), NotificationService.class);
        intent.setPackage(message.getPackageName());
        intent.addCategory(message.getPackageName());
        intent.setData(Uri.parse("custom://" + System.currentTimeMillis()));
        intent.putExtra("command_type", "reflect_receiver");
        intent.setAction(PushConstants.MZ_PUSH_ON_MESSAGE_ACTION);
        intent.putExtra(PushConstants.EXTRA_APP_PUSH_SCHEDULE_NOTIFICATION_MESSAGE, message);
        intent.putExtra(PushConstants.MZ_PUSH_MESSAGE_METHOD,PushConstants.MZ_PUSH_MESSAGE_METHOD_ACTION_SCHEDULE_NOTIFICATION);
        PendingIntent alarmIntent = PendingIntent.getService(context(), 0, intent, PendingIntent.FLAG_ONE_SHOT);
        String time = message.getmTimeDisplaySetting().getStartShowTime();

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        String showTime = null;
        if(!TextUtils.isEmpty(time)){
            Date showDate = new Date(Long.valueOf(time));
            showTime = sdf.format(showDate);
        }

        long afterTime = Long.valueOf(time)-System.currentTimeMillis();
        DebugLogger.i(TAG,"after "+(afterTime) /1000 +" seconds Notification AlarmManager execute At "+showTime);

        //https://developer.android.com/training/scheduling/alarms.html?hl=zh-cn
        if(Build.VERSION.SDK_INT >= 19){
            DebugLogger.i(TAG, "setAlarmManager setWindow ELAPSED_REALTIME_WAKEUP");
            alarmManager.setExact(AlarmManager.ELAPSED_REALTIME_WAKEUP, SystemClock.elapsedRealtime() + afterTime, alarmIntent);
            //alarmManager.setWindow(AlarmManager.ELAPSED_REALTIME_WAKEUP,SystemClock.elapsedRealtime()+afterTime, afterTime,alarmIntent);
        } else {
            alarmManager.set(AlarmManager.ELAPSED_REALTIME_WAKEUP, SystemClock.elapsedRealtime()+afterTime, alarmIntent);
        }
    }

    @Override
    public boolean messageMatch(Intent intent) {
        DebugLogger.i(TAG, "start MessageV3Handler match");
        if(!canReceiveMessage(0,getPushServiceDefaultPackageName(intent))){
            return false;
        }
        if(PushConstants.MZ_PUSH_ON_MESSAGE_ACTION.equals(intent.getAction())
                && PushConstants.MZ_PUSH_MESSAGE_METHOD_ACTION_NOTIFICATION_SHOW_V3.equals(getIntentMethod(intent))){
            return true;
        }
        if(TextUtils.isEmpty(getIntentMethod(intent))){
            String cloudPushMessage = intent.getStringExtra(PushConstants.EXTRA_PUSH_MESSAGE);
            //判断消息格式
            //1 通知栏消息格式
            if(!TextUtils.isEmpty(cloudPushMessage) && isNotificationJson(cloudPushMessage)){
                DebugLogger.e(TAG,"old cloud notification message");
               return true;
            }
        }
        return false;
    }

    @Override
    public int getProcessorType() {
        return MESSAGE_TYPE_PUSH_SERVICE_V3;
    }

    @Override
    protected void onBeforeEvent(MessageV3 message) {
        UxIPUtils.onReceivePushMessageEvent(context(), message.getUploadDataPackageName(), message.getDeviceId(),message.getTaskId(),message.getSeqId(),message.getPushTimestamp());
    }

    @Override
    protected void onAfterEvent(MessageV3 message) {
        UxIPUtils.onShowPushMessageEvent(context(), message.getUploadDataPackageName(), message.getDeviceId(), message.getTaskId(),message.getSeqId(),message.getPushTimestamp());
    }
}
