package com.meizu.cloud.pushsdk.handler;

import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import com.meizu.cloud.pushinternal.DebugLogger;
import com.meizu.cloud.pushsdk.constants.PushConstants;
import com.meizu.cloud.pushsdk.notification.MPushMessage;
import com.meizu.cloud.pushsdk.notification.model.AdvanceSetting;
import com.meizu.cloud.pushsdk.notification.model.AppIconSetting;
import com.meizu.cloud.pushsdk.notification.model.NotificationStyle;
import com.meizu.cloud.pushsdk.notification.model.TimeDisplaySetting;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * 以下定义push Message V3 版本消息体，兼容V2版
 */
public class MessageV3 implements Parcelable{
    private static final String TAG = "Message_V3";
    /**任务id*/
    private String taskId;
    /**x消息序列号**/
    private String seqId;
    /**设备唯一标识*/
    private String deviceId;
    /**标题*/
    private String title;
    /**内容*/
    private String content;
    /**通知栏应用包名*/
    private String packageName;
    /**
     * 点击通知栏的效果
     * 1.启动应用主界面,即应用的主界面--notify_launcher_activity
     * 2.启动应用的任意一个界面--notify_activity
     * 3.启动一个web页面--notify_web
     * */
    private int clickType;
    /**是否忽略该消息*/
    private boolean isDiscard;
    /**打开的activity名称*/
    private String activity;
    /**需要打开的uri地址*/
    private String webUrl;
    /**uri package*/
    private String uriPackageName;
    /**平台通知栏创建时间**/
    private String pushTimestamp;
    /**应用数据上报包名**/
    private String uploadDataPackageName;
    /**打开activity参数名称*/
    private Map<String,String> paramsMap = new HashMap();
    /**透传消息*/
    private String throughMessage;
    /**通知栏原始消息*/
    private String notificationMessage;
    /**高级设置*/
    private AdvanceSetting mAdvanceSetting;
    /**应用图标设置*/
    private AppIconSetting mAppIconSetting;
    /**Notification样式设置*/
    private NotificationStyle mNotificationStyle;
    /**定时显示设置*/
    private TimeDisplaySetting mTimeDisplaySetting;

    public MessageV3(Parcel in){
        taskId = in.readString();
        seqId = in.readString();
        deviceId = in.readString();
        title = in.readString();
        content = in.readString();
        packageName = in.readString();
        clickType = in.readInt();
        isDiscard = in.readByte() != 0;
        activity = in.readString();
        webUrl = in.readString();
        uriPackageName = in.readString();
        uploadDataPackageName = in.readString();
        pushTimestamp = in.readString();
        paramsMap = in.readHashMap(this.getClass().getClassLoader());
        throughMessage = in.readString();
        notificationMessage = in.readString();
        mAdvanceSetting = in.readParcelable(AdvanceSetting.class.getClassLoader());
        mAppIconSetting = in.readParcelable(AppIconSetting.class.getClassLoader());
        mNotificationStyle = in.readParcelable(NotificationStyle.class.getClassLoader());
        mTimeDisplaySetting = in.readParcelable(TimeDisplaySetting.class.getClassLoader());
    }

    public static final Parcelable.Creator<MessageV3> CREATOR
            = new Parcelable.Creator<MessageV3>() {
        public MessageV3 createFromParcel(Parcel in) {
            return new MessageV3(in);
        }

        public MessageV3[] newArray(int size) {
            return new MessageV3[size];
        }
    };

    public MessageV3() {
        super();
    }


    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel parcel, int flag) {
        parcel.writeString(taskId);
        parcel.writeString(seqId);
        parcel.writeString(deviceId);
        parcel.writeString(title);
        parcel.writeString(content);
        parcel.writeString(packageName);
        parcel.writeInt(clickType);
        parcel.writeByte((byte) (isDiscard ? 1 : 0));
        parcel.writeString(activity);
        parcel.writeString(webUrl);
        parcel.writeString(uriPackageName);
        parcel.writeString(uploadDataPackageName);
        parcel.writeString(pushTimestamp);
        parcel.writeMap(paramsMap);
        parcel.writeString(throughMessage);
        parcel.writeString(notificationMessage);
        parcel.writeParcelable(mAdvanceSetting, flag);
        parcel.writeParcelable(mAppIconSetting, flag);
        parcel.writeParcelable(mNotificationStyle, flag);
        parcel.writeParcelable(mTimeDisplaySetting,flag);
    }

    public enum CLICK_TYPE_DEFINE {
        CLICK_TYPE_LAUNCHER_ACTIVITY,
        CLICK_TYPE_ACTIVITY,
        CLICK_TYPE_WEB
    }

    public String getDeviceId() {
        return deviceId;
    }

    public void setDeviceId(String deviceId) {
        this.deviceId = deviceId;
    }

    public String getTaskId() {
        return taskId;
    }

    public void setTaskId(String taskId) {
        this.taskId = taskId;
    }


    public AdvanceSetting getmAdvanceSetting() {
        return mAdvanceSetting;
    }

    public void setmAdvanceSetting(AdvanceSetting mAdvanceSetting) {
        this.mAdvanceSetting = mAdvanceSetting;
    }

    public AppIconSetting getmAppIconSetting() {
        return mAppIconSetting;
    }

    public void setmAppIconSetting(AppIconSetting mAppIconSetting) {
        this.mAppIconSetting = mAppIconSetting;
    }

    public NotificationStyle getmNotificationStyle() {
        return mNotificationStyle;
    }

    public void setmNotificationStyle(NotificationStyle mNotificationStyle) {
        this.mNotificationStyle = mNotificationStyle;
    }

    public TimeDisplaySetting getmTimeDisplaySetting() {
        return mTimeDisplaySetting;
    }

    public void setmTimeDisplaySetting(TimeDisplaySetting mTimeDisplaySetting) {
        this.mTimeDisplaySetting = mTimeDisplaySetting;
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public String getContent() {
        return content;
    }

    public void setContent(String content) {
        this.content = content;
    }

    public String getPackageName() {
        return packageName;
    }

    public void setPackageName(String packageName) {
        this.packageName = packageName;
    }

    public int getClickType() {
        return clickType;
    }

    public void setClickType(int clickType) {
        this.clickType = clickType;
    }

    public boolean isDiscard() {
        return isDiscard;
    }

    public void setIsDiscard(boolean isDiscard) {
        this.isDiscard = isDiscard;
    }

    public String getActivity() {
        return activity;
    }

    public void setActivity(String activity) {
        this.activity = activity;
    }

    public String getWebUrl() {
        return webUrl;
    }

    public void setWebUrl(String webUrl) {
        this.webUrl = webUrl;
    }

    public String getUriPackageName() {
        return uriPackageName;
    }

    public void setUriPackageName(String uriPackageName) {
        this.uriPackageName = uriPackageName;
    }

    public String getPushTimestamp() {
        return pushTimestamp;
    }

    public void setPushTimestamp(String pushTimestamp) {
        this.pushTimestamp = pushTimestamp;
    }

    public Map<String, String> getParamsMap() {
        return paramsMap;
    }

    public void setParamsMap(Map<String, String> paramsMap) {
        this.paramsMap = paramsMap;
    }

    public String getThroughMessage() {
        return throughMessage;
    }

    public void setThroughMessage(String throughMessage) {
        this.throughMessage = throughMessage;
    }


    public String getNotificationMessage() {
        return notificationMessage;
    }

    public void setNotificationMessage(String notificationMessage) {
        this.notificationMessage = notificationMessage;
    }

    public String getSeqId() {
        return seqId;
    }

    public void setSeqId(String seqId) {
        this.seqId = seqId;
    }

    public String getUploadDataPackageName() {
        return uploadDataPackageName;
    }

    public void setUploadDataPackageName(String uploadDataPackageName) {
        this.uploadDataPackageName = uploadDataPackageName;
    }

    /**
     * parse MpushMessage V2 to MessageV3
     * @param packageName
     * @param deviceId
     * @param taskId
     * @param pushMessage
     * */
    public static MessageV3 parse(String packageName,String deviceId,String taskId, MPushMessage pushMessage){
        DebugLogger.e(TAG,"V2 message "+pushMessage);
        MessageV3 messageV3 = new MessageV3();
        messageV3.setPackageName(packageName);
        messageV3.setUploadDataPackageName(packageName);
        messageV3.setDeviceId(deviceId);
        messageV3.setTaskId(taskId);
        messageV3.setTitle(pushMessage.getTitle());
        messageV3.setContent(pushMessage.getContent());
        messageV3.setIsDiscard("true".equals(pushMessage.getIsDiscard()) ? true : false);
        messageV3.setClickType(Integer.valueOf(pushMessage.getClickType()));
        for (Map.Entry<String,String> extra : pushMessage.getExtra().entrySet()){
            String key = extra.getKey();
            String value = extra.getValue();
            if(PushConstants.INTENT_ACTIVITY_NAME.equals(key)){
                messageV3.setActivity(value);
            }
            if(PushConstants.WEB_URL.equals(key)){
                messageV3.setWebUrl(value);
            }
            if(PushConstants.URI_PACKAGE_NAME.equals(key)){
                messageV3.setUriPackageName(value);
            }
            if(NotificationStyle.NOTIFICATION_STYLE.equals(key)){
                messageV3.setmNotificationStyle(NotificationStyle.parse(value));
            }
            if(AdvanceSetting.ADVANCE_SETTING.equals(key)){
                messageV3.setmAdvanceSetting(AdvanceSetting.parse(value));
            }
            if(AppIconSetting.APP_ICON_SETTING.equals(key)){
                messageV3.setmAppIconSetting(AppIconSetting.parse(value));
            }
            if(TimeDisplaySetting.TIME_DISPLAY_SETTING.equals(key)){
                messageV3.setmTimeDisplaySetting(TimeDisplaySetting.parse(value));
            }
        }
        messageV3.setParamsMap(pushMessage.getParams());
        DebugLogger.i(TAG, "parase V2 message to V3 message " + messageV3);
        return messageV3;
    }

    /**
     * add uploadDataPackageName
     * @param uploadDataPackageName
     *                 数据上报包名
     * @param packageName
     *                 当前应用包名
     * @param timestamp
     *                推送消息时间戳
     * @param deviceId
     *                设备唯一id
     * @param seqId
     *                消息序列号
     * @param taskId
     *                消息任务id
     * @param pushMessage
     *                原始通知栏消息
     * */
    public static MessageV3 parse(String packageName,String uploadDataPackageName,String timestamp,String deviceId,String taskId, String seqId,String pushMessage){
        MessageV3 messageV3 = parse(packageName,deviceId,taskId, seqId,pushMessage);
        messageV3.setUploadDataPackageName(uploadDataPackageName);
        messageV3.setPushTimestamp(timestamp);
        return messageV3;
    }

    /**
     * add seqId
     * @param seqId
     * */
    public static MessageV3 parse(String packageName,String deviceId,String taskId, String seqId,String pushMessage){
        MessageV3 messageV3 = parse(packageName,deviceId,taskId,pushMessage);
        messageV3.setSeqId(seqId);
        return messageV3;
    }

    public static MessageV3 parse(String packageName,String deviceId,String taskId, String pushMessage){
        MessageV3 messageV3 = new MessageV3();
        messageV3.setNotificationMessage(pushMessage);
        messageV3.setTaskId(taskId);
        messageV3.setDeviceId(deviceId);
        messageV3.setPackageName(packageName);
        try {
            JSONObject pushDataObj = new JSONObject(pushMessage);
            JSONObject pushMessageObj = pushDataObj.getJSONObject("data");
            if(!pushMessageObj.isNull(PushConstants.TITLE)){
                messageV3.setTitle(pushMessageObj.getString(PushConstants.TITLE));
            }
            if(!pushMessageObj.isNull(PushConstants.CONTENT)){
               messageV3.setContent(pushMessageObj.getString(PushConstants.CONTENT));
            }
            if(!pushMessageObj.isNull(PushConstants.IS_DISCARD)){
                messageV3.setIsDiscard(pushMessageObj.getBoolean(PushConstants.IS_DISCARD));
            }
            if(!pushMessageObj.isNull(PushConstants.CLICK_TYPE)){
                messageV3.setClickType(pushMessageObj.getInt(PushConstants.CLICK_TYPE));
            }
            if(!pushMessageObj.isNull(PushConstants.EXTRA)){
                JSONObject extraObj = pushMessageObj.getJSONObject(PushConstants.EXTRA);
                if(!extraObj.isNull(NotificationStyle.NOTIFICATION_STYLE)){
                    messageV3.setmNotificationStyle(NotificationStyle.parse(extraObj.getJSONObject(NotificationStyle.NOTIFICATION_STYLE)));
                }
                if(!extraObj.isNull(AppIconSetting.APP_ICON_SETTING)){
                    messageV3.setmAppIconSetting(AppIconSetting.parse(extraObj.getJSONObject(AppIconSetting.APP_ICON_SETTING)));
                }
                if(!extraObj.isNull(AdvanceSetting.ADVANCE_SETTING)){
                    messageV3.setmAdvanceSetting(AdvanceSetting.parse(extraObj.getJSONObject(AdvanceSetting.ADVANCE_SETTING)));
                }
                if(!extraObj.isNull(TimeDisplaySetting.TIME_DISPLAY_SETTING)){
                    messageV3.setmTimeDisplaySetting(TimeDisplaySetting.parse(extraObj.getJSONObject(TimeDisplaySetting.TIME_DISPLAY_SETTING)));
                }
                if(!extraObj.isNull(PushConstants.INTENT_ACTIVITY_NAME)){
                    messageV3.setActivity(extraObj.getString(PushConstants.INTENT_ACTIVITY_NAME));
                }
                if(!extraObj.isNull(PushConstants.WEB_URL)){
                    messageV3.setWebUrl(extraObj.getString(PushConstants.WEB_URL));
                }
                if(!extraObj.isNull(PushConstants.TASK_ID) && TextUtils.isEmpty(taskId)){
                    DebugLogger.e(TAG,"Flyme 4 notification message by through message or taskId is null");
                    messageV3.setTaskId(extraObj.getString(PushConstants.TASK_ID));
                }
                if(!extraObj.isNull(PushConstants.URI_PACKAGE_NAME)){
                    messageV3.setUriPackageName(extraObj.getString(PushConstants.URI_PACKAGE_NAME));
                }
                if(!extraObj.isNull(PushConstants.PARAMS)){
                    messageV3.setParamsMap(getParamsMap(extraObj.getJSONObject(PushConstants.PARAMS)));
                }
            }
        } catch (JSONException e) {
            DebugLogger.e(TAG, "parse message error " + e.getMessage());
            //记录日志上报错误
        }
        return messageV3;
    }

    public static Map<String,String> getParamsMap(JSONObject paramsJsonObject){
        Map<String,String> paramsMap = new HashMap<>();
        try {
            Iterator<String> extraIterator = paramsJsonObject.keys();
            while (extraIterator.hasNext()){
                String key = extraIterator.next();
                String value = paramsJsonObject.getString(key);
                paramsMap.put(key, value);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return paramsMap;
    }

    @Override
    public String toString() {
        return "MessageV3{" +
                "taskId='" + taskId + '\'' +
                ", seqId='" + seqId + '\'' +
                ", deviceId='" + deviceId + '\'' +
                ", title='" + title + '\'' +
                ", content='" + content + '\'' +
                ", packageName='" + packageName + '\'' +
                ", clickType=" + clickType +
                ", isDiscard=" + isDiscard +
                ", activity='" + activity + '\'' +
                ", webUrl='" + webUrl + '\'' +
                ", uriPackageName='" + uriPackageName + '\'' +
                ", pushTimestamp='" + pushTimestamp + '\'' +
                ", uploadDataPackageName='" + uploadDataPackageName + '\'' +
                ", paramsMap=" + paramsMap +
                ", throughMessage='" + throughMessage + '\'' +
                ", notificationMessage='" + notificationMessage + '\'' +
                ", mAdvanceSetting=" + mAdvanceSetting +
                ", mAppIconSetting=" + mAppIconSetting +
                ", mNotificationStyle=" + mNotificationStyle +
                ", mTimeDisplaySetting=" + mTimeDisplaySetting +
                '}';
    }
}
