package com.meizu.cloud.pushsdk.base.reflect;


import com.meizu.cloud.pushsdk.base.Logger;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;

/**
 * Created by zbin on 17-2-16.
 */

public class ReflectMethod {
    private String TAG = "ReflectMethod";
    private static HashMap<String, Method> mCachedMethods = new HashMap<>();
    private ReflectClass mReflectClass;
    private String mMethodName;
    private Class<?>[] mTypes;

    ReflectMethod(ReflectClass reflectClass, String name, Class<?>... types) {
        mReflectClass = reflectClass;
        mMethodName = name;
        mTypes = types;
    }

    private boolean match(Class<?>[] declaredTypes, Class<?>[] actualTypes) {
        if (declaredTypes.length == actualTypes.length) {
            for (int i = 0; i < actualTypes.length; i++) {
                if (actualTypes[i] == NULL.class)
                    continue;
                if (wrapper(declaredTypes[i]).isAssignableFrom(wrapper(actualTypes[i])))
                    continue;
                return false;
            }
            return true;
        } else {
            return false;
        }
    }

    private boolean isSimilarSignature(Method possiblyMatchingMethod,
                                              String desiredMethodName, Class<?>[] desiredParamTypes) {
        return possiblyMatchingMethod.getName().equals(desiredMethodName) &&
                match(possiblyMatchingMethod.getParameterTypes(), desiredParamTypes);
    }

    private Method similarMethod()
            throws NoSuchMethodException, ClassNotFoundException {
        Class<?> clz = mReflectClass.getRealClass();
        for (Method method : clz.getMethods()) {
            if (isSimilarSignature(method, mMethodName, mTypes)) {
                return method;
            }
        }
        for (Method method : clz.getDeclaredMethods()) {
            if (isSimilarSignature(method, mMethodName, mTypes)) {
                return method;
            }
        }
        throw new NoSuchMethodException("No similar method " + mMethodName +
                " with params " + Arrays.toString(mTypes) + " could be found on type " + clz);
    }

    private String getKey() throws ClassNotFoundException {
        StringBuffer buffer = new StringBuffer(mReflectClass.getRealClass().getName());
        buffer.append(mMethodName);
        for(Class<?> type : mTypes) {
            buffer.append(type.getName());
        }
        return buffer.toString();
    }

    /**
     * 用于静态或非静态方法的调用
     * @param receiver 方法的接收者
     * @param args 调用参数
     * @param <T> 调用结果的参数类型
     * @return 反射调用的结果
     */
    public <T> ReflectResult<T> invoke(Object receiver, Object... args) {
        ReflectResult<T> result = new ReflectResult<>();
        try {
            String key = getKey();
            Method method = mCachedMethods.get(key);
            if(method == null) {
                if(mTypes.length == args.length) {
                    method = mReflectClass.getRealClass().getMethod(mMethodName, mTypes);
                } else {
                    if(args.length > 0) {
                        mTypes = new Class<?>[args.length];
                        for(int i = 0; i < args.length; ++i) {
                            mTypes[i] = args[i].getClass();
                        }
                    }
                    method = similarMethod();
                }
                mCachedMethods.put(key, method);
            }
            method.setAccessible(true);
            result.value = (T) method.invoke(receiver, args);
            result.ok = true;
        } catch (Exception e) {
            Logger.get().e(TAG, "invoke", e);
        }
        return result;
    }

    /**
     * 用于静态方法的反射调用
     * @param args 调用参数
     * @param <T> 调用结果的参数类型
     * @return 反射调用的结果
     */
    public <T> ReflectResult<T> invokeStatic(Object... args) {
        try {
            return invoke(mReflectClass.getRealClass(), args);
        } catch (ClassNotFoundException e) {
            return new ReflectResult<>();
        }
    }

    private Class<?> wrapper(Class<?> type) {
        if (type == null) {
            return null;
        } else if (type.isPrimitive()) {
            if (boolean.class == type) {
                return Boolean.class;
            } else if (int.class == type) {
                return Integer.class;
            } else if (long.class == type) {
                return Long.class;
            } else if (short.class == type) {
                return Short.class;
            } else if (byte.class == type) {
                return Byte.class;
            } else if (double.class == type) {
                return Double.class;
            } else if (float.class == type) {
                return Float.class;
            } else if (char.class == type) {
                return Character.class;
            } else if (void.class == type) {
                return Void.class;
            }
        }
        return type;
    }

    private class NULL {
    }
}
