package com.meizu.cloud.pushsdk.base;

import android.os.Environment;
import android.os.Handler;
import android.os.Looper;
import android.os.Process;
import android.util.Log;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;

/**
 * Created by zbin on 17-2-16.
 */

class DefaultLog implements ICacheLog {
    private SimpleDateFormat mDateFormat;
    private List<LogInfo> mCachedList;
    private Handler mDelayHandler;
    private long mCacheDuration = 60;
    private int mCacheCounter = 10;
    private EncryptionWriter mWriter;
    private String mPath;
    private String mPid;
    private boolean mDebugMode = false;

    public DefaultLog() {
        mDateFormat = new SimpleDateFormat("MM-dd HH:mm:ss");
        mCachedList = Collections.synchronizedList(new ArrayList<LogInfo>());
        mDelayHandler = new Handler(Looper.getMainLooper());
        mPath = Environment.getExternalStorageDirectory().getAbsolutePath()+"/Android/data/pushSdk/defaultLog";
        mWriter = new EncryptionWriter();
        mPid = String.valueOf(Process.myPid());
    }

    private void startDelayTimer() {
        //不使用同步锁，避免锁的开销，即使遇到并发竞争问题也最多多跑一次flush
        if(mCachedList.size() == 0) {
            mDelayHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    flush(true);
                }
            }, mCacheDuration * 1000);
        }
    }

    private void checkLogCount() {
        //不使用同步锁，避免锁的开销，即使遇到并发竞争问题也最多多跑一次flush
        if(mCachedList.size() == mCacheCounter) {
            flush(true);
        }
    }

    @Override
    public void d(String tag, String msg) {
        if(mDebugMode) {
            Log.d(tag, msg);
        }
        synchronized (mCachedList) {
            startDelayTimer();
            addLogInfo(new LogInfo("D", tag, msg));
            checkLogCount();
        }
    }

    @Override
    public void i(String tag, String msg) {
        if(mDebugMode) {
            Log.i(tag, msg);
        }
        synchronized (mCachedList) {
            startDelayTimer();
            addLogInfo(new LogInfo("I", tag, msg));
            checkLogCount();
        }
    }

    @Override
    public void w(String tag, String msg) {
        if(mDebugMode) {
            Log.w(tag, msg);
        }
        synchronized (mCachedList) {
            startDelayTimer();
            addLogInfo(new LogInfo("W", tag, msg));
            checkLogCount();
        }
    }

    @Override
    public void e(String tag, String msg) {
        if(mDebugMode) {
            Log.e(tag, msg);
        }
        synchronized (mCachedList) {
            startDelayTimer();
            addLogInfo(new LogInfo("E", tag, msg));
            checkLogCount();
        }
    }

    @Override
    public void e(String tag, String msg, Throwable tr) {
        if(mDebugMode) {
            Log.e(tag, msg, tr);
        }
        synchronized (mCachedList) {
            startDelayTimer();
            addLogInfo(new LogInfo("E", tag, msg + "\n" + Log.getStackTraceString(tr)));
            checkLogCount();
        }
    }

    @Override
    public void setCacheDuration(long seconds) {
        mCacheDuration = seconds;
    }

    @Override
    public void setCacheCount(int counter) {
        mCacheCounter = counter;
    }

    @Override
    public void setFilePath(String path) {
        mPath = path;
    }

    @Override
    public void flush(boolean async) {
        Runnable task = new Runnable() {
            @Override
            public void run() {
                List<LogInfo> tmp = new ArrayList<>();
                synchronized (mCachedList) {
                    mDelayHandler.removeCallbacksAndMessages(null);
                    tmp.addAll(mCachedList);
                    mCachedList.clear();
                }
                try {
                    mWriter.open(mPath);
                    for(LogInfo logInfo : tmp) {
                        mWriter.write(logInfo.header, logInfo.tag, logInfo.msg);
                    }
                } catch (Exception e) {
                    //ignore
                } finally {
                    try {
                        mWriter.close();
                    } catch (Exception e) {
                        //ignore
                    }
                }
            }
        };
        if(async) {
            ExecutorProxy.get().execute(task);
        } else {
            task.run();
        }
    }

    @Override
    public void setDebugMode(boolean debug) {
        mDebugMode = debug;
    }

    @Override
    public boolean isDebugMode() {
        return mDebugMode;
    }

    class LogInfo {
        String header;
        String tag, msg;

        public LogInfo(String level, String tag, String msg) {
            StringBuffer buffer = new StringBuffer(mDateFormat.format(new Date()));
            buffer.append(" ");
            buffer.append(mPid);
            buffer.append("-");
            buffer.append(String.valueOf(Thread.currentThread().getId()));
            buffer.append(" ");
            buffer.append(level);
            buffer.append("/");
            header = buffer.toString();
            this.tag = tag;
            this.msg = msg;
        }
    }

    /**
     * add logInfo
     * @param logInfo
     * */
    private void addLogInfo(LogInfo logInfo){
        try {
            mCachedList.add(logInfo);
        } catch (Exception e){
            Log.e("Logger","add logInfo error "+ e.getMessage());
        }
    }
}
