package com.medy.retrofitwrapper;

import android.app.ProgressDialog;
import android.content.Context;
import android.net.Uri;
import android.util.Base64;
import android.util.Log;
import android.webkit.MimeTypeMap;
import android.widget.Toast;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.RequestBody;
import okhttp3.ResponseBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;

/**
 * @author Manish Kumar
 * @since 28/8/17
 */


public class WebRequest {

    public boolean isDebug = false;

    public static final String LINE_SEPARATOR = "\n";

    /**
     * Request type post
     */
    public static final String POST_REQ = "POST";
    /**
     * Request type get
     */
    public static final String GET_REQ = "GET";
    /**
     * Request type delete
     */
    public static final String DELETE_REQ = "DELETE";

    public static final int DEFAULT_TIME_OUT = (int) (0.12 * 60 * 1000);


    final int timeout;
    final HashMap<String, String> webRequestParams = new LinkedHashMap<>();
    final HashMap<String, String> webRequestHeaders = new LinkedHashMap<>();
    final HashMap<String, String> webRequestQuery = new LinkedHashMap<>();
    final HashMap<String, File> webRequestFiles = new LinkedHashMap<>();
    final HashMap<String, Object> extraDataMap = new LinkedHashMap<>();
    final int webRequestId;
    final String webRequestUrl;
    final String webRequestMethod;
    final TrustManager[] trustAllCerts = new TrustManager[]{
            new X509TrustManager() {
                @Override
                public void checkClientTrusted (X509Certificate[] chain, String authType) throws CertificateException {
                }

                @Override
                public void checkServerTrusted (X509Certificate[] chain, String authType) throws CertificateException {
                }

                @Override
                public X509Certificate[] getAcceptedIssuers () {
                    return new X509Certificate[]{};
                }
            }
    };
    Context context;
    String username;
    String password;
    String webRequestResponse;
    int responseCode = -1;
    Exception webRequestException;
    WebRequestErrorDialog webRequestErrorDialog;
    ProgressDialog progressDialog;
    String progressMessage = "";
    private ConnectionDetector cd;

    public WebRequest (int webRequestId, String webRequestUrl) {
        this(webRequestId, webRequestUrl, GET_REQ);
    }

    public WebRequest (int webRequestId, String webRequestUrl,
                       String webRequestMethod) {
        this(webRequestId, webRequestUrl, webRequestMethod, DEFAULT_TIME_OUT);

    }


    public WebRequest (int webRequestId, String webRequestUrl,
                       String webRequestMethod, int timeout) {
        this.webRequestId = webRequestId;
        this.webRequestMethod = webRequestMethod;
        this.timeout = timeout;

        Uri uri = Uri.parse(webRequestUrl);
        parseQueryParams(uri);
        this.webRequestUrl = uri.getScheme() + "://" + uri.getHost() + uri.getPath();


    }


    private void parseQueryParams (Uri uri) {
        webRequestQuery.clear();
        if (uri.isOpaque()) {
            return;
        }

        String query = uri.getEncodedQuery();
        if (query == null) {
            return;
        }


        int start = 0;
        do {
            int next = query.indexOf('&', start);
            int end = (next == -1) ? query.length() : next;

            int separator = query.indexOf('=', start);
            if (separator > end || separator == -1) {
                separator = end;
            }

            String name = query.substring(start, separator);
            String value;
            if (separator < end)
                value = query.substring(separator + 1, end);
            else
                value = "";

            webRequestQuery.put(Uri.decode(name), Uri.decode(value));

            // Move start to end of name.
            start = end + 1;
        } while (start < query.length());
    }


    public void setRequestModel (WebServiceBaseRequestModel webServiceBaseRequestModel) {
        webRequestHeaders.clear();
        webRequestParams.clear();
        webRequestFiles.clear();
        if (webServiceBaseRequestModel != null) {
            for (Field f : webServiceBaseRequestModel.getClass().getFields()) {
                f.setAccessible(true);
                try {
                    String key = f.getName();
                    Object value = f.get(webServiceBaseRequestModel);
                    boolean isHeader = f.isAnnotationPresent(WebServiceHeader.class);
                    if (value != null) {
                        if (isHeader) {
                            addHeader(key, ((String) value));
                            continue;
                        }
                        if (f.getType() == List.class) {
                            List dataList = (List) value;
                            int i = 0;
                            for (Object data : dataList) {
                                String newKey = key + "[" + i + "]";
                                if (data instanceof File) {
                                    addFile(newKey, (File) data);
                                } else {
                                    addParam(newKey, String.valueOf(data));
                                }
                            }
                        } else if (f.getType() == Object.class) {
                            if (value instanceof File) {
                                addFile(key, (File) value);
                            } else {
                                addParam(key, String.valueOf(value));
                            }
                        } else if (f.getType() == File.class) {
                            addFile(key, (File) value);
                        } else {
                            addParam(key, String.valueOf(value));
                        }
                    }
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                }
            }
        }

    }

    private void printLog (String msg) {
        if (isDebug) {
            Log.e(getClass().getSimpleName() + webRequestId, msg);
        }
    }

    private boolean isDebugBuild (Context context) {

        try {
            final String packageName = context.getPackageName();
            final Class<?> buildConfig = Class.forName(packageName + ".BuildConfig");
            final Field DEBUG = buildConfig.getField("DEBUG");
            DEBUG.setAccessible(true);
            return DEBUG.getBoolean(null);
        } catch (final Throwable t) {
            final String message = t.getMessage();
            if (message != null && message.contains("BuildConfig")) {
                // Proguard obfuscated build. Most likely a production build.
                return false;
            } else {
                return BuildConfig.DEBUG;
            }
        }
    }


    public int getWebRequestId () {
        return webRequestId;
    }

    private void onRequestCompleted (Exception e, Response<ResponseBody> response) {
        this.webRequestException = e;
        if (response != null) {
            this.responseCode = response.code();
            try {
                if (response.isSuccessful() && response.body() != null) {
                    webRequestResponse = response.body().string();
                } else if (response.errorBody() != null) {
                    webRequestResponse = response.errorBody().string();
                }
            } catch (IOException e1) {
                e1.printStackTrace();
            }
        }
    }

    public void setBasicAuth (String username, String password) {
        this.username = username;
        this.password = password;
    }

    public void addParam (String key, String value) {
        webRequestParams.put(key, value);
    }

    public void addHeader (String key, String value) {
        webRequestHeaders.put(key, value);
    }

    public void addQuery (String key, String value) {
        webRequestQuery.put(key, value);
    }

    public void clearQuery () {
        webRequestQuery.clear();
    }

    public void addFile (String key, File file) {
        webRequestFiles.put(key, file);
    }

    public void addExtra (String key, Object value) {
        extraDataMap.put(key, value);
    }

    public Object getExtraData (String key) {
        return extraDataMap.get(key);
    }

    public static MultipartBody.Part createFilePart (String variableName, String filePath, MediaType mediaType) {
        File file = new File(filePath);
        RequestBody requestFile =
                RequestBody.create(mediaType, file);
        return MultipartBody.Part.createFormData(variableName, file.getName(), requestFile);
    }

    private Certificate getCertificate () {
        InputStream cert = null;
        Certificate certificate = null;
        try {
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            cert = context.getResources().openRawResource(R.raw.ubuntumemberslinodecom);
            certificate = cf.generateCertificate(cert);
        } catch (CertificateException e) {
            e.printStackTrace();
        } finally {
            try {
                if (cert != null)
                    cert.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return certificate;
    }

    private Call<ResponseBody> generateRequest (Context context) {
        String basicAuth = getBasicAuth();
        if (basicAuth != null) {
            addHeader("Authorization", basicAuth);
        }


        Call<ResponseBody> call = null;
        WebRequestInterface webRequestInterface = getWebServiceInterface(getCertificate());
        if (this.webRequestMethod.equals(DELETE_REQ)) {
            call = webRequestInterface.callDeleteRequest(this.webRequestUrl,
                    webRequestQuery, webRequestHeaders);
        } else if (this.webRequestMethod.equals(GET_REQ)) {

            call = webRequestInterface.callGetRequest(this.webRequestUrl,
                    webRequestQuery, webRequestHeaders);

        } else if (this.webRequestMethod.equals(POST_REQ)) {

            MultipartBody.Builder builder = new MultipartBody.Builder();
            if (!webRequestParams.isEmpty()) {
                for (String key : webRequestParams.keySet()) {
                    if (key == null) continue;
                    String value = webRequestParams.get(key);
                    if (value == null) continue;
                    builder.addFormDataPart(key, value);
                }
            }
            if (!webRequestFiles.isEmpty()) {
                for (String key : webRequestFiles.keySet()) {
                    if (key == null) continue;
                    File value = webRequestFiles.get(key);
                    if (value == null) continue;
                    String mimeType = getMimeType(value.getAbsolutePath());
                    if (mimeType == null) continue;
                    RequestBody requestBody = RequestBody.create(MediaType.parse(mimeType), value);
                    builder.addFormDataPart(key, value.getName(), requestBody);
                }
            }
            MultipartBody body = builder.build();
            addHeader("Content-Type", "multipart/form-data; boundary=" + body.boundary());
            call = webRequestInterface.callPostRequest(this.webRequestUrl, webRequestQuery,
                    webRequestHeaders,
                    body);


        }
        return call;
    }


    private String getBasicAuth () {
        if (this.username == null || this.password == null) return null;
        String basicAuth = "Basic " +
                Base64.encodeToString(String.format("%s:%s", username, password)
                                .getBytes(),
                        Base64.NO_WRAP);
        return basicAuth;
    }

    private String getMimeType (String filePath) {
        String type = null;
        String extension = MimeTypeMap.getFileExtensionFromUrl(filePath);
        if (extension != null && !extension.trim().isEmpty()) {
            type = MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension);
        }
        return type;
    }

    private String getWebRequestFullUrl () {
        StringBuilder builder = new StringBuilder(this.webRequestUrl);
        if (!webRequestQuery.isEmpty()) {
            builder.append("?");
            Set<Map.Entry<String, String>> entrySet = webRequestQuery.entrySet();
            for (Map.Entry<String, String> entry : entrySet) {
                builder.append(entry.getKey()).append("=").append(entry.getValue());
            }
        }
        return builder.toString();
    }

    public void printRequestLog () {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("webRequestId = ").append(webRequestId).append(LINE_SEPARATOR);
        stringBuilder.append("webRequestUrl = ").append(getWebRequestFullUrl()).append(LINE_SEPARATOR);

        stringBuilder.append("webRequestMethod = ").append(webRequestMethod).append(LINE_SEPARATOR);
        if (this.username != null && this.password != null) {
            stringBuilder.append("username = ").append(username).append(LINE_SEPARATOR);
            stringBuilder.append("password = ").append(password).append(LINE_SEPARATOR);
        }
        if (!webRequestHeaders.isEmpty()) {
            stringBuilder.append("Headers=>").append(LINE_SEPARATOR);
            for (Map.Entry<String, String> data : webRequestHeaders.entrySet()) {
                stringBuilder.append(data.getKey() + " = " + data.getValue()).append(LINE_SEPARATOR);
            }
        }
        if (!webRequestParams.isEmpty()) {
            stringBuilder.append("Parameters=>").append(LINE_SEPARATOR);
            for (Map.Entry<String, String> data : webRequestParams.entrySet()) {
                stringBuilder.append(data.getKey() + " = " + data.getValue()).append(LINE_SEPARATOR);
            }
        }
        if (!webRequestFiles.isEmpty()) {
            stringBuilder.append("FileParameters=>").append(LINE_SEPARATOR);
            for (Map.Entry<String, File> data : webRequestFiles.entrySet()) {
                stringBuilder.append(data.getKey() + " = " + data.getValue().getAbsolutePath())
                        .append(", MIME = ").append(getMimeType(data.getValue().getAbsolutePath())).append(LINE_SEPARATOR);
            }
        }
        printLog(stringBuilder.toString());
    }

    public void printResponseLog () {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("webRequestId = ").append(webRequestId).append(LINE_SEPARATOR);
        stringBuilder.append("webRequestUrl = ").append(getWebRequestFullUrl()).append(LINE_SEPARATOR);
        stringBuilder.append("webRequestMethod = ").append(webRequestMethod).append(LINE_SEPARATOR);
        if (this.username != null && this.password != null) {
            stringBuilder.append("username = ").append(username).append(LINE_SEPARATOR);
            stringBuilder.append("password = ").append(password).append(LINE_SEPARATOR);
        }
        if (webRequestException != null) {
            stringBuilder.append("webRequestException = ").append(webRequestException.toString()).append(LINE_SEPARATOR);
        }
        if (webRequestResponse != null) {
            String result = getResponseString();
            stringBuilder.append("responseCode = ").append(responseCode).append(LINE_SEPARATOR);
            stringBuilder.append("result = ").append(result).append(LINE_SEPARATOR);

        }
        printLog(stringBuilder.toString());
    }

    public void printRequestFullLog () {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("webRequestId = ").append(webRequestId).append(LINE_SEPARATOR);
        stringBuilder.append("webRequestUrl = ").append(getWebRequestFullUrl()).append(LINE_SEPARATOR);
        stringBuilder.append("webRequestMethod = ").append(webRequestMethod).append(LINE_SEPARATOR);
        if (this.username != null && this.password != null) {
            stringBuilder.append("username = ").append(username).append(LINE_SEPARATOR);
            stringBuilder.append("password = ").append(password).append(LINE_SEPARATOR);
        }
        if (!webRequestHeaders.isEmpty()) {
            stringBuilder.append("Headers=>").append(LINE_SEPARATOR);
            for (Map.Entry<String, String> data : webRequestHeaders.entrySet()) {
                stringBuilder.append(data.getKey() + " = " + data.getValue()).append(LINE_SEPARATOR);
            }
        }
        if (!webRequestParams.isEmpty()) {
            stringBuilder.append("Parameters=>").append(LINE_SEPARATOR);
            for (Map.Entry<String, String> data : webRequestParams.entrySet()) {
                stringBuilder.append(data.getKey() + " = " + data.getValue()).append(LINE_SEPARATOR);
            }
        }
        if (!webRequestFiles.isEmpty()) {
            stringBuilder.append("FileParameters=>").append(LINE_SEPARATOR);
            for (Map.Entry<String, File> data : webRequestFiles.entrySet()) {
                stringBuilder.append(data.getKey() + " = " + data.getValue().getAbsolutePath()).append(LINE_SEPARATOR);
            }
        }
        if (webRequestException != null) {
            stringBuilder.append("webRequestException = ").append(webRequestException.toString()).append(LINE_SEPARATOR);
        }
        if (webRequestResponse != null) {
            String result = getResponseString();
            stringBuilder.append("responseCode = ").append(responseCode).append(LINE_SEPARATOR);
            stringBuilder.append("result = ").append(result).append(LINE_SEPARATOR);
        }
        printLog(stringBuilder.toString());
    }

    public void setProgressMessage (String progressMessage) {
        this.progressMessage = progressMessage;
    }

    public void showProgressDialog () {
        if (progressDialog != null &&
                progressMessage != null && !progressMessage.trim().isEmpty()) {
            progressDialog.setMessage(progressMessage);
            progressDialog.show();
        }
    }

    public void hideProgressDialog () {
        if (progressDialog != null && progressDialog.isShowing()) {
            progressDialog.dismiss();
        }
    }

    public String getResponseString () {
        if (webRequestResponse != null) {
            return webRequestResponse;
        }
        return "";
    }

    public WebServiceBaseResponseModel getBaseResponsePojo () throws JsonSyntaxException {
        String response = getResponseString();
        if (response != null && !response.trim().isEmpty()) {
            WebServiceBaseResponseModel baseWebServiceModel = new Gson().fromJson(response, WebServiceBaseResponseModel.class);
            return baseWebServiceModel;

        }
        return null;
    }

    public boolean isSuccess () {
        if (webRequestException == null) {
            try {
                WebServiceBaseResponseModel baseWebServiceModel = getBaseResponsePojo();
                if (baseWebServiceModel != null && !baseWebServiceModel.isError()) {
                    return true;
                }
            } catch (JsonSyntaxException e) {
                showInvalidResponse(getResponseString());
            }

        } else {
            if (webRequestException.getMessage() != null)
                showInvalidResponse(webRequestException.getMessage());
        }
        return false;
    }

    public boolean checkSuccess () {
        if (webRequestException == null) {
            try {
                WebServiceBaseResponseModel baseWebServiceModel = getBaseResponsePojo();
                if (baseWebServiceModel != null && !baseWebServiceModel.isError()) {
                    return true;
                }
            } catch (JsonSyntaxException e) {

            }
        }
        return false;
    }


    public String getErrorMessageFromResponse () {
        try {
            WebServiceBaseResponseModel webServiceBaseResponseModel = getBaseResponsePojo();
            if (webServiceBaseResponseModel != null && webServiceBaseResponseModel.isError()) {
                return webServiceBaseResponseModel.getMessage();
            }
        } catch (JsonSyntaxException e) {

        }
        if (webRequestException != null && webRequestException.getMessage() != null) {
            return webRequestException.getMessage();
        }
        return "";
    }


    public Object getResponsePojo (Class tClass) {

        String response = getResponseString();
        if (response != null && !response.trim().isEmpty()) {
            try {
                return (new Gson().fromJson(response, tClass));
            } catch (JsonSyntaxException e) {
                showInvalidResponse(response);
            }
        }

        return null;
    }


    public int getResponseCode () {
        return responseCode;
    }

    public void showInvalidResponse (String msg) {
        if (context == null) return;
        if (webRequestErrorDialog != null && webRequestErrorDialog.isShowing()) {
            webRequestErrorDialog.dismiss();
        }
        webRequestErrorDialog = new WebRequestErrorDialog(context, msg);
        webRequestErrorDialog.show();
    }

    public void showToast (String message) {
        if (context == null || message == null || message.trim().isEmpty()) return;
        Toast.makeText(context, message, Toast.LENGTH_SHORT).show();
    }

    public void send (Context context) {
        if (!(context instanceof WebServiceResponseListener)) {
            throw new IllegalArgumentException("context should be implement WebServiceResponseListener");
        }
        this.send(context, ((WebServiceResponseListener) context));
    }

    public void send (Context context, final WebServiceResponseListener webServiceResponseListener) {
        isDebug = isDebugBuild(context);
        progressDialog = new ProgressDialog(context);
        progressDialog.setCancelable(false);

        this.context = context;
        if (this.cd == null)
            cd = new ConnectionDetector(context);
        try {
            webServiceResponseListener.onWebRequestCall(this);
            final Call<ResponseBody> call = generateRequest(context);
            printRequestLog();

            if (cd.isConnectingToInternet()) {
                showProgressDialog();
                call.enqueue(new Callback<ResponseBody>() {
                    @Override
                    public void onResponse (Call<ResponseBody> call, Response<ResponseBody> response) {
                        onRequestCompleted(null, response);
                        printResponseLog();
                        hideProgressDialog();
                        webServiceResponseListener.onWebRequestResponse(WebRequest.this);
                    }

                    @Override
                    public void onFailure (Call<ResponseBody> call, Throwable t) {
                        String message = (t != null ? t.getMessage() : "");
                        onRequestCompleted(new IllegalStateException(message), null);
                        printResponseLog();
                        hideProgressDialog();
                        webServiceResponseListener.onWebRequestResponse(WebRequest.this);
                    }
                });

            } else {
                String msg = "Please check your internet connection.";
                WebServiceException e = new WebServiceException(msg,
                        WebServiceException.INTERNET_NOT_AVAILABLE);
                onRequestCompleted(e, null);
                printResponseLog();
                webServiceResponseListener.onWebRequestResponse(WebRequest.this);
            }
        } catch (Exception e) {
            String msg = e.getMessage() == null ? "Exception in call webRequest" : e.getMessage();
            WebServiceException e1 = new WebServiceException(msg,
                    WebServiceException.EXCEPTION_IN_CALL);
            onRequestCompleted(e1, null);
            printResponseLog();
            hideProgressDialog();
            webServiceResponseListener.onWebRequestResponse(WebRequest.this);
        }

    }


    private WebRequestInterface getWebServiceInterface (Certificate certificate) {
        OkHttpClient client = getOkHttpClient(certificate);
        WebRequestInterface webRequestInterface = new Retrofit.Builder()
                .baseUrl("http://www.test.com/")
                .client(client)
                .addConverterFactory(GsonConverterFactory.create())
                .build()
                .create(WebRequestInterface.class);
        return webRequestInterface;
    }

    private OkHttpClient getOkHttpClient (Certificate certificate) {
        OkHttpClient.Builder okClientBuilder = new OkHttpClient.Builder();

        if (certificate != null) {
            // Create an ssl socket factory with our all-trusting manager
            try {
                SSLSocketFactory sslSocketFactory = getSSLConfig(certificate).getSocketFactory();
                if (sslSocketFactory != null)
                    okClientBuilder.sslSocketFactory(sslSocketFactory);
            } catch (CertificateException e) {
                e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (KeyStoreException e) {
                e.printStackTrace();
            } catch (NoSuchAlgorithmException e) {
                e.printStackTrace();
            } catch (KeyManagementException e) {
                e.printStackTrace();
            }
        }


        okClientBuilder.connectTimeout(timeout, TimeUnit.SECONDS);
        okClientBuilder.readTimeout(timeout, TimeUnit.SECONDS);
        okClientBuilder.writeTimeout(timeout, TimeUnit.SECONDS);

        return okClientBuilder.build();
    }

    private SSLContext getSSLConfig (Certificate certificate) throws CertificateException,
            IOException, KeyStoreException, NoSuchAlgorithmException, KeyManagementException {
        if (certificate == null) return null;
        CertificateFactory cf = CertificateFactory.getInstance("X.509");

        // creating a KeyStore containing our trusted CAs
        String keyStoreType = KeyStore.getDefaultType();
        KeyStore keyStore = KeyStore.getInstance(keyStoreType);
        keyStore.load(null, null);
        keyStore.setCertificateEntry("ca", certificate);

        // creating a TrustManager that trusts the CAs in our KeyStore
        String tmfAlgorithm = TrustManagerFactory.getDefaultAlgorithm();
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(tmfAlgorithm);
        tmf.init(keyStore);

        // creating an SSLSocketFactory that uses our TrustManager
        SSLContext sslContext = SSLContext.getInstance("TLS");
        //        sslContext.init(null, tmf.getTrustManagers(), null);
        sslContext.init(null, trustAllCerts, new java.security.SecureRandom());

        return sslContext;
    }


}
