package com.mediatek.mcs.push;

import android.text.TextUtils;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.mediatek.mcs.Mcs;
import com.mediatek.mcs.McsLog;
import com.mediatek.mcs.R;
import com.mediatek.mcs.Utils.ResponseMessage;
import com.mediatek.mcs.domain.McsResponse;
import com.mediatek.mcs.exception.SdkInstallationException;
import com.mediatek.mcs.net.McsJsonRequest;
import com.mediatek.mcs.net.RequestApi;
import com.mediatek.mcs.net.RequestManager;
import com.mediatek.mcs.net.RequestUtils;
import com.mediatek.mcs.pref.McsMobile;
import lombok.Getter;
import lombok.Setter;
import org.json.JSONObject;

public final class McsPushInstallation {

  private static McsPushInstallation INSTANCE = new McsPushInstallation();
  @Getter private String gcmSenderId;
  @Getter @Setter private String appKey;
  @Getter @Setter private String regId;
  @Getter @Setter private boolean isUpdateRegId;

  private McsPushInstallation() {
    this.gcmSenderId = "";
    this.appKey = "";
    this.regId = "";
    this.isUpdateRegId = false;
  }

  public static McsPushInstallation getInstance() {
    return INSTANCE;
  }

  public void registerInBackground(String gcm_sender_id, String gcm_api_key,
      McsResponse.SuccessListener<JSONObject> successListener,
      McsResponse.ErrorListener errorListener) {
    if (!isValidInstall(gcm_sender_id, gcm_api_key)) return;

    this.gcmSenderId = gcm_sender_id;
    this.appKey = gcm_api_key;
    new PushRegistration(gcmSenderId, (regId, isUpdateRegId) -> {
      this.regId = regId;
      this.isUpdateRegId = isUpdateRegId;
      McsLog.d("Gcm installed successfully: " + regId);

      McsPushInstallation.getInstance().requestRegisterMobile(
          successListener::onSuccess, errorListener::onError);
    });
  }

  public void registerInBackground(String gcm_sender_id, String gcm_api_key) {
    this.registerInBackground(gcm_sender_id, gcm_api_key,
        response -> McsLog.d(response.toString()),
        error -> McsLog.e(error)
    );
  }

  boolean isValidInstall(String gcmSenderId, String appKey) {
    if (Mcs.getApplicationContext() == null) {
      McsLog.e(new SdkInstallationException(R.string.error_install_not_init));
      return false;
    }

    int playServiceResult = GooglePlayServicesUtil
        .isGooglePlayServicesAvailable(Mcs.getApplicationContext());
    if (playServiceResult != ConnectionResult.SUCCESS) {
      McsLog.e(new SdkInstallationException(Mcs.getApplicationContext()
          .getString(R.string.error_install_no_google_play)
          + "\nplayServiceResult errorCode: " + playServiceResult));
      return false;
    }

    if (TextUtils.isEmpty(gcmSenderId) || TextUtils.isEmpty(appKey)) {
      McsLog.e(new SdkInstallationException(R.string.error_install_push_gcm_invalid));
      return false;
    }

    return true;
  }

  /**
   * Note that you should take care isGooglePlayServicesAvailable in your SignIn activity.
   * You could use PlayServicesUtils.isGooglePlayServicesAvailable(Context);
   *
   * @return Whether developer enable push installation by using
   * {@link McsPushInstallation#registerInBackground(String, String)}
   * with correct parameters. It's up to developers to decide
   * whether to use the push notification feature or not.
   */
  public boolean isPushInstalled() {
    return !(TextUtils.isEmpty(gcmSenderId) || TextUtils.isEmpty(appKey));
  }

  /**
   * Required {@link McsPushInstallation#registerInBackground(String, String)
   * McsPushInstallation registerInBackground} method in advanced
   */
  public void requestRegisterMobile(McsResponse.SuccessListener<JSONObject> successListener,
      McsResponse.ErrorListener errorListener) {
    if (TextUtils.isEmpty(regId)) {
      registerInBackground(gcmSenderId, appKey);
      errorListener.onError(new SdkInstallationException(R.string.error_install_push_reg_token_empty));
      return ;
    }

    int method;
    String url;
    String responseAction;

    if (isUpdateRegId) {
      responseAction = "Update mobile";
      method = McsJsonRequest.Method.PUT;
      url = RequestApi.MOBILE
          .replace("{mobileId}", McsMobile.getInstance().getMobileId());
    } else {
      responseAction = "Register mobile";
      method = McsJsonRequest.Method.POST;
      url = RequestApi.MOBILES;
    }

    final String finalResponseAction = responseAction;
    RequestManager.sendInBackground(new McsJsonRequest(
        method,
        url,
        new McsRegisterMobile(this.regId, this.appKey).toString(),
        response -> {
          if (!isUpdateRegId) {
            /**
             * only the register mobile one has mobileId inside the response
             */
            McsMobile.getInstance().saveToPref(response);
          }
          successListener.onSuccess(ResponseMessage.create(finalResponseAction + " succeeded."));
        },
        error -> errorListener.onError(RequestUtils.parseVolleyError(error))
    ));
  }

  public void requestUnRegisterMobile() {
    if (TextUtils.isEmpty(McsMobile.getInstance().getMobileId())) {
      McsLog.e("mobileId is empty, still sign out.");
      return ;
    }

    RequestManager.sendInBackground(new McsJsonRequest(
        McsJsonRequest.Method.DELETE,
        RequestApi.MOBILE
            .replace("{mobileId}", McsMobile.getInstance().getMobileId()),
        response -> {
          McsMobile.getInstance().clearPref();
          McsLog.d("mobile unregistered.");
        },
        error -> McsLog.e("Unexpected error: " + error.toString())));
  }
}
