package com.mediatek.mcs.domain;

import android.text.TextUtils;
import com.mediatek.mcs.Mcs;
import com.mediatek.mcs.McsLog;
import com.mediatek.mcs.R;
import com.mediatek.mcs.Utils.ResponseMessage;
import com.mediatek.mcs.entity.AuthEntity;
import com.mediatek.mcs.exception.InvalidSessionException;
import com.mediatek.mcs.exception.SdkInstallationException;
import com.mediatek.mcs.net.McsJsonRequest;
import com.mediatek.mcs.net.RequestApi;
import com.mediatek.mcs.net.RequestManager;
import com.mediatek.mcs.net.RequestUtils;
import com.mediatek.mcs.net.SignInRequest;
import com.mediatek.mcs.pref.McsUser;
import com.mediatek.mcs.pref.McsUserInfo;
import com.mediatek.mcs.push.McsPushInstallation;
import java.util.HashMap;
import lombok.Setter;
import org.json.JSONObject;

public class McsSession {

  private static McsSession INSTANCE = new McsSession();
  protected McsResponse.SuccessListener<JSONObject> successListener;
  @Setter protected McsResponse.ErrorListener errorListener;
  protected McsSession() {

  }

  public static McsSession getInstance() {
    return INSTANCE;
  }

  public McsUser getCurrentUser() {
    return McsUser.getInstance();
  }

  public boolean isSignedIn() {
    return !"".equals(getCurrentUser().getAccessToken());
  }

  public String getAccessToken() {
    return McsUser.getInstance().getAccessToken();
  }

  public McsUserInfo getUserInfo() {
    return McsUserInfo.getInstance();
  }

  public void saveTokenResultsToPref(AuthEntity.AccessToken.Results results) {
    McsUser.getInstance().setEmail(results.getEmail());
    McsUser.getInstance().setToken(results.getToken());
    McsUser.getInstance().setAccessToken(results.getAccessToken());
    McsUser.getInstance().savePref();

    McsUserInfo.getInstance().setNickname(results.getNickname());
    McsUserInfo.getInstance().setUserImageURL(results.getUserImageURL());
    McsUserInfo.getInstance().savePref();
  }

  /**
   * Request Methods
   */
  public void requestSignIn(String email, String pwd,
      McsResponse.SuccessListener<JSONObject> successListener) {
    this.requestSignIn(email, pwd, successListener, (error) -> McsLog.e(error.getMessage()));
  }

  public void requestSignIn(String email, String pwd,
      McsResponse.SuccessListener<JSONObject> successListener,
      McsResponse.ErrorListener errorListener) {
    this.successListener = successListener;
    this.errorListener = errorListener;

    if (isValidAuth(email, pwd)) {
      McsUser.getInstance().setEmail(email);
      McsUser.getInstance().setPassword(pwd);
      this.requestSignInToken();
    }
  }

  public void requestSignOut(McsResponse.SuccessListener<JSONObject> successListener) {
    McsPushInstallation.getInstance().requestUnRegisterMobile();

    McsUser.getInstance().clearPref();
    McsUserInfo.getInstance().clearPref();
    successListener.onSuccess(ResponseMessage.create("Sign out successfully"));
  }

  private void requestSignInToken() {
    RequestManager.sendInBackground(new SignInRequest(
        McsJsonRequest.Method.POST,
        RequestApi.SIGN_IN,
        SignInRequest.getSignInParams(),
        response -> {
          McsUser.getInstance().saveToPref(response);

          requestUserInfo();
        },
        error -> errorListener.onError(RequestUtils.parseVolleyError(error))
    ));
  }

  protected void requestUserInfo() {
    RequestManager.sendInBackground(new McsJsonRequest(RequestApi.USER_INFO, response -> {
      McsUserInfo.getInstance().saveToPref(response);

      if (McsPushInstallation.getInstance().isPushInstalled()) {
        McsPushInstallation.getInstance().requestRegisterMobile(successListener, errorListener);
      } else {
        successListener.onSuccess(ResponseMessage.create("Sign in without register mobile"));
      }
    }, error -> errorListener.onError(RequestUtils.parseVolleyError(error))) {
      @Override public HashMap<String, String> getHeaders() {
        HashMap<String, String> params = new HashMap<>();
        params.put("Authorization", "Bearer " + McsSession.getInstance().getAccessToken());
        return params;
      }
    });
  }

  protected boolean isValidAuth() {
    boolean isValid = false;

    if (TextUtils.isEmpty(Mcs.MCS_APP_ID) || TextUtils.isEmpty(Mcs.MCS_APP_SECRET)) {
      errorListener.onError(new SdkInstallationException(R.string.error_install_credential_invalid));
    } else {
      isValid = true;
    }
    return isValid;
  }

  /**
   * Validator
   */
  protected boolean isValidAuth(final String _user, final String _pwd) {
    boolean isValid = false;

    if (TextUtils.isEmpty(Mcs.MCS_APP_ID) || TextUtils.isEmpty(Mcs.MCS_APP_SECRET)) {
      errorListener.onError(new SdkInstallationException(R.string.error_install_credential_invalid));
    } else if (_user.equals("")) {
      errorListener.onError(new InvalidSessionException(R.string.error_signin_username_blank));
    } else if (!isEmailValid(_user)) {
      errorListener.onError(new InvalidSessionException(R.string.error_signin_email_invalid));
    } else if (_pwd.equals("")) {
      errorListener.onError(new InvalidSessionException(R.string.error_signin_pwd_blank));
    } else {
      isValid = true;
    }
    return isValid;
  }

  private boolean isEmailValid(CharSequence email) {
    return android.util.Patterns.EMAIL_ADDRESS.matcher(email).matches();
  }
}
