/*
 * Copyright (c) 2023 MarkLogic Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.marklogic.xcc.exceptions;

import com.marklogic.xcc.Request;
import com.marklogic.xcc.Version;
import com.marklogic.xcc.impl.SessionImpl;

/**
 * The base class for exceptions related to submitting requests to the server.
 */
public class RequestException extends XccException {
    private static final long serialVersionUID = -7193834172330203276L;
    private transient Request request;
    private final boolean isRetryable;


    public RequestException(String message, Request request) {
        this(message, request, false);
    }

    public RequestException(String message, Request request, Throwable cause) {
        this(message, request, cause, false);
    }

    public RequestException(String message, Request request,
        boolean isRetryable) {
        super(message);
        this.request = request;
        this.isRetryable = isRetryable;
    }

    public RequestException(String message, Request request, Throwable cause,
        boolean isRetryable) {
        super(message, cause);
        this.request = request;
        this.isRetryable = isRetryable;
    }

    /**
     * The initiating request to which this exception applies.
     * 
     * @return An instance of {@link Request}.
     */
    public Request getRequest() {
        return request;
    }

    // We are keeping both isRetryable and isRetryAdvised for backward
    // compatibility of RequestPermissionException
    public boolean isRetryable() {
        return isRetryable;
    }

    public boolean isRetryAdvised() {
        return isRetryable;
    }

    @Override
    public String toString() {
        StringBuffer sb = new StringBuffer(super.toString());
        appendRequestInfo(sb);
        return sb.toString();
    }
    
    protected void appendRequestInfo(StringBuffer sb) {
        String nl = System.getProperty("line.separator");
        String serverVersion = ((SessionImpl)request.getSession()).getServerVersion();
        sb.append(nl);
        sb.append(" [Session: ");
        sb.append(request.getSession().toString());
        sb.append("]");
        sb.append(nl);
        sb.append(" [Client: XCC/");
        sb.append(Version.getVersionString());
        if (serverVersion != null) {
            sb.append(", Server: XDBC/");
            sb.append(serverVersion);
        }
        sb.append("]");
    }
}
