/**
 Copyright (c) 2021 MarkLogic Corporation

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */
'use strict';

/**
 * Feature that handles the schema validation of the artifacts and instances.
 */

import consts from "/data-hub/5/impl/consts.mjs";
import hubUtils from "/data-hub/5/impl/hub-utils.mjs";
import flowUtils from "/data-hub/5/impl/flow-utils.mjs";
import entityValidation from "/data-hub/5/builtins/steps/mapping/entity-services/entity-validation-lib.mjs";
const es = require('/data-hub/core/entity-services/entity-services-impl.xqy');
const hent = require("/data-hub/5/impl/hub-entities.xqy");

const INFO_EVENT = consts.TRACE_CORE;
const DEBUG_EVENT = consts.TRACE_CORE_DEBUG;
const DEBUG_ENABLED = xdmp.traceEnabled(DEBUG_EVENT);

function onArtifactSave(artifactType, artifactName, artifactUri, entityModel) {
  if ("model" === artifactType) {
    let invalidModel = false;
    try {
      es.modelValidate(xdmp.toJSON(entityModel));
    } catch (ignore) {
      invalidModel = true;
    }
    if (invalidModel) {
      hubUtils.hubTrace(INFO_EVENT, `Not creating schemas for ${artifactName} because the model is invalid.`);
      return;
    }
    hubUtils.hubTrace(INFO_EVENT, `Schema validation feature: Creating schemas for ${artifactName}.`);

    let schemaPermissions = xdmp.defaultPermissions().concat([
      xdmp.permission("data-hub-common", "read"),
      xdmp.permission("data-hub-entity-model-writer", "update")]);
    let xmlSchemaCollection = "ml-data-hub-xml-schema";
    let jsonSchemaCollection = "ml-data-hub-json-schema";

    let xmlSchema = fn.head(es.schemaGenerate(entityModel));

    const jsonSchema = hent.jsonSchemaGenerate(entityModel.info.title, entityModel);
    xdmp.invokeFunction(function () {
      xdmp.documentInsert(fn.replace(artifactUri, "\\.json$", ".xsd"), xmlSchema, schemaPermissions, xmlSchemaCollection);
      xdmp.documentInsert(fn.replace(artifactUri, "\\.json$", ".schema.json"), jsonSchema, schemaPermissions, jsonSchemaCollection);
    }, {database: xdmp.schemaDatabase(), update: "true"});

    hubUtils.hubTrace(INFO_EVENT, `Schema validation feature: Finished creating schemas for ${artifactName}.`);

  }
}

function onInstanceSave(stepContext, model, contentArray) {
  if (stepContext.stepDefinition && stepContext.stepDefinition.type.toLowerCase() !== "mapping") {
    hubUtils.hubTrace(INFO_EVENT, `Schema validation feature: Cannot validate. This is not a mapping step.`);
    return contentArray;
  }
  if (!model) {
    hubUtils.hubTrace(INFO_EVENT, `Schema validation feature: Skipping. Missing model information. Look for targetEntityType.`);
    return contentArray;
  }

  const options = stepContext.combinedOptions;
  if (!entityValidation.shouldValidateEntity(options)) {
    hubUtils.hubTrace(INFO_EVENT, `Schema validation feature: Skipping since options are set to not validate schema for ${model.info.title}.`);
    return contentArray;
  }

  hubUtils.hubTrace(INFO_EVENT, `Schema validation feature: Validation schema for content of type ${model.info.title} started.`);
  const modelDef = model.info.title;
  let outputContentArray = [];

  contentArray.forEach(contentObject => {
    const contentValue = contentObject.value;
    let instance = {};
    //We assume there will be an envelope since mapping function makes the envelope and it is the previous step.
    if (hubUtils.isXmlNode(contentValue)) {
      instance = fn.head(contentValue.xpath('/*:envelope/*:instance/*:' + modelDef));
    } else {
      instance[modelDef] = contentValue.toObject().envelope.instance[model.info.title];
    }
    try {
      entityValidation.validateEntity(instance, options, model.info);
      if (!options.headers || !options.headers.datahub || !options.headers.datahub.validationErrors) {
        outputContentArray.push(contentObject);
      } else {

        const contentUpdated = generateUpdatedContent(options, contentObject, contentValue);
        outputContentArray.push(contentUpdated);

        // Must remove these so that they're not carried over to another item in a batch
        entityValidation.removeValidationErrorsFromHeaders(options);

      }
    } catch (error) {
      hubUtils.hubTrace(INFO_EVENT, `Schema validation feature: Errors were caught when validating schema: ${error}.`);
      if (stepContext && stepContext.isStopOnError && stepContext.stopWithError && stepContext.addStepErrorForItem && stepContext.removeCompletedItem) {
        if (stepContext.isStopOnError()) {
          stepContext.stopWithError(error, contentObject.uri);
          return [];
        }
        stepContext.addStepErrorForItem(error, contentObject.uri);
        stepContext.removeCompletedItem(contentObject.uri);
      }
    }
  });
  hubUtils.hubTrace(INFO_EVENT, `Schema validation feature: Validation schema for content of type ${model.info.title} ended.`);
  return outputContentArray;
}

function generateUpdatedContent(options, contentObject, contentValue) {
  const errors = options.headers.datahub.validationErrors;
  hubUtils.hubTrace(INFO_EVENT, `Schema validation feature: Schema validation for content ${contentObject.uri} got errors.`);
  if (DEBUG_ENABLED && errors.formattedMessages) {
    errors.formattedMessages.forEach(msg => {
      hubUtils.hubTrace(DEBUG_EVENT, `Schema validation feature: Error - ${msg}`);
    });
  }

  // Add the validation errors to the content
  let headers = flowUtils.createHeaders(options);
  let docHeaders = flowUtils.normalizeValuesInNode(flowUtils.getHeaders(contentValue)) || {};

  headers = flowUtils.mergeHeaders(headers, docHeaders, options.outputFormat);

  if (hubUtils.isXmlNode(contentValue)) {
    const nb = new NodeBuilder();
    nb.startDocument();
    nb.startElement("envelope", "http://marklogic.com/entity-services");
    nb.startElement("headers", "http://marklogic.com/entity-services");
    nb.addNode(headers);
    nb.endElement();
    nb.addNode(contentValue.xpath('/*:envelope/*:triples'));
    nb.addNode(contentValue.xpath('/*:envelope/*:instance'));
    let attachments = contentValue["$attachments"];
    if (hubUtils.isXmlNode(attachments)) {
      nb.addNode(attachments);
    }
    nb.endElement();
    nb.endDocument();
    contentObject.value = nb.toNode();

  } else {
    let contentObj = contentValue.toObject();
    contentObj.envelope.headers = headers;
    contentObject.value = contentObj;
  }

  return contentObject;
}

export default {
  onArtifactSave,
  onInstanceSave
};
