/**
 Copyright (c) 2022 MarkLogic Corporation

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

import config from "/com.marklogic.hub/config.mjs";
import jobQueryLib from "/data-hub/5/flow/job-query-lib.mjs";
import consts from "/data-hub/5/impl/consts.mjs";
import entityLib from "/data-hub/5/impl/entity-lib.mjs";
const hent = require("/data-hub/5/impl/hub-entities.xqy");

const granularPrivileges = ["admin-database-clear-data-hub-STAGING",
  "admin-database-clear-data-hub-FINAL",
  "admin-database-clear-data-hub-JOBS",
  "admin-database-index-data-hub-STAGING",
  "admin-database-index-data-hub-FINAL",
  "admin-database-index-data-hub-JOBS",
  "admin-database-triggers-data-hub-staging-TRIGGERS",
  "admin-database-triggers-data-hub-final-TRIGGERS",
  "admin-database-temporal-data-hub-STAGING",
  "admin-database-temporal-data-hub-FINAL",
  "admin-database-alerts-data-hub-STAGING",
  "admin-database-alerts-data-hub-FINAL",
  "admin-database-amp-data-hub-MODULES"];

const stagingDbId = xdmp.database(config.STAGINGDATABASE);
const stagingTriggersDbId = xdmp.triggersDatabase(stagingDbId);
const finalDbId = xdmp.database(config.FINALDATABASE);
const finalTriggersDbId = xdmp.triggersDatabase(finalDbId);
const jobDbId = xdmp.database(config.JOBDATABASE);
const modulesDbId = xdmp.modulesDatabase();
const hostStatuses = xdmp.hostStatus(xdmp.hosts());

const databaseIds = {
  stagingDbId,
  stagingTriggersDbId,
  finalDbId,
  finalTriggersDbId,
  jobDbId,
  modulesDbId
};

function appServersCheck() {
  const component = {
    componentName: "Data Hub App Servers",
    checks: []
  };
  component.checks.push(...appServersAuthCheck());
  component.success = component.checks.every(check => check.success);
  return component;
}


function appServersAuthCheck() {
  const components = [];
  for (const hostStatus of hostStatuses) {
    const hostName = hostStatus.hostName;
    const component = {
      componentName: `Data Hub App Servers Authentication: ${hostName}`,
      checks: []
    };
    component.checks.push(...portsCheck(hostStatus));
    component.success = component.checks.every(check => check.success);
    components.push(component);
  }
  return components;
}

function portsCheck() {
  const appserverName = "";
  const component = {
    componentName: `Data Hub App Server Port -- ${appserverName}`,
    checks: []
  };

  component.success = component.checks.length === 0 || component.checks.every(check => check.success);
  return [component];
}

function securityCheck() {
  const component = {
    componentName: "Data Hub Security",
    checks: []
  };
  component.checks.push(rolesCheck());
  component.checks.push(privilegesCheck());
  component.success = component.checks.every(check => check.success);
  return component;
}

function rolesCheck() {
  const component = {
    componentName: "Data Hub Security -- Roles",
    checks: []
  };
  for (const [key, roleName] of Object.entries(consts)) {
    if (key.endsWith("_ROLE")) {
      let exists = false;
      try {
        xdmp.role(roleName);
        exists = true;
      } catch (ignore) {}
      component.checks.push({componentName: `Role ${key} exists!`, success: exists});
    }
  }
  component.success = component.checks.length === 0 || component.checks.every(check => check.success);
  return component;
}
function privilegesCheck() {
  const component = {
    componentName: "Data Hub Security -- Granular Privileges",
    checks: []
  };
  xdmp.invokeFunction(() => {
    for (const privilegeName of granularPrivileges) {
      const privilegeXML = fn.head(cts.search(cts.elementValueQuery(fn.QName("http://marklogic.com/xdmp/security", "privilege-name"), privilegeName)));
      const privilegeChecks = [];
      const privilegeExists = fn.exists(privilegeXML);
      privilegeChecks.push({
        componentName: "Privilege exists",
        success: privilegeExists
      });
      const expectedDbId = granularPrivilegeDatabaseID(privilegeName);
      const bothPrivilegeChecksPass = privilegeExists && fn.contains(fn.string(privilegeXML.xpath("sec:privilege/sec:action")), `/${expectedDbId}`);
      privilegeChecks.push({
        componentName: "References expected database",
        success: bothPrivilegeChecksPass
      });
      component.checks.push({
        componentName: `Granular Privilege: ${privilegeName}`,
        success: bothPrivilegeChecksPass,
        checks: privilegeChecks
      });
    }
  }, {database: xdmp.securityDatabase()});
  component.success = component.checks.length === 0 || component.checks.every(check => check.success);
  return component;
}

function granularPrivilegeDatabaseID(privilegeName) {
  if (privilegeName.endsWith("data-hub-MODULES")) {
    return databaseIds.modulesDbId;
  }
  if (privilegeName.endsWith("data-hub-STAGING")) {
    return databaseIds.stagingDbId;
  }
  if (privilegeName.endsWith("data-hub-staging-TRIGGERS")) {
    return databaseIds.stagingTriggersDbId;
  }
  if (privilegeName.endsWith("data-hub-FINAL")) {
    return databaseIds.finalDbId;
  }
  if (privilegeName.endsWith("data-hub-final-TRIGGERS")) {
    return databaseIds.finalTriggersDbId;
  }
  if (privilegeName.endsWith("data-hub-JOBS")) {
    return databaseIds.jobDbId;
  }
}



function entityModelsCheck() {
  const component = {
    componentName: "Data Hub Models",
    checks: []
  };
  xdmp.invokeFunction(() => {
    const models = cts.search(cts.collectionQuery("http://marklogic.com/entity-services/models"));
    for (const model of models) {
      let modelObject = model.toObject();
      component.checks.push(modelCheck(modelObject));
    }
  }, {database: xdmp.database(config.FINALDATABASE)});
  component.success = component.checks.every(check => check.success);
  return component;
}

function modelCheck(model) {
  const protectedPathsFunc = getProtectedPathsFunc();
  const component = {
    componentName: `Data Hub Model -- ${model.info.title}`,
    checks: []
  };

  component.checks.push(structureCheck(model));
  component.checks.push(tdeCheck(model));
  component.checks.push(protectedPathsFunc(model));
  component.checks.push(indexesCheck(model));
  component.checks.push(facetCheck(model));


  component.success = component.checks.length === 0 || component.checks.every(check => check.success);
  return component;
}

function tdeCheck(model) {
  const component = {
    checks: []
  };
  xdmp.invokeFunction(() => {

    let isTdeGenerationEnabled = hent.isTdeGenerationEnabled(model);
    if (isTdeGenerationEnabled) {
      component.checks.push({componentName: `Template Driven Extraction`, success: checkForTde("/tde/"+model.info.title + "-" + model.info.version + ".tdex")});
    } else {
      component.checks.push({componentName: `Template Driven Extraction`, success: true});
    }
  }, {database: xdmp.database(config.FINALDATABASE)});
  component.success = component.checks.length === 0 || component.checks.every(check => check.success);
  return component;
}

function structureCheck(model) {
  const component = {
    checks: []
  };

  try {
    entityLib.validateModelDefinitions(model.definitions);
    component.checks.push({componentName: 'model definitions', success: true});
  } catch (ignore) {
    component.checks.push({componentName: 'model definitions', success: false});
  }



  const name = model.info.title;
  if (name == null) {
    component.checks.push({componentName: `title property`, success: false});
  } else {
    component.checks.push({componentName: `title property`, success: true});
  }

  let isTdeGenerationEnabled = hent.isTdeGenerationEnabled(model);
  if (isTdeGenerationEnabled) {
    component.checks.push({componentName: `Template Driven Extraction`, success: checkForTde("/tde/"+model.info.title + "-" + model.info.version + ".tdex")});
  } else {
    component.checks.push({componentName: `Template Driven Extraction`, success: true});
  }

  component.success = component.checks.length === 0 || component.checks.every(check => check.success);
  return component;
}

function checkForTde(uri) {
  return fn.head(xdmp.eval(
    `fn.docAvailable('${uri}') `,
    {uri: uri}, {database: xdmp.schemaDatabase()}
  ));
}

function protectedPathCheck(model) {
  const component = {
    checks: []
  };
  xdmp.invokeFunction(() => {
    const entityName = model.info.title;
    if (model.definitions && model.definitions[entityName] && model.definitions[entityName].pii) {
      const piiElements = model.definitions[entityName].pii;

      for (const piiElement of piiElements) {
        const endWith = model.info.title + "/" + piiElement;
        const protectedPathXML = fn.head(cts.search(cts.elementValueQuery(fn.QName("http://marklogic.com/xdmp/security", "path-expression"), "/(es:envelope|envelope)/(es:instance|instance)/" + endWith)));
        const protectedPathExists = fn.exists(protectedPathXML);
        component.checks.push({
          componentName: `protected path -- ${piiElement}`,
          success: protectedPathExists
        });

      }
    }
  }, {database: xdmp.securityDatabase()});
  component.success = component.checks.length === 0 || component.checks.every(check => check.success);
  return component;
}

function indexesCheck(model) {
  const component = {
    checks: []
  };
  xdmp.invokeFunction(() => {

    const entityName = model.info.title;

    const admin = require('/MarkLogic/admin.xqy');
    const xmlIndexes = admin.databaseGetRangePathIndexes(admin.getConfiguration(), xdmp.database(config.FINALDATABASE));

    if (model.definitions && model.definitions[entityName] && model.definitions[entityName].properties) {
      const properties = model.definitions[entityName].properties;
      for (const propertyName in properties) {
        const property = properties[propertyName];

        if (property.sortable === true) {
          const xmlString = xmlIndexes.toString();
          const indludeIndex = xmlString.includes(model.info.title) && xmlString.includes(propertyName);

          component.checks.push({
            componentName: `index path -- ${propertyName}`,
            success: indludeIndex
          });
        }
      }
    }
  }, {database: xdmp.database(config.FINALDATABASE)});
  component.success = component.checks.length === 0 || component.checks.every(check => check.success);
  return component;
}

function facetCheck(model) {
  const component = {
    checks: []
  };
  xdmp.invokeFunction(() => {

    const entityName = model.info.title;
    const entityOptionXML = cts.doc(`/${xdmp.groupName()}/${config.FINALDATABASE}/rest-api/options/exp-final-entity-options.xml`);

    if (model.definitions && model.definitions[entityName] && model.definitions[entityName].properties) {
      const properties = model.definitions[entityName].properties;
      for (const propertyName in properties) {
        const property = properties[propertyName];

        if (property.facetable === true) {
          const constraintName = model.info.title + "." + propertyName;
          const xpathExpression = `//*[@name='${constraintName}' and */@facet=true()]`;
          const facetExists = fn.exists(entityOptionXML);

          component.checks.push({
            componentName: `Search options -- ${propertyName}`,
            success: facetExists && fn.exists(fn.string(entityOptionXML.xpath(xpathExpression)))
          });
        }
      }
    }
  }, {database: xdmp.modulesDatabase()});
  component.success = component.checks.length === 0 || component.checks.every(check => check.success);
  return component;
}

function jobsCheck() {
  const component = {
    componentName: "Data Hub Jobs",
    checks: []
  };
  xdmp.invokeFunction(() => {

    const jobs = jobQueryLib.findLastFailedJobs();

    if (jobs.results && jobs.results.length > 0) {
      for (const job of jobs.results) {
        let errorMessage = "";

        if (job.jobId) errorMessage += `JobID: ${job.jobId}`;
        const errorMessageAndStepName = returnErrorMessageAndStepName(job.jobId);
        if (errorMessageAndStepName) errorMessage +=  errorMessageAndStepName;
        component.checks.push({componentName: errorMessage, success: false});
      }
    } else {
      component.checks.push({componentName: "There are no jobs in the failure state", success: true});
    }
  }, {database: xdmp.database(config.JOBDATABASE)});
  component.success = component.checks.every(check => check.success);
  return component;
}
function  returnErrorMessageAndStepName(jobId) {
  const valueStep = fn.head(cts.search(cts.andQuery([
    cts.collectionQuery("Jobs"),
    cts.jsonPropertyValueQuery("jobId", jobId, "case-insensitive")
  ]), ["score-zero", "unfaceted"], 0));

  const valueStepAux =  valueStep.toObject();
  let message = "";
  if (valueStepAux.batch) {
    if (valueStepAux.batch.error != null && valueStepAux.batch.error !== undefined) {
      message += " Error message: " + valueStepAux.batch.error;
    }
    if (valueStepAux.batch && valueStepAux.batch.step) {
      message += " Step name: " + valueStepAux.batch.step.name;
    }
  }
  return message;
}

function getProtectedPathsFunc() {
  return import.meta.amp(protectedPathCheck);
}

function componentChecks() {
  return [
    appServersCheck(),
    securityCheck(),
    entityModelsCheck(),
    jobsCheck()
  ];
}

export default {
  componentChecks
};
