package com.mapbox.android.accounts.navigation.sku.v1;

import android.annotation.SuppressLint;
import android.content.SharedPreferences;

import androidx.annotation.Keep;
import androidx.annotation.NonNull;

import com.mapbox.android.accounts.v1.MapboxAccounts;

/**
 * The class is responsible to request @link[MapboxAccounts.java] to generate the session token
 * and rotate the SKU token based on the refresh time
 */
public class TripsSku implements SkuGenerator {

    enum RotateTripsType {
        NONE_EXPIRED_NO_ROTATION,
        ROTATE_ON_TIMER_EXPIRE,
        ROTATE_ON_REQUEST_COUNT_EXPIRE
    }

    private static final String KEY_PREFERENCE_NAV_TRIPS_SKU = "com.mapbox.navigation.accounts.trips.skutoken";
    private static final String KEY_PREFERENCE_NAV_ROUTE_REQ_COUNT = "com.mapbox.navigation.accounts.trips.request.count";
    private static final String KEY_PREFERENCE_NAV_TRIPS_TIMESTAMP = "com.mapbox.navigation.accounts.trips.timestamp";
    private static final int DEFAULT_TRIP_REQUEST_COUNT = 1;
    private static final long DEFAULT_TRIP_TOKEN_TIMER = 0L;

    private boolean isNavigating = false;
    private SharedPreferences preferences;
    private long timerExpireAfter;
    private int routeRequestThreshold;
    private String applicationId;

    @Keep
    public TripsSku(
            @NonNull SharedPreferences preferences,
            long timerExpireAfter,
            int routeRequestThreshold,
            @NonNull String applicationId
    ) {
        this.preferences = preferences;
        this.applicationId = applicationId;
        this.timerExpireAfter = timerExpireAfter;
        this.routeRequestThreshold = routeRequestThreshold;
    }

    /**
     * Generates a SKU token. It also sets the route request count to default and timer to current time
     */
    @Keep
    @Override
    public void initializeSKU() {
        setTimerExpiry(getNow());
        setRouteRequestCount(DEFAULT_TRIP_REQUEST_COUNT);
        persistTripsSkuToken();
    }

    /**
     * Generates the SKU token
     * @return the SKU token
     */
    @Keep
    @Override
    public String generateToken() {
        refreshSkuToken();
        return retrieveTripsSkuToken();
    }

    /**
     * Sets the number of request made to DirectionsAPI count to the default value
     */
    @Keep
    @Override
    public void onNavigationStart() {
        this.isNavigating = true;
        setRouteRequestCount(DEFAULT_TRIP_REQUEST_COUNT);
    }

    /**
     * Sets the last time a request was made to DirectionsAPI to the default value
     */
    @Keep
    @Override
    public void onNavigationEnd() {
        this.isNavigating = false;
        setTimerExpiry(DEFAULT_TRIP_TOKEN_TIMER);
    }

    private void refreshSkuToken() {
        if (isNavigating) {
            handleRotationOnNavActive();
        } else {
            handleRotationOnNavInActive();
        }
    }

    private void handleRotationOnNavActive() {
        if (validateTimerExpiry()) {
            setTimerExpiry(getNow());
            persistTripsSkuToken();
        }
    }

    private void handleRotationOnNavInActive() {
        int requestCount;
        RotateTripsType rotationType = validateRotation();
        if (rotationType == RotateTripsType.ROTATE_ON_TIMER_EXPIRE
                || rotationType == RotateTripsType.ROTATE_ON_REQUEST_COUNT_EXPIRE) {
            requestCount = 1;
            setTimerExpiry(getNow());
            persistTripsSkuToken();
        } else {
            requestCount = getRouteRequestCount();
            requestCount++;
        }
        setRouteRequestCount(requestCount);
    }

    private RotateTripsType validateRotation() {
        boolean routeReqCountExpired = validateRouteRequestCountExpiry();
        boolean timerExpired = validateTimerExpiry();
        if (routeReqCountExpired) {
            return RotateTripsType.ROTATE_ON_REQUEST_COUNT_EXPIRE;
        } else if (timerExpired) {
            return RotateTripsType.ROTATE_ON_TIMER_EXPIRE;
        } else {
            return RotateTripsType.NONE_EXPIRED_NO_ROTATION;
        }
    }

    private boolean validateTimerExpiry() {
        long skuTokenTimeStamp = getTimerExpiry();
        return isTwoHoursExpired(skuTokenTimeStamp);
    }

    private boolean validateRouteRequestCountExpiry() {
        int routeRequestCount = getRouteRequestCount();
        return routeRequestCount > routeRequestThreshold;
    }

    @SuppressLint("ApplySharedPref")
    private void setRouteRequestCount(int count) {
        preferences.edit().putInt(applicationId + "." + KEY_PREFERENCE_NAV_ROUTE_REQ_COUNT, count).apply();
    }

    private int getRouteRequestCount() {
        return preferences.getInt(applicationId + "." + KEY_PREFERENCE_NAV_ROUTE_REQ_COUNT, DEFAULT_TRIP_REQUEST_COUNT);
    }

    @SuppressLint("ApplySharedPref")
    private void persistTripsSkuToken() {
        String token = generateTripsSkuToken();
        preferences.edit().putString(applicationId + "." + KEY_PREFERENCE_NAV_TRIPS_SKU, token).apply();
    }

    private String retrieveTripsSkuToken() {
        return preferences.getString(applicationId + "." + KEY_PREFERENCE_NAV_TRIPS_SKU, "");
    }

    private String generateTripsSkuToken() {
        return MapboxAccounts.obtainNavigationSkuSessionToken();
    }

    @SuppressLint("ApplySharedPref")
    private void setTimerExpiry(long then) {
        preferences.edit().putLong(applicationId + "." + KEY_PREFERENCE_NAV_TRIPS_TIMESTAMP, then).apply();
    }

    private long getTimerExpiry() {
        return preferences.getLong(applicationId + "." + KEY_PREFERENCE_NAV_TRIPS_TIMESTAMP, DEFAULT_TRIP_TOKEN_TIMER);
    }

    private boolean isTwoHoursExpired(long then) {
        return isExpired(getNow(), then);
    }

    private long getNow() {
        return System.currentTimeMillis();
    }

    private boolean isExpired(long now, long then) {
        return (now - then) / 1000 > timerExpireAfter;
    }
}
