package com.mapbox.mapboxsdk.plugins.china.shift;

import android.app.Activity;
import android.location.Location;

import com.mapbox.geojson.FeatureCollection;
import com.mapbox.geojson.Point;
import com.mapbox.geojson.Polygon;
import com.mapbox.mapboxsdk.geometry.LatLng;
import com.mapbox.mapboxsdk.log.Logger;
import com.mapbox.turf.TurfJoins;

import java.io.InputStream;
import java.util.Scanner;

/**
 * The class provides methods which check whether the device's real-time
 * location coordinates are inside China's borders.
 */
public class ChinaBoundsChecker {

  /**
   * Determines whether a {@link Location}'s coordinates are within
   * China's borders. The borders are represented by a single {@link Polygon},
   * which represents the area considered part of China.
   *
   * @param activity the current activity that this method is being used in
   * @param locationToQuery the location to use in determining whether it's in or
   *                     outside of China.
   * @return true if the Location is inside of China, false if it's not.
   */
  public static boolean locationIsInChina(Activity activity, Location locationToQuery) {
    return pointIsInChina(activity, Point.fromLngLat(locationToQuery.getLongitude(),
        locationToQuery.getLatitude()));
  }

  /**
   * Determines whether a {@link LatLng}'s coordinates are within
   * China's borders. The borders are represented by a single {@link Polygon},
   * which represents the area considered part of China.
   *
   * @param activity the current activity that this method is being used in
   * @param latLng the coordinate to use in determining whether it's in or
   *                     outside of China.
   * @return true if the LatLng is inside of China, false if it's not.
   */
  public static boolean latLngIsInChina(Activity activity, LatLng latLng) {
    return pointIsInChina(activity, Point.fromLngLat(latLng.getLongitude(),
        latLng.getLatitude()));
  }

  /**
   * Determines whether a {@link Point}'s coordinates are within
   * China's borders. The borders are represented by a single
   * {@link Polygon}, which represents the area considered part of China.
   *
   * @param activity the current activity that this method is being used in
   * @param pointToQuery the coordinate to use in determining whether it's in or
   *                     outside of China.
   * @return true if the Point is inside of China, false if it's not.
   */
  public static boolean pointIsInChina(Activity activity, Point pointToQuery) {
    Scanner scanner = null;
    try {
      InputStream inputStream = activity.getAssets().open("china.geojson");
      scanner = new Scanner(inputStream).useDelimiter("\\A");
      FeatureCollection chinaPolygonFeatureCollection =
          FeatureCollection.fromJson(scanner.hasNext() ? scanner.next() : "");
      if (chinaPolygonFeatureCollection != null) {
        if (chinaPolygonFeatureCollection.features() != null
            && chinaPolygonFeatureCollection.features().size() > 0) {
          if (chinaPolygonFeatureCollection.features().get(0).geometry() != null) {
            Polygon chinaPolygon = (Polygon) chinaPolygonFeatureCollection.features().get(0).geometry();
            if (chinaPolygon != null) {
              return TurfJoins.inside(pointToQuery, chinaPolygon);
            } else {
              Logger.e("ShiftLocation", String.format("Exception Loading GeoJSON: %s",
                String.format("Exception Loading GeoJSON: %s", "the China polygon is null")));
            }
          } else {
            Logger.e("ShiftLocation", String.format("Exception Loading GeoJSON: %s",
                "the China polygon geometry is null"));
          }
        } else {
          Logger.e("ShiftLocation", String.format("Exception Loading GeoJSON: %s",
              "the China polygon FeatureCollection's list of features is null"));
        }
      } else {
        Logger.e("ShiftLocation", String.format("Exception Loading GeoJSON: %s",
            "the China polygon FeatureCollection is null"));
      }
    } catch (Exception exception) {
      Logger.e("Exception Loading GeoJSON: %s", exception.toString());
    } finally {
      if (scanner != null) {
        scanner.close();
      }
    }
    return false;
  }
}
