package com.mapbox.mapboxsdk.plugins.china.constants;

import android.support.annotation.IntDef;
import android.support.annotation.NonNull;
import android.support.annotation.StringDef;

import com.mapbox.mapboxsdk.plugins.china.maps.ChinaMapView;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;

/**
 * <p>
 * {@link ChinaStyle} provides easy access to Chinese map style URLs. The styles included here are
 * professionally designed by the Mapbox team.
 * </p>
 * To load a Chinese map style, pass a style URL into {@link ChinaMapView#setStyleUrl(String)}
 *
 * @see ChinaMapView#setStyleUrl(String)
 * @since 0.4.0
 */
public class ChinaStyle {

  public static final int MAPBOX_CHINA_STYLE_VERSION_1 = 1;

  /**
   * Indicates the which styles URLs are acceptable as method parameters. Using one of these
   * constants means your map style will always use the latest version and may change as we
   * improve the styles.
   *
   * @since 0.4.0
   */
  @StringDef( {MAPBOX_STREETS_CHINESE, MAPBOX_LIGHT_CHINESE, MAPBOX_DARK_CHINESE})
  @Retention(RetentionPolicy.SOURCE)
  public @interface ChinaStyleUrl {
  }

  /**
   * Indicates the which style version numbers are acceptable as method parameters. Using one of these
   * constants enables the retrieval of a certain version of a Chinese map style. Map style design
   * may change as we improve the styles.
   *
   * @since 0.4.7
   */
  @IntDef( {MAPBOX_CHINA_STYLE_VERSION_1})
  @Retention(RetentionPolicy.SOURCE)
  public @interface ChinaStyleVersion {
  }

  private ChinaStyle() {
    // No instances
  }

  // IMPORTANT: If you change any of these you also need to edit them in the `china` module's strings.xml file

  /**
   * A complete basemap, perfect for incorporating your own data. Using this constant means your map
   * style will always use the latest version and may change as we improve
   * the style.
   *
   * @since 0.4.0
   */
  public static final String MAPBOX_STREETS_CHINESE = "mapbox://styles/mapbox/streets-zh-v1";

  /**
   * Subtle light backdrop for data visualizations. Using this constant means your map style will
   * always use the latest version and may change as we improve the style.
   *
   * @since 0.4.0
   */
  public static final String MAPBOX_LIGHT_CHINESE = "mapbox://styles/mapbox/light-zh-v1";

  /**
   * Subtle dark backdrop for data visualizations. Using this constant means your map style will
   * always use the latest version and may change as we improve the style.
   *
   * @since 0.4.0
   */
  public static final String MAPBOX_DARK_CHINESE = "mapbox://styles/mapbox/dark-zh-v1";

  /**
   * Retrieve a certain version of a Chinese map style.
   *
   * @param chineseStyleUrl   the Chinese map style
   * @param desiredVersion the desired version number
   * @return the Chinese map style of a certain version
   * <p>
   * * @since 1.0.0
   */
  public static String specificChinaStyleVersion(@NonNull @ChinaStyleUrl String chineseStyleUrl,
                                                 @ChinaStyleVersion int desiredVersion) {
    StringBuilder stringBuilder = new StringBuilder();
    switch (chineseStyleUrl) {
      case MAPBOX_STREETS_CHINESE:
      case MAPBOX_LIGHT_CHINESE:
      case MAPBOX_DARK_CHINESE:
        stringBuilder.append(chineseStyleUrl);
        stringBuilder.deleteCharAt(chineseStyleUrl.length() - 1);
        stringBuilder.append(String.valueOf(desiredVersion));
        break;
      default:
        stringBuilder.append(MAPBOX_STREETS_CHINESE);
        stringBuilder.deleteCharAt(MAPBOX_STREETS_CHINESE.length() - 1);
        stringBuilder.append(String.valueOf(desiredVersion));
        break;
    }
    return stringBuilder.toString();
  }
}
