package com.mapbox.geocoder.android;

import android.content.Context;
import android.location.Address;

import com.mapbox.geocoder.MapboxGeocoder;
import com.mapbox.geocoder.service.models.GeocoderFeature;
import com.mapbox.geocoder.service.models.GeocoderResponse;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import retrofit.Response;

/**
 * A class for handling geocoding and reverse geocoding.  Geocoding is
 * the process of transforming a street address or other description
 * of a location into a (latitude, longitude) coordinate.  Reverse
 * geocoding is the process of transforming a (latitude, longitude)
 * coordinate into a (partial) address.  The amount of detail in a
 * reverse geocoded location description may vary, for example one
 * might contain the full street address of the closest building, while
 * another might contain only a city name and postal code.
 *
 * The Geocoder class requires a backend service that is not included in
 * the core android framework.  The Geocoder query methods will return an
 * empty list if there no backend service in the platform.  Use the
 * isPresent() method to determine whether a Geocoder implementation
 * exists.
 *
 * Note: We only replicate the API of the original Android Geocoder class.
 * Implementation is our own.
 */
public class AndroidGeocoder {

    private final static String LOG_TAG = "AndroidGeocoder";

    private Context _context;
    private Locale _locale;

    private String _accessToken;

    /**
     * Constructs a Geocoder whose responses will be localized for the
     * given Locale.
     *
     * @param context the Context of the calling Activity
     * @param locale the desired Locale for the query results
     *
     * @throws NullPointerException if Locale is null
     */
    public AndroidGeocoder(Context context, Locale locale) {
        _context = context;
        _locale = locale;
    }

    /**
     * Constructs a Geocoder whose responses will be localized for the
     * default system Locale.
     *
     * @param context the Context of the calling Activity
     */
    public AndroidGeocoder(Context context) {
        _context = context;
    }

    /**
     * Returns true if the Geocoder methods getFromLocation and
     * getFromLocationName are implemented.  Lack of network
     * connectivity may still cause these methods to return null or
     * empty lists.
     */
    public static boolean isPresent() {
        return true;
    }

    /**
     * Returns an array of Addresses that are known to describe the
     * area immediately surrounding the given latitude and longitude.
     * The returned addresses will be localized for the locale
     * provided to this class's constructor.
     *
     * <p> The returned values may be obtained by means of a network lookup.
     * The results are a best guess and are not guaranteed to be meaningful or
     * correct. It may be useful to call this method from a thread separate from your
     * primary UI thread.
     *
     * @param latitude the latitude a point for the search
     * @param longitude the longitude a point for the search
     * @param maxResults max number of addresses to return. Smaller numbers (1 to 5) are recommended
     *
     * @return a list of Address objects. Returns null or empty list if no matches were
     * found or there is no backend service available.
     *
     * @throws IllegalArgumentException if latitude is
     * less than -90 or greater than 90
     * @throws IllegalArgumentException if longitude is
     * less than -180 or greater than 180
     * @throws IOException if the network is unavailable or any other
     * I/O problem occurs
     */
    public List<Address> getFromLocation(double latitude, double longitude, int maxResults)
            throws IOException {
        List<Address> addresses = new ArrayList<>();

        MapboxGeocoder client = new MapboxGeocoder.Builder()
                .setAccessToken(_accessToken)
                .setCoordinates(longitude, latitude)
                .build();

        Response<GeocoderResponse> response = client.execute();
        if (!response.isSuccess()) {
            return addresses;
        }

        List<GeocoderFeature> features = response.body().getFeatures();

        // Trim the list if needed
        if (features.size() > maxResults) {
            features = features.subList(0, maxResults);
        }

        // Convert from FeatureModel to Address
        for (GeocoderFeature feature: features) {
            addresses.add(feature.toAddress(_locale));
        }

        return addresses;
    }

    /**
     * Returns an array of Addresses that are known to describe the
     * named location, which may be a place name such as "Dalvik,
     * Iceland", an address such as "1600 Amphitheatre Parkway,
     * Mountain View, CA", an airport code such as "SFO", etc..  The
     * returned addresses will be localized for the locale provided to
     * this class's constructor.
     *
     * <p> The query will block and returned values will be obtained by means of a network lookup.
     * The results are a best guess and are not guaranteed to be meaningful or
     * correct. It may be useful to call this method from a thread separate from your
     * primary UI thread.
     *
     * @param locationName a user-supplied description of a location
     * @param maxResults max number of results to return. Smaller numbers (1 to 5) are recommended
     *
     * @return a list of Address objects. Returns null or empty list if no matches were
     * found or there is no backend service available.
     *
     * @throws IllegalArgumentException if locationName is null
     * @throws IOException if the network is unavailable or any other
     * I/O problem occurs
     */
    public List<Address> getFromLocationName(String locationName, int maxResults) throws IOException {
        List<Address> addresses = new ArrayList<>();

        MapboxGeocoder client = new MapboxGeocoder.Builder()
                .setAccessToken(_accessToken)
                .setLocation(locationName)
                .build();

        Response<GeocoderResponse> response = client.execute();
        if (!response.isSuccess()) {
            return addresses;
        }

        List<GeocoderFeature> features = response.body().getFeatures();

        // Trim the list if needed
        if (features.size() > maxResults) {
            features = features.subList(0, maxResults);
        }

        // Convert from FeatureModel to Address
        for (GeocoderFeature feature: features) {
            addresses.add(feature.toAddress(_locale));
        }

        return addresses;
    }

    /**
     * Returns an array of Addresses that are known to describe the
     * named location, which may be a place name such as "Dalvik,
     * Iceland", an address such as "1600 Amphitheatre Parkway,
     * Mountain View, CA", an airport code such as "SFO", etc..  The
     * returned addresses will be localized for the locale provided to
     * this class's constructor.
     *
     * <p> You may specify a bounding box for the search results by including
     * the Latitude and Longitude of the Lower Left point and Upper Right
     * point of the box.
     *
     * <p> The query will block and returned values will be obtained by means of a network lookup.
     * The results are a best guess and are not guaranteed to be meaningful or
     * correct. It may be useful to call this method from a thread separate from your
     * primary UI thread.
     *
     * @param locationName a user-supplied description of a location
     * @param maxResults max number of addresses to return. Smaller numbers (1 to 5) are recommended
     * @param lowerLeftLatitude the latitude of the lower left corner of the bounding box
     * @param lowerLeftLongitude the longitude of the lower left corner of the bounding box
     * @param upperRightLatitude the latitude of the upper right corner of the bounding box
     * @param upperRightLongitude the longitude of the upper right corner of the bounding box
     *
     * @return a list of Address objects. Returns null or empty list if no matches were
     * found or there is no backend service available.
     *
     * @throws IllegalArgumentException if locationName is null
     * @throws IllegalArgumentException if any latitude is
     * less than -90 or greater than 90
     * @throws IllegalArgumentException if any longitude is
     * less than -180 or greater than 180
     * @throws IOException if the network is unavailable or any other
     * I/O problem occurs
     */
    public List<Address> getFromLocationName(String locationName, int maxResults,
                                             double lowerLeftLatitude, double lowerLeftLongitude,
                                             double upperRightLatitude, double upperRightLongitude) throws IOException {
        List<Address> addresses = new ArrayList<>();

        // We use the bbox to infer a proximity location
        double proximityLatitude = (lowerLeftLatitude + upperRightLatitude) / 2.0;
        double proximityLongitude = (lowerLeftLongitude + upperRightLongitude) / 2.0;

        MapboxGeocoder client = new MapboxGeocoder.Builder()
                .setAccessToken(_accessToken)
                .setLocation(locationName)
                .setProximity(proximityLongitude, proximityLatitude)
                .build();

        Response<GeocoderResponse> response = client.execute();
        if (!response.isSuccess()) {
            return addresses;
        }

        List<GeocoderFeature> features = response.body().getFeatures();

        // Find results not contained within the bbox
        List<GeocoderFeature> featuresToRemove = new ArrayList<>();
        for (GeocoderFeature feature: features) {
            if (feature.getLatitude() < lowerLeftLatitude) {
                featuresToRemove.add(feature);
            } else if (feature.getLatitude() > upperRightLatitude) {
                featuresToRemove.add(feature);
            } else if (feature.getLongitude() < lowerLeftLongitude) {
                featuresToRemove.add(feature);
            } else if (feature.getLongitude() > upperRightLongitude) {
                featuresToRemove.add(feature);
            }
        }

        // Remove features outside the bbox
        if (featuresToRemove.size() > 0) {
            features.removeAll(featuresToRemove);
        }

        // Trim the list if needed
        if (features.size() > maxResults) {
            features = features.subList(0, maxResults);
        }

        // Convert from FeatureModel to Address
        for (GeocoderFeature feature: features) {
            addresses.add(feature.toAddress(_locale));
        }

        return addresses;
    }

    /*
     * Extended API
     */

    public void setAccessToken(String accessToken) {
        _accessToken = accessToken;
    }

}
