package com.mapbox.android.accounts.v1;

import androidx.annotation.Keep;

import java.security.SecureRandom;
import java.util.Random;
import java.util.UUID;

/**
 * Main entry point to the Mapbox Accounts SDK for Android.
 */
public class MapboxAccounts {

  /**
   * This is the same value as Character.MAX_RADIX but instead of using Character.MAX_RADIX
   * (which could change with new Java releases) we hardcode the current maximum value to
   * guarantee value stability.
   */
  private static final int RADIX = 36;

  private static final String BASE_62_CHARS = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

  /**
   * We're versioning the functionality using package names. This MapboxAccounts only
   * supports v1 of the specification.
   */
  static final String VERSION_1 = "1";

  /**
   * Maps skuId is 2 base-62 characters identifying the SKU.
   */
  @Keep
  public static final String SKU_ID_MAPS_MAUS = "00";

  /**
   * Navigation skuId is 2 base-62 characters identifying the maus SKU.
   */
  @Keep
  public static final String SKU_ID_NAVIGATION_MAUS = "02";

  /**
   * Navigation skuId is 2 base-62 characters identifying the trips SKU.
   */
  @Keep
  public static final String SKU_ID_NAVIGATION_TRIPS = "07";

  /**
   * Vision skuId is 2 base-62 characters identifying the SKU.
   */
  @Keep
  public static final String SKU_ID_VISION_MAUS = "04";

  /**
   * Vision Fleet skuId is 2 base-62 characters identifying the SKU.
   */
  @Keep
  public static final String SKU_ID_VISION_FLEET_MAUS = "06";

  /**
   * Builds a v1 maps SKU end-user token.
   * <p>
   * A SKU token is created by concatenating a set of predefined fields.
   * All fields have constant character counts, except endUserId, which may
   * have different lengths in different clients and so will be the last
   * field of end-user SKU tokens. Because of that rule and the tokenVersion
   * field, we do not need a separator between fields: the tokenVersion will
   * tell the parser which fields to expect and the length of each expected
   * field.
   *
   * @return the maps end-user SKU token
   */
  @Keep
  public static String obtainMapsSkuUserToken(String userId) {
    return join("", new String[]{VERSION_1, SKU_ID_MAPS_MAUS, obtainEncodedTimestamp(), userId});
  }

  /**
   * Builds a v1 navigation SKU end-user token.
   * <p>
   * A SKU token is created by concatenating a set of predefined fields.
   * All fields have constant character counts, except endUserId, which may
   * have different lengths in different clients and so will be the last
   * field of end-user SKU tokens. Because of that rule and the tokenVersion
   * field, we do not need a separator between fields: the tokenVersion will
   * tell the parser which fields to expect and the length of each expected
   * field.
   *
   * @return the navigation end-user SKU token
   */
  @Keep
  public static String obtainNavigationSkuUserToken(String userId) {
    return join("", new String[]{VERSION_1, SKU_ID_NAVIGATION_MAUS, obtainEncodedTimestamp(), userId});
  }

  /**
   * Builds a v1 navigation SKU session token.
   * <p>
   * A SKU token is created by concatenating a set of predefined fields.
   * All fields have constant character counts, except endUserId, which may
   * have different lengths in different clients and so will be the last
   * field of end-user SKU tokens. Because of that rule and the tokenVersion
   * field, we do not need a separator between fields: the tokenVersion will
   * tell the parser which fields to expect and the length of each expected
   * field.
   *
   * @return the navigation end-session SKU token
   */
  @Keep
  public static String obtainNavigationSkuSessionToken() {
    return join("", new String[]{VERSION_1, SKU_ID_NAVIGATION_TRIPS, obtainSessionRandomizer(10)});
  }

  /**
   * Builds a v1 vision SKU end-user token.
   * <p>
   * A SKU token is created by concatenating a set of predefined fields.
   * All fields have constant character counts, except endUserId, which may
   * have different lengths in different clients and so will be the last
   * field of end-user SKU tokens. Because of that rule and the tokenVersion
   * field, we do not need a separator between fields: the tokenVersion will
   * tell the parser which fields to expect and the length of each expected
   * field.
   *
   * @return the vision end-user SKU token
   */
  @Keep
  public static String obtainVisionSkuUserToken(String userId) {
    return join("", new String[]{VERSION_1, SKU_ID_VISION_MAUS, obtainEncodedTimestamp(), userId});
  }

  /**
   * Builds a v1 vision SKU end-user token.
   * <p>
   * A SKU token is created by concatenating a set of predefined fields.
   * All fields have constant character counts, except endUserId, which may
   * have different lengths in different clients and so will be the last
   * field of end-user SKU tokens. Because of that rule and the tokenVersion
   * field, we do not need a separator between fields: the tokenVersion will
   * tell the parser which fields to expect and the length of each expected
   * field.
   *
   * @return the vision end-user SKU token
   */
  @Keep
  public static String obtainVisionFleetSkuUserToken(String userId) {
    return join("", new String[]{VERSION_1, SKU_ID_VISION_FLEET_MAUS, obtainEncodedTimestamp(), userId});
  }

  /**
   * Provides a timestamp, in milliseconds, used to compute the timestamp component.
   * This method can be used by consuming SDKs to keep track of the rotation period.
   *
   * @return a timestamp for a default locale.
   */
  @Keep
  public static long getNow() {
    return System.currentTimeMillis();
  }

  /**
   * timestamp is the last 8 characters of a base-36 encoded Unix timestamp.
   * By default, we return the timestamp for the current time, unless a different
   * timestamp is specified.
   *
   * @return the encoded timestamp
   */
  static String obtainEncodedTimestamp() {
    return obtainEncodedTimestamp(getNow());
  }

  /**
   * timestamp is the last 8 characters of a base-36 encoded Unix timestamp.
   *
   * @return the encoded timestamp
   */
  static String obtainEncodedTimestamp(long timestamp) {
    return baseEncoding(timestamp, RADIX, 8);
  }

  /**
   * It is 10 characters from the base-62 set. It must ensure that session SKU tokens are unique per session per user
   *
   * @param length of the session randomizer
   * @return the unique per session per user session randomizer
   */
  static String obtainSessionRandomizer(int length) {
      char[] symbols = BASE_62_CHARS.toCharArray();
      char[] buffer = new char[length];
      Random random = new SecureRandom();
      for (int index = 0; index < buffer.length; ++index)
          buffer[index] = symbols[random.nextInt(symbols.length)];
      return new String(buffer);
  }

  /**
   * Encodes a long value as a string, ensuring the length of the resulting string.
   *
   * @param number the value to encode, a timestamp
   * @param radix  the radix to use for the new representation
   * @param length the resulting length to enforce
   * @return a string of length characters
   */
  static String baseEncoding(long number, int radix, int length) {
    String encoded = Long.toString(number, radix);
    if (encoded.length() > length) {
      // last `length` characters if longer than `length` characters
      return encoded.substring(encoded.length() - length);
    } else if (encoded.length() < length) {
      return String
          // left padding with spaces for up to length characters
          .format("%1$" + length + "s", encoded)
          // replace space characters with a zero (the first character in the encoding dictionary)
          .replace(" ", "0");
    } else {
      return encoded;
    }
  }

  /**
   * endUserId is a string that is unique per end user.
   *
   * @return the unique endUserId
   */
  @Keep
  public static String obtainEndUserId() {
    return UUID.randomUUID().toString().replace("-", "");
  }

  /**
   * Returns a string containing the tokens joined by delimiters.
   * @param tokens an array objects to be joined. Strings will be formed from
   *     the objects by calling object.toString().
   */
  private static String join(CharSequence delimiter, Object[] tokens) {
      StringBuilder sb = new StringBuilder();
      boolean firstTime = true;
      for (Object token: tokens) {
          if (firstTime) {
              firstTime = false;
          } else {
              sb.append(delimiter);
          }
          sb.append(token);
      }
      return sb.toString();
  }
}
