/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.servlet;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.Meter;
import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Timer;
import com.linkedin.cruisecontrol.servlet.EndPoint;
import com.linkedin.kafka.cruisecontrol.common.KafkaCruiseControlThreadFactory;
import com.linkedin.kafka.cruisecontrol.servlet.handler.async.runnable.OperationFuture;
import com.linkedin.kafka.cruisecontrol.servlet.parameters.ParameterUtils;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.apache.kafka.common.utils.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SessionManager {
    private static final Logger LOG = LoggerFactory.getLogger(SessionManager.class);
    private final int _capacity;
    private final long _sessionExpiryMs;
    private final Map<HttpSession, SessionInfo> _inProgressSessions;
    private final Time _time;
    private final ScheduledExecutorService _sessionCleaner = Executors.newSingleThreadScheduledExecutor(new KafkaCruiseControlThreadFactory("SessionCleaner", true, null));
    private final Timer _sessionLifetimeTimer;
    private final Meter _sessionCreationFailureMeter;
    private final Map<EndPoint, Timer> _successfulRequestExecutionTimer;

    SessionManager(int capacity, long sessionExpiryMs, Time time, MetricRegistry dropwizardMetricRegistry, Map<EndPoint, Timer> successfulRequestExecutionTimer) {
        this._capacity = capacity;
        this._sessionExpiryMs = sessionExpiryMs;
        this._time = time;
        this._inProgressSessions = new HashMap<HttpSession, SessionInfo>();
        this._sessionCleaner.scheduleAtFixedRate(new ExpiredSessionCleaner(), 0L, 5L, TimeUnit.SECONDS);
        this._successfulRequestExecutionTimer = successfulRequestExecutionTimer;
        this._sessionLifetimeTimer = dropwizardMetricRegistry.timer(MetricRegistry.name((String)"SessionManager", (String[])new String[]{"session-lifetime-timer"}));
        this._sessionCreationFailureMeter = dropwizardMetricRegistry.meter(MetricRegistry.name((String)"SessionManager", (String[])new String[]{"session-creation-failure-rate"}));
        dropwizardMetricRegistry.register(MetricRegistry.name((String)"SessionManager", (String[])new String[]{"num-active-sessions"}), (Metric)((Gauge)this._inProgressSessions::size));
    }

    public void close() {
        this._sessionCleaner.shutdownNow();
    }

    synchronized int numSessions() {
        return this._inProgressSessions.size();
    }

    synchronized OperationFuture getAndCreateSessionIfNotExist(HttpServletRequest request, Supplier<OperationFuture> operation, int step) {
        HttpSession session = request.getSession();
        SessionInfo info = this._inProgressSessions.get(session);
        String requestString = SessionManager.toRequestString(request);
        if (info != null) {
            LOG.info("Found existing session {}", (Object)session);
            info.ensureSameRequest(requestString, request.getParameterMap());
            if (step < info.numFutures()) {
                return info.future(step);
            }
            if (step == info.numFutures()) {
                LOG.info("Adding new future to existing session {}.", (Object)session);
                OperationFuture future = operation.get();
                info.addFuture(future);
                return future;
            }
            throw new IllegalArgumentException(String.format("There are %d steps in the session. Cannot add step %d.", info.numFutures(), step));
        }
        if (step > 0) {
            throw new IllegalArgumentException(String.format("There are no step in the session. Cannot add step %d.", step));
        }
        if (this._inProgressSessions.size() >= this._capacity) {
            this._sessionCreationFailureMeter.mark();
            throw new RuntimeException("There are already " + this._inProgressSessions.size() + " active sessions, which has reached the servlet capacity.");
        }
        LOG.info("Created session for {}", (Object)session);
        info = new SessionInfo(requestString, request.getParameterMap(), ParameterUtils.endPoint(request));
        OperationFuture future = operation.get();
        info.addFuture(future);
        this._inProgressSessions.put(session, info);
        return future;
    }

    synchronized <T> T getFuture(HttpServletRequest request) {
        SessionInfo info = this._inProgressSessions.get(request.getSession());
        if (info == null) {
            return null;
        }
        if (!info.requestUrl().equals(SessionManager.toRequestString(request))) {
            throw new IllegalStateException("The session has an ongoing operation " + info.requestUrl() + " while it is trying another operation of " + SessionManager.toRequestString(request));
        }
        return (T)info.lastFuture();
    }

    synchronized void closeSession(HttpServletRequest request, boolean hasError) {
        HttpSession session = request.getSession(false);
        if (session == null) {
            return;
        }
        SessionInfo info = this._inProgressSessions.remove(session);
        if (info != null && info.lastFuture().isDone()) {
            LOG.info("Closing session {}", (Object)session);
            session.invalidate();
            this._sessionLifetimeTimer.update(System.nanoTime() - info.requestStartTimeNs(), TimeUnit.NANOSECONDS);
            if (!hasError && info.executionTime() > 0L) {
                this._successfulRequestExecutionTimer.get(info.endPoint()).update(info.executionTime(), TimeUnit.NANOSECONDS);
            }
        }
    }

    synchronized void expireOldSessions() {
        long now = this._time.milliseconds();
        Iterator<Map.Entry<HttpSession, SessionInfo>> iter = this._inProgressSessions.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<HttpSession, SessionInfo> entry = iter.next();
            HttpSession session = entry.getKey();
            SessionInfo info = entry.getValue();
            if (LOG.isTraceEnabled()) {
                LOG.trace("Session {} was last accessed at {}, age is {} ms", new Object[]{session, session.getLastAccessedTime(), now - session.getLastAccessedTime()});
            }
            if (now < session.getLastAccessedTime() + this._sessionExpiryMs) continue;
            LOG.info("Expiring session {}.", (Object)session);
            iter.remove();
            session.invalidate();
            this._sessionLifetimeTimer.update(System.nanoTime() - info.requestStartTimeNs(), TimeUnit.NANOSECONDS);
            if (info.lastFuture().isDone() && info.executionTime() > 0L) {
                this._successfulRequestExecutionTimer.get(info.endPoint()).update(info.executionTime(), TimeUnit.NANOSECONDS);
                continue;
            }
            info.lastFuture().cancel(true);
        }
    }

    private static String toRequestString(HttpServletRequest request) {
        return String.format("%s(%s %s)", request.getClass().getSimpleName(), request.getMethod(), request.getRequestURI());
    }

    private class ExpiredSessionCleaner
    implements Runnable {
        private ExpiredSessionCleaner() {
        }

        @Override
        public void run() {
            try {
                SessionManager.this.expireOldSessions();
            }
            catch (Throwable t) {
                LOG.warn("Received exception when trying to expire sessions.", t);
            }
        }
    }

    private static class SessionInfo {
        private final String _requestUrl;
        private final Map<String, String[]> _requestParameters;
        private final List<OperationFuture> _operationFuture = new ArrayList<OperationFuture>();
        private final long _requestStartTimeNs;
        private final EndPoint _endPoint;

        private SessionInfo(String requestUrl, Map<String, String[]> requestParameters, EndPoint endPoint) {
            this._requestUrl = requestUrl;
            this._requestParameters = requestParameters;
            this._requestStartTimeNs = System.nanoTime();
            this._endPoint = endPoint;
        }

        private int numFutures() {
            return this._operationFuture.size();
        }

        private void addFuture(OperationFuture future) {
            this._operationFuture.add(future);
        }

        private OperationFuture future(int index) {
            return this._operationFuture.get(index);
        }

        private OperationFuture lastFuture() {
            return this._operationFuture.get(this._operationFuture.size() - 1);
        }

        private String requestUrl() {
            return this._requestUrl;
        }

        private long requestStartTimeNs() {
            return this._requestStartTimeNs;
        }

        private long executionTime() {
            return this.lastFuture().finishTimeNs() == -1L ? -1L : this.lastFuture().finishTimeNs() - this._requestStartTimeNs;
        }

        private EndPoint endPoint() {
            return this._endPoint;
        }

        private boolean paramEquals(Map<String, String[]> parameters) {
            boolean isSameParameters = this._requestParameters.keySet().equals(parameters.keySet());
            if (isSameParameters) {
                for (Map.Entry<String, String[]> entry : this._requestParameters.entrySet()) {
                    HashSet<String> param2;
                    HashSet<String> param1 = new HashSet<String>(Arrays.asList(entry.getValue()));
                    if (param1.equals(param2 = new HashSet<String>(Arrays.asList(parameters.get(entry.getKey()))))) continue;
                    return false;
                }
            }
            return isSameParameters;
        }

        private void ensureSameRequest(String requestUrl, Map<String, String[]> parameters) {
            if (!this._requestUrl.equals(requestUrl) || !this.paramEquals(parameters)) {
                throw new IllegalStateException(String.format("The session has an ongoing operation [URL: %s, Parameters: %s] while it is trying another operation of [URL: %s, Parameters: %s].", this._requestUrl, this._requestParameters, requestUrl, parameters));
            }
        }
    }
}

