/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.model;

import com.linkedin.cruisecontrol.monitor.sampling.aggregator.AggregatedMetricValues;
import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.config.BrokerCapacityInfo;
import com.linkedin.kafka.cruisecontrol.model.Disk;
import com.linkedin.kafka.cruisecontrol.model.DiskStats;
import com.linkedin.kafka.cruisecontrol.model.Host;
import com.linkedin.kafka.cruisecontrol.model.Load;
import com.linkedin.kafka.cruisecontrol.model.Rack;
import com.linkedin.kafka.cruisecontrol.model.Replica;
import com.linkedin.kafka.cruisecontrol.model.SortedReplicas;
import java.io.IOException;
import java.io.OutputStream;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Function;
import org.apache.kafka.common.TopicPartition;

public class Broker
implements Serializable,
Comparable<Broker> {
    private static final double DEAD_BROKER_CAPACITY = -1.0;
    private final int _id;
    private final Host _host;
    private final double[] _brokerCapacity;
    private final Set<Replica> _replicas;
    private final Set<Replica> _leaderReplicas;
    private final Map<String, SortedReplicas> _sortedReplicas;
    private final Set<Replica> _immigrantReplicas;
    private final Set<Replica> _currentOfflineReplicas;
    private final Map<String, Map<Integer, Replica>> _topicReplicas;
    private final Load _load;
    private final Load _leadershipLoadForNwResources;
    private final SortedMap<String, Disk> _diskByLogdir;
    private State _state;

    Broker(Host host, int id, BrokerCapacityInfo brokerCapacityInfo, boolean populateReplicaPlacementInfo) {
        Map<Resource, Double> brokerCapacity = brokerCapacityInfo.capacity();
        if (brokerCapacity == null) {
            throw new IllegalArgumentException("Attempt to create broker " + id + " on host " + host.name() + " with null capacity.");
        }
        this._host = host;
        this._id = id;
        this._brokerCapacity = new double[Resource.cachedValues().size()];
        for (Map.Entry<Resource, Double> entry : brokerCapacity.entrySet()) {
            Resource resource = entry.getKey();
            this._brokerCapacity[resource.id()] = resource == Resource.CPU ? entry.getValue() * (double)brokerCapacityInfo.numCpuCores() : entry.getValue();
        }
        if (populateReplicaPlacementInfo) {
            this._diskByLogdir = new TreeMap<String, Disk>();
            brokerCapacityInfo.diskCapacityByLogDir().forEach((key, value) -> this._diskByLogdir.put((String)key, new Disk((String)key, this, (double)value)));
        } else {
            this._diskByLogdir = Collections.emptySortedMap();
        }
        this._replicas = new HashSet<Replica>();
        this._leaderReplicas = new HashSet<Replica>();
        this._topicReplicas = new HashMap<String, Map<Integer, Replica>>();
        this._sortedReplicas = new HashMap<String, SortedReplicas>();
        this._immigrantReplicas = new HashSet<Replica>();
        this._currentOfflineReplicas = new HashSet<Replica>();
        this._load = new Load();
        this._leadershipLoadForNwResources = new Load();
        this._state = State.ALIVE;
    }

    public Host host() {
        return this._host;
    }

    public State state() {
        return this._state;
    }

    public Rack rack() {
        return this._host.rack();
    }

    public int id() {
        return this._id;
    }

    public double capacityFor(Resource resource) {
        return this._brokerCapacity[resource.id()];
    }

    public Set<Replica> replicas() {
        return Collections.unmodifiableSet(this._replicas);
    }

    public Set<Replica> leaderReplicas() {
        return Collections.unmodifiableSet(this._leaderReplicas);
    }

    public Set<Replica> immigrantReplicas() {
        return Collections.unmodifiableSet(this._immigrantReplicas);
    }

    public Set<Replica> currentOfflineReplicas() {
        return this._currentOfflineReplicas;
    }

    public Replica replica(TopicPartition tp) {
        Map<Integer, Replica> topicReplicas = this._topicReplicas.get(tp.topic());
        if (topicReplicas == null) {
            return null;
        }
        return topicReplicas.get(tp.partition());
    }

    public Collection<Replica> replicasOfTopicInBroker(String topic) {
        Map<Integer, Replica> topicReplicas = this._topicReplicas.get(topic);
        return topicReplicas == null ? Collections.emptySet() : topicReplicas.values();
    }

    public int numReplicasOfTopicInBroker(String topic) {
        Map<Integer, Replica> topicReplicas = this._topicReplicas.get(topic);
        return topicReplicas == null ? 0 : topicReplicas.size();
    }

    public boolean isAlive() {
        return this._state != State.DEAD;
    }

    public boolean isUsingJBOD() {
        return !this._diskByLogdir.isEmpty();
    }

    public boolean isNew() {
        return this._state == State.NEW;
    }

    public boolean isDemoted() {
        return this._state == State.DEMOTED;
    }

    public boolean hasBadDisks() {
        return this._state == State.BAD_DISKS;
    }

    public Load load() {
        return this._load;
    }

    public Load leadershipLoadForNwResources() {
        return this._leadershipLoadForNwResources;
    }

    public Set<String> topics() {
        return this._topicReplicas.keySet();
    }

    public SortedReplicas trackedSortedReplicas(String sortName) {
        SortedReplicas sortedReplicas = this._sortedReplicas.get(sortName);
        if (sortedReplicas == null) {
            throw new IllegalStateException("The sort name " + sortName + "  is not found. Make sure trackSortedReplicas() has been called for the sort name");
        }
        return sortedReplicas;
    }

    public Comparator<Replica> replicaComparator() {
        return (r1, r2) -> {
            int result;
            boolean isR1Offline = this._currentOfflineReplicas.contains(r1);
            boolean isR2Offline = this._currentOfflineReplicas.contains(r2);
            if (isR1Offline && !isR2Offline) {
                return -1;
            }
            if (!isR1Offline && isR2Offline) {
                return 1;
            }
            boolean isR1Immigrant = this._immigrantReplicas.contains(r1);
            boolean isR2Immigrant = this._immigrantReplicas.contains(r2);
            int n = isR1Immigrant && !isR2Immigrant ? -1 : (result = !isR1Immigrant && isR2Immigrant ? 1 : 0);
            if (result == 0) {
                if (r1.topicPartition().partition() > r2.topicPartition().partition()) {
                    return 1;
                }
                if (r1.topicPartition().partition() < r2.topicPartition().partition()) {
                    return -1;
                }
            }
            return result;
        };
    }

    private double loadDensity(Replica replica, Resource resource) {
        double expectedLoad = replica.load().expectedUtilizationFor(resource);
        if (expectedLoad == 0.0) {
            return 0.0;
        }
        if (resource == Resource.DISK) {
            return expectedLoad;
        }
        double diskLoad = replica.load().expectedUtilizationFor(Resource.DISK);
        if (diskLoad == 0.0) {
            return 1000000.0;
        }
        return expectedLoad / diskLoad;
    }

    void setState(State newState) {
        this._state = newState;
        if (!this.isAlive()) {
            this._currentOfflineReplicas.addAll(this.replicas());
            this._diskByLogdir.values().forEach(d -> d.setState(Disk.State.DEAD));
            Resource.cachedValues().forEach(r -> {
                this._brokerCapacity[r.id()] = -1.0;
            });
        }
    }

    void addReplica(Replica replica) {
        if (this._replicas.contains(replica)) {
            throw new IllegalStateException(String.format("Broker %d already has replica %s", this._id, replica.topicPartition()));
        }
        this._replicas.add(replica);
        if (replica.originalBroker().id() != this._id) {
            this._immigrantReplicas.add(replica);
        } else if (replica.isOriginalOffline()) {
            this._currentOfflineReplicas.add(replica);
        }
        this._topicReplicas.computeIfAbsent(replica.topicPartition().topic(), t -> new HashMap()).put(replica.topicPartition().partition(), replica);
        if (replica.isLeader()) {
            this._leadershipLoadForNwResources.addLoad(replica.load());
            this._leaderReplicas.add(replica);
        }
        this._load.addLoad(replica.load());
        this._sortedReplicas.values().forEach(sr -> sr.add(replica));
        if (replica.disk() != null) {
            ((Disk)this._diskByLogdir.get(replica.disk().logDir())).addReplica(replica);
        }
    }

    Disk addDeadDisk(String logdir) {
        Disk disk = new Disk(logdir, this, -1.0);
        this._diskByLogdir.put(logdir, disk);
        return disk;
    }

    void trackSortedReplicas(String sortName, Set<Function<Replica, Boolean>> selectionFuncs, List<Function<Replica, Integer>> priorityFuncs, Function<Replica, Double> scoreFunc) {
        this._sortedReplicas.putIfAbsent(sortName, new SortedReplicas(this, selectionFuncs, priorityFuncs, scoreFunc));
        for (Disk disk : this._diskByLogdir.values()) {
            disk.trackSortedReplicas(sortName, selectionFuncs, priorityFuncs, scoreFunc);
        }
    }

    public void untrackSortedReplicas(String sortName) {
        this._sortedReplicas.remove(sortName);
        for (Disk disk : this._diskByLogdir.values()) {
            disk.untrackSortedReplicas(sortName);
        }
    }

    public void clearSortedReplicas() {
        this._sortedReplicas.clear();
        for (Disk disk : this._diskByLogdir.values()) {
            disk.clearSortedReplicas();
        }
    }

    private void updateSortedReplicas(Replica replica) {
        this._sortedReplicas.values().forEach(sr -> {
            sr.remove(replica);
            sr.add(replica);
        });
    }

    AggregatedMetricValues makeFollower(TopicPartition tp) {
        Replica replica = this.replica(tp);
        this._leadershipLoadForNwResources.subtractLoad(replica.load());
        AggregatedMetricValues leadershipLoadDelta = replica.makeFollower();
        this._load.subtractLoad(leadershipLoadDelta);
        this._leaderReplicas.remove(replica);
        this.updateSortedReplicas(replica);
        return leadershipLoadDelta;
    }

    void makeLeader(TopicPartition tp, AggregatedMetricValues leadershipLoadDelta) {
        Replica replica = this.replica(tp);
        replica.makeLeader(leadershipLoadDelta);
        this._leadershipLoadForNwResources.addLoad(replica.load());
        this._load.addLoad(leadershipLoadDelta);
        this._leaderReplicas.add(replica);
        this.updateSortedReplicas(replica);
    }

    Replica removeReplica(TopicPartition tp) {
        Replica removedReplica = this.replica(tp);
        if (removedReplica != null) {
            this._replicas.remove(removedReplica);
            this._load.subtractLoad(removedReplica.load());
            Map<Integer, Replica> topicReplicas = this._topicReplicas.get(tp.topic());
            if (topicReplicas != null) {
                topicReplicas.remove(tp.partition());
            }
            if (removedReplica.isLeader()) {
                this._leadershipLoadForNwResources.subtractLoad(removedReplica.load());
                this._leaderReplicas.remove(removedReplica);
            }
            this._immigrantReplicas.remove(removedReplica);
            this._currentOfflineReplicas.remove(removedReplica);
            this._sortedReplicas.values().forEach(sr -> sr.remove(removedReplica));
        }
        return removedReplica;
    }

    void moveReplicaBetweenDisks(TopicPartition tp, String sourceLogdir, String destinationLogdir) {
        Replica replica = this.replica(tp);
        ((Disk)this._diskByLogdir.get(sourceLogdir)).removeReplica(replica);
        ((Disk)this._diskByLogdir.get(destinationLogdir)).addReplica(replica);
    }

    double markDiskDead(String logdir) {
        Disk disk = (Disk)this._diskByLogdir.get(logdir);
        double diskCapacity = disk.capacity();
        int n = Resource.DISK.id();
        this._brokerCapacity[n] = this._brokerCapacity[n] - diskCapacity;
        disk.setState(Disk.State.DEAD);
        disk.replicas().forEach(Replica::markOriginalOffline);
        return diskCapacity;
    }

    void clearLoad() {
        this._replicas.forEach(Replica::clearLoad);
    }

    void clearReplicas() {
        this._replicas.clear();
        this._leaderReplicas.clear();
        this._topicReplicas.clear();
        this._immigrantReplicas.clear();
        this._currentOfflineReplicas.clear();
        this._load.clearLoad();
        this._leadershipLoadForNwResources.clearLoad();
    }

    void setReplicaLoad(TopicPartition tp, AggregatedMetricValues aggregatedMetricValues, List<Long> windows) {
        Replica replica = this.replica(tp);
        replica.setMetricValues(aggregatedMetricValues, windows);
        if (replica.disk() != null) {
            replica.disk().addReplicaLoad(replica);
        }
        if (replica.isLeader()) {
            this._leadershipLoadForNwResources.addMetricValues(aggregatedMetricValues, windows);
        }
        this._load.addMetricValues(aggregatedMetricValues, windows);
    }

    public Disk disk(String logdir) {
        return (Disk)this._diskByLogdir.get(logdir);
    }

    public Collection<Disk> disks() {
        return this._diskByLogdir.values();
    }

    public Map<String, Object> getJsonStructure() {
        ArrayList<Map<String, Object>> replicaList = new ArrayList<Map<String, Object>>();
        for (Replica replica : this._replicas) {
            replicaList.add(replica.getJsonStructure());
        }
        HashMap<String, Object> brokerMap = new HashMap<String, Object>(3);
        brokerMap.put("brokerid", this._id);
        brokerMap.put("brokerstate", (Object)this._state);
        brokerMap.put("replicas", replicaList);
        return brokerMap;
    }

    public Map<String, DiskStats> diskStats() {
        if (this._diskByLogdir.isEmpty()) {
            return Collections.emptyMap();
        }
        HashMap<String, DiskStats> diskStatMap = new HashMap<String, DiskStats>(this._diskByLogdir.size());
        this._diskByLogdir.forEach((k, v) -> diskStatMap.put((String)k, v.diskStats()));
        return diskStatMap;
    }

    public void writeTo(OutputStream out) throws IOException {
        String broker = String.format("<Broker id=\"%d\" state=\"%s\">%n", new Object[]{this._id, this._state});
        out.write(broker.getBytes(StandardCharsets.UTF_8));
        for (Disk disk : this._diskByLogdir.values()) {
            disk.writeTo(out);
        }
        if (this._diskByLogdir.isEmpty()) {
            for (Replica replica : this._replicas) {
                replica.writeTo(out);
            }
        }
        out.write("</Broker>%n".getBytes(StandardCharsets.UTF_8));
    }

    public String toString() {
        return String.format("Broker[id=%d,rack=%s,state=%s,replicaCount=%d,logdirs=%s]", new Object[]{this._id, this.rack().id(), this._state, this._replicas.size(), this._diskByLogdir.keySet()});
    }

    @Override
    public int compareTo(Broker o) {
        return Integer.compare(this._id, o.id());
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        Broker broker = (Broker)o;
        return this._id == broker._id;
    }

    public int hashCode() {
        return this._id;
    }

    public static enum State {
        ALIVE,
        DEAD,
        NEW,
        DEMOTED,
        BAD_DISKS;

    }
}

