/*
 * Decompiled with CFR 0.152.
 */
package com.linkedin.kafka.cruisecontrol.analyzer.goals;

import com.linkedin.kafka.cruisecontrol.analyzer.ActionType;
import com.linkedin.kafka.cruisecontrol.analyzer.OptimizationOptions;
import com.linkedin.kafka.cruisecontrol.analyzer.goals.Goal;
import com.linkedin.kafka.cruisecontrol.common.Resource;
import com.linkedin.kafka.cruisecontrol.exception.OptimizationFailureException;
import com.linkedin.kafka.cruisecontrol.model.Broker;
import com.linkedin.kafka.cruisecontrol.model.ClusterModel;
import com.linkedin.kafka.cruisecontrol.model.ClusterModelStats;
import com.linkedin.kafka.cruisecontrol.model.Disk;
import com.linkedin.kafka.cruisecontrol.model.Replica;
import com.linkedin.kafka.cruisecontrol.servlet.CruiseControlEndPoint;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;
import java.util.stream.Collectors;

public class GoalUtils {
    public static final int MIN_NUM_VALID_WINDOWS_FOR_SELF_HEALING = 1;
    public static final int DENOMINATOR_FOR_MIN_VALID_WINDOWS_FOR_SELF_HEALING = 14;
    private static final double DEAD_BROKER_UTILIZATION = 1.0;
    private static final double DEAD_DISK_UTILIZATION = 1.0;

    private GoalUtils() {
    }

    public static void filterOutBrokersExcludedForLeadership(List<Broker> originalBrokers, OptimizationOptions optimizationOptions, Replica replica, ActionType action) {
        Set<Integer> requestedDestinationBrokerIds = optimizationOptions.requestedDestinationBrokerIds();
        if (requestedDestinationBrokerIds.isEmpty() || action == ActionType.LEADERSHIP_MOVEMENT) {
            Set<Integer> excludedBrokers = optimizationOptions.excludedBrokersForLeadership();
            if (!excludedBrokers.isEmpty() && (action == ActionType.LEADERSHIP_MOVEMENT || replica.isLeader())) {
                originalBrokers.removeIf(broker -> excludedBrokers.contains(broker.id()));
            }
        } else {
            originalBrokers.removeIf(b -> !requestedDestinationBrokerIds.contains(b.id()));
        }
    }

    public static void filterOutBrokersExcludedForReplicaMove(List<Broker> originalBrokers, OptimizationOptions optimizationOptions, ActionType action) {
        Set<Integer> requestedDestinationBrokerIds = optimizationOptions.requestedDestinationBrokerIds();
        if (requestedDestinationBrokerIds.isEmpty()) {
            Set<Integer> excludedBrokers = optimizationOptions.excludedBrokersForReplicaMove();
            if (!excludedBrokers.isEmpty() && action == ActionType.INTER_BROKER_REPLICA_MOVEMENT) {
                originalBrokers.removeIf(broker -> excludedBrokers.contains(broker.id()));
            }
        } else if (action != ActionType.LEADERSHIP_MOVEMENT) {
            originalBrokers.removeIf(b -> !requestedDestinationBrokerIds.contains(b.id()));
        }
    }

    public static List<Broker> eligibleBrokers(ClusterModel clusterModel, Replica replica, Collection<Broker> candidates, ActionType action, OptimizationOptions optimizationOptions) {
        ArrayList<Broker> eligibleBrokers = new ArrayList<Broker>(candidates);
        GoalUtils.filterOutBrokersExcludedForLeadership(eligibleBrokers, optimizationOptions, replica, action);
        GoalUtils.filterOutBrokersExcludedForReplicaMove(eligibleBrokers, optimizationOptions, action);
        if (!optimizationOptions.requestedDestinationBrokerIds().isEmpty()) {
            return eligibleBrokers;
        }
        if (clusterModel.newBrokers().isEmpty()) {
            return eligibleBrokers;
        }
        return eligibleBrokers.stream().filter(b -> b.isNew() || b == replica.originalBroker()).collect(Collectors.toList());
    }

    public static boolean legitMove(Replica replica, Broker destinationBroker, ClusterModel clusterModel, ActionType actionType) {
        switch (actionType) {
            case INTER_BROKER_REPLICA_MOVEMENT: {
                return clusterModel.partition(replica.topicPartition()).canAssignReplicaToBroker(destinationBroker) && destinationBroker.replica(replica.topicPartition()) == null;
            }
            case LEADERSHIP_MOVEMENT: {
                return replica.isLeader() && destinationBroker.replica(replica.topicPartition()) != null;
            }
        }
        return false;
    }

    public static boolean legitMoveBetweenDisks(Replica replica, Disk destinationDisk, ActionType actionType) {
        return actionType == ActionType.INTRA_BROKER_REPLICA_MOVEMENT && destinationDisk != null && destinationDisk.broker() == replica.broker() && destinationDisk.isAlive();
    }

    public static SortedSet<Replica> eligibleReplicasForSwap(ClusterModel clusterModel, Replica sourceReplica, SortedSet<Replica> candidateReplicas, OptimizationOptions optimizationOptions) {
        if (candidateReplicas.isEmpty()) {
            return candidateReplicas;
        }
        Broker destinationBroker = candidateReplicas.first().broker();
        if (optimizationOptions.excludedBrokersForLeadership().contains(destinationBroker.id()) && !sourceReplica.isOriginalOffline() && sourceReplica.isLeader()) {
            return Collections.emptySortedSet();
        }
        if (optimizationOptions.excludedBrokersForReplicaMove().contains(destinationBroker.id()) && !sourceReplica.isOriginalOffline()) {
            return Collections.emptySortedSet();
        }
        Broker sourceBroker = sourceReplica.broker();
        if (clusterModel.newBrokers().isEmpty() || sourceBroker.isNew() && (destinationBroker.isNew() || sourceReplica.originalBroker() == destinationBroker)) {
            return candidateReplicas;
        }
        if (destinationBroker.isNew()) {
            candidateReplicas.removeIf(replica -> replica.originalBroker() != sourceBroker);
            return candidateReplicas;
        }
        return Collections.emptySortedSet();
    }

    public static void ensureNoOfflineReplicas(ClusterModel clusterModel, String goalName) throws OptimizationFailureException {
        for (Replica replica : clusterModel.selfHealingEligibleReplicas()) {
            if (!replica.isCurrentOffline()) continue;
            throw new OptimizationFailureException(String.format("[%s] Self healing failed to move the replica %s from %s broker %d (contains %d replicas).", new Object[]{goalName, replica, replica.broker().state(), replica.broker().id(), replica.broker().replicas().size()}));
        }
    }

    public static void ensureReplicasMoveOffBrokersWithBadDisks(ClusterModel clusterModel, String goalName) throws OptimizationFailureException {
        for (Broker broker : clusterModel.brokersWithBadDisks()) {
            for (Replica replica : broker.replicas()) {
                if (clusterModel.partition(replica.topicPartition()).canAssignReplicaToBroker(broker)) continue;
                throw new OptimizationFailureException(String.format("[%s] A replica of partition %s has been moved back to broker %d, where it was originally hosted on a broken disk.", goalName, clusterModel.partition(replica.topicPartition()), replica.broker().id()));
            }
        }
    }

    public static Set<Replica> filterLeaders(Broker broker, boolean immigrantsOnly) {
        Set<Replica> filteredLeaders;
        if (immigrantsOnly) {
            filteredLeaders = new HashSet<Replica>(broker.immigrantReplicas());
            filteredLeaders.removeIf(replica -> !replica.isLeader());
        } else {
            filteredLeaders = broker.leaderReplicas();
        }
        return filteredLeaders;
    }

    public static double utilizationPercentage(Broker broker, Resource resource) {
        double brokerCapacity = broker.capacityFor(resource);
        return brokerCapacity > 0.0 ? broker.load().expectedUtilizationFor(resource) / brokerCapacity : 1.0;
    }

    public static double averageDiskUtilizationPercentage(Broker broker) {
        double totalAliveDiskCapacity = 0.0;
        double totalAliveDiskUtilization = 0.0;
        for (Disk disk : broker.disks()) {
            if (!disk.isAlive()) continue;
            totalAliveDiskCapacity += disk.capacity();
            totalAliveDiskUtilization += disk.utilization();
        }
        return totalAliveDiskCapacity > 0.0 ? totalAliveDiskUtilization / totalAliveDiskCapacity : 1.0;
    }

    public static double diskUtilizationPercentage(Disk disk) {
        double diskCapacity = disk.capacity();
        return diskCapacity > 0.0 ? disk.utilization() / diskCapacity : 1.0;
    }

    public static void sortReplicasInAscendingOrderByBrokerResourceUtilization(List<Replica> replicas, Resource resource) {
        replicas.sort((r1, r2) -> {
            double expectedBrokerLoad2;
            double expectedBrokerLoad1 = r1.broker().load().expectedUtilizationFor(resource);
            int result = Double.compare(expectedBrokerLoad1, expectedBrokerLoad2 = r2.broker().load().expectedUtilizationFor(resource));
            return result == 0 ? Integer.compare(r1.broker().id(), r2.broker().id()) : result;
        });
    }

    public static String replicaSortName(Goal goal, boolean reverse, boolean leaderOnly) {
        return String.format("%s%s%s", goal.name(), reverse ? "-REVERSE" : "", leaderOnly ? "-LEADER" : "");
    }

    public static String mitigationForOptimizationFailures(OptimizationOptions optimizationOptions) {
        StringBuilder sb = new StringBuilder();
        if (optimizationOptions.onlyMoveImmigrantReplicas()) {
            sb.append(String.format("The optimization is limited to replicas to be added to/removed from brokers. Potential mitigation: First, rebalance the cluster using %s endpoint with a a superset of hard-goals defined via %s config.%n", new Object[]{CruiseControlEndPoint.REBALANCE, "hard.goals"}));
        }
        if (!optimizationOptions.requestedDestinationBrokerIds().isEmpty()) {
            sb.append(String.format("The destination brokers are limited to %s. Potential mitigation: Relax the constraint on destination brokers using %s parameter.%n", optimizationOptions.requestedDestinationBrokerIds(), "destination_broker_ids"));
        }
        if (!optimizationOptions.excludedBrokersForReplicaMove().isEmpty()) {
            sb.append(String.format("The following brokers are excluded from replica moves %s. Potential mitigation: Drop brokers from exclusion for replica move using %s endpoint.%n", new Object[]{optimizationOptions.excludedBrokersForReplicaMove(), CruiseControlEndPoint.ADMIN}));
        }
        if (!optimizationOptions.excludedBrokersForLeadership().isEmpty()) {
            sb.append(String.format("The following brokers are excluded from leadership moves %s. Potential mitigation: Drop brokers from exclusion for leadership move using %s endpoint.%n", new Object[]{optimizationOptions.excludedBrokersForReplicaMove(), CruiseControlEndPoint.ADMIN}));
        }
        if (!optimizationOptions.excludedTopics().isEmpty()) {
            sb.append(String.format("There are %d topics excluded from replica move. Potential mitigation: Remove selected topics from exclusion using %s parameter.%n", optimizationOptions.excludedTopics().size(), "excluded_topics"));
        }
        if (sb.length() > 0) {
            sb.append(String.format("Then, re-run your original request.%n", new Object[0]));
        }
        return sb.toString();
    }

    public static class HardGoalStatsComparator
    implements Goal.ClusterModelStatsComparator {
        @Override
        public int compare(ClusterModelStats stats1, ClusterModelStats stats2) {
            return 0;
        }

        @Override
        public String explainLastComparison() {
            return null;
        }
    }
}

