/*
 * Copyright (C) 2020 Link Development
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.linkdev.fileattachmentview.view.adapter

import android.content.Context
import android.util.Size
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import androidx.recyclerview.widget.RecyclerView
import com.linkdev.fileattachmentview.models.AttachFileData
import com.linkdev.fileattachmentview.models.FileExtensionsData
import com.linkdev.fileattachmentview.utils.Utils.getCorrespondingAttachmentTypeIcon
import com.linkdev.fileattachmentview.utils.Utils.isValidResource
import com.linkdev.fileattachmentview.view.FileAttachmentView.Companion.DEFAULT_ATTACHMENT_THUMBNAIL_SIZE

/**
 * recycler view adapter for the multi selected files
 * [context] refers to the current context
 * [fileDataList] array list of the selected files data [AttachFileData]
 * [adapterLayout] reference to the layout which represent the file data item
 * [deleteMultipleSelectionItemView] reference to the action view used to delete one attached file
 * [multipleSelectionItemNameView] reference to the view which display the attached file name
 * [multipleSelectionItemThumbnail] reference to th view which display the attached image thumbnail or the file extension icon if needed
 * [multipleSelectionErrorView] reference to the error view
 * [isDisplayImageThumbnail] boolean indicates if the consumer need to display the attached file thumbnail or not in case of the attached files are images
 * [fileExtensionsDataList] array list of available extensions data
 * [imageThumbnailWidth] width of the Thumbnail
 * [imageThumbnailHeight] height of the thumbnail
 * [onFileClickListener] reference to [OnFileClickListener]
 */
class MultipleFileRecyclerViewAdapter(
    private val context: Context,
    private val fileDataList: ArrayList<AttachFileData>,
    private val adapterLayout: Int,
    private val deleteMultipleSelectionItemView: Int?,
    private val multipleSelectionItemNameView: Int?,
    private val multipleSelectionItemThumbnail: Int?,
    private val multipleSelectionErrorView: Int?,
    private val isDisplayImageThumbnail: Boolean = false,
    private val fileExtensionsDataList: ArrayList<FileExtensionsData>? = null,
    private val imageThumbnailWidth: Int = DEFAULT_ATTACHMENT_THUMBNAIL_SIZE,
    private val imageThumbnailHeight: Int = DEFAULT_ATTACHMENT_THUMBNAIL_SIZE,
    private val onFileClickListener: OnFileClickListener? = null

) : RecyclerView.Adapter<MultipleFileRecyclerViewAdapter.MultipleFileViewHolder>() {


    /**
     * callBack method used to create the view Holder of the recycler view
     */
    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): MultipleFileViewHolder {
        val view = LayoutInflater.from(context).inflate(adapterLayout, parent, false)
        return MultipleFileViewHolder(view)
    }

    /**
     * callBack method to bind the current file data item in the view
     */
    override fun onBindViewHolder(holder: MultipleFileViewHolder, position: Int) {
        holder.onBind(fileDataList[position])
    }

    /**
     * callBack method return size of items
     */
    override fun getItemCount(): Int {
        return fileDataList.size
    }

    /**
     * delete file data
     */
    fun deleteFile(fileData: AttachFileData) {
        fileDataList.remove(fileData)
        notifyDataSetChanged()
    }

    inner class MultipleFileViewHolder(itemView: View) : RecyclerView.ViewHolder(itemView) {
        private var imgThumbnail: ImageView? = null
        private var txtFileName: TextView? = null
        private var viewDeleteFile: View? = null
        private var viewError: View? = null

        init {
            if (isValidResource(multipleSelectionItemThumbnail))
                imgThumbnail = itemView.findViewById(multipleSelectionItemThumbnail!!)
            if (isValidResource(multipleSelectionItemNameView))
                txtFileName = itemView.findViewById(multipleSelectionItemNameView!!)
            if (isValidResource(deleteMultipleSelectionItemView))
                viewDeleteFile = itemView.findViewById(deleteMultipleSelectionItemView!!)
            if (isValidResource(multipleSelectionErrorView))
                viewError = itemView.findViewById(multipleSelectionErrorView!!)
        }

        /**
         * bind data of each file to the views
         */
        fun onBind(fileData: AttachFileData) {
            txtFileName?.text = fileData.fileName
            if (isDisplayImageThumbnail) {
                val thumbnail = fileData.convertAttachFileDataToFileData().getThumbnail(
                    context,
                    Size(imageThumbnailWidth, imageThumbnailHeight)
                )
                thumbnail?.let { imgThumbnail?.setImageBitmap(it) }
            } else if (!fileExtensionsDataList.isNullOrEmpty()) {
                getCorrespondingAttachmentTypeIcon(
                    fileExtensionsDataList,
                    fileData
                )?.let { imgThumbnail?.setImageResource(it) }
            }
            if (fileData.isExceedMaxFile) {
                viewError?.visibility = View.VISIBLE
            } else {
                viewError?.visibility = View.GONE
            }
            viewDeleteFile?.setOnClickListener { onDeleteFileClicked() }

        }

        /**
         * Calls when delete file view clicked
         */
        private fun onDeleteFileClicked() {
            onFileClickListener?.onDeleteAttachmentClicked(
                fileDataList[layoutPosition]
            )
        }
    }


}