/*
 * Copyright (C) 2020 Link Development
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.linkdev.fileattachmentview.view

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.os.Build
import android.util.AttributeSet
import android.util.Size
import android.view.View
import android.widget.ImageView
import android.widget.TextView
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.widget.TextViewCompat
import androidx.fragment.app.Fragment
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.linkdev.fileattachmentview.R
import com.linkdev.fileattachmentview.models.AttachFileData
import com.linkdev.fileattachmentview.models.FileExtensionsData
import com.linkdev.fileattachmentview.utils.Constants
import com.linkdev.fileattachmentview.utils.Constants.FileTypes.PICK_FILE
import com.linkdev.fileattachmentview.utils.Constants.SelectionMode.SINGLE
import com.linkdev.fileattachmentview.utils.Logger
import com.linkdev.fileattachmentview.utils.UIUtils
import com.linkdev.fileattachmentview.utils.UIUtils.showToast
import com.linkdev.fileattachmentview.utils.Utils.convertFileDataToAttachFileData
import com.linkdev.fileattachmentview.utils.Utils.getCorrespondingAttachmentTypeIcon
import com.linkdev.fileattachmentview.utils.Utils.getCorrespondingFileSelectionMode
import com.linkdev.fileattachmentview.utils.Utils.getCorrespondingFileType
import com.linkdev.fileattachmentview.utils.Utils.getCorrespondingMimeTypes
import com.linkdev.fileattachmentview.utils.Utils.getCorrespondingRequestCode
import com.linkdev.fileattachmentview.utils.Utils.isValidAttachmentSize
import com.linkdev.fileattachmentview.utils.Utils.isValidResource
import com.linkdev.fileattachmentview.view.adapter.MultipleFileRecyclerViewAdapter
import com.linkdev.fileattachmentview.view.adapter.OnFileClickListener
import com.linkdev.fileattachmentview.view.listeners.IAttachFileActionsCallback
import com.linkdev.fileattachmentview.view.listeners.IAttachFileClickListener
import com.linkdev.fileattachmentview.view.listeners.IPickFileStatus
import com.linkdev.fileattachmentview.view.listeners.IRequestPermissions
import com.linkdev.fileattachmentview.view.mapper.Caller.Companion.getCallerActivity
import com.linkdev.fileattachmentview.view.mapper.Caller.Companion.getCallerFragment
import com.linkdev.filepicker.factory.IPickFilesFactory
import com.linkdev.filepicker.factory.PickFilesFactory
import com.linkdev.filepicker.interactions.PickFilesStatusCallback
import com.linkdev.filepicker.models.*

/**
 * FileAttachmentView is a view used to dynmically represent the add attachment view and handel the different action of it
 * like add action and delete action display the thumbnail of the attached image, file name if needed
 */
class FileAttachmentView @kotlin.jvm.JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0
) :
    ConstraintLayout(context, attrs, defStyleAttr) {

    companion object {
        const val NO_RESOURCE = -1
        const val DEFAULT_VIEW_VISIBILITY = 4
        const val NO_STYLE = 0
        const val DEFAULT_FILE_TYPE = PICK_FILE
        const val DEFAULT_SELECTION_MODE = SINGLE
        const val DEFAULT_SAVE_TO_GALLERY = false
        const val DEFAULT_MAX_ATTACHMENT_SIZE = 5
        const val DEFAULT_ATTACHMENT_THUMBNAIL_SIZE = 200
        const val TAG = "FileAttachmentView"
        const val IS_DISPLAY_IMAGE_THUMBNAIL_IN_MULTI_SELECTED_FILES_DEFAULT = false
    }

    /**
     * reference to the container of attachment view
     */
    private var mAttachmentLayout: Int? = null

    /**
     * reference to the attachment size hint view
     */
    private var mSizeHintView: Int? = null

    /**
     * attachment size hint textView
     */
    private var txtAttachmentSizeHint: TextView? = null

    /**
     * reference to the delete attachment view
     */
    private var mDeleteAttachmentView: Int? = null

    /**
     * delete attachment view
     */
    private var deleteAttachmentView: View? = null

    /**
     * reference to the imageView that will display the thumbnail of the attachment
     */
    private var mAttachmentThumbnailImageView: Int? = null

    /**
     * imageView that will display the thumbnail of the attachment
     */
    private var imgAttachmentThumbnail: ImageView? = null

    /**
     * reference to the the attachment description view
     */
    private var mAddAttachmentTextView: Int? = null

    /**
     * Add Attachment TextView
     */
    private var txtAddAttachment: TextView? = null

    /**
     * reference to the view which perform the attach action
     */
    private var mAddAttachmentView: Int? = null

    /**
     * add attachment view
     */

    private var viewAddAttachment: View? = null

    /**
     * refers to the attachment name view
     */
    private var mAttachmentNameView: Int? = null

    /**
     * refers to attachment name textView
     */
    private var txtAttachmentNameView: TextView? = null

    /**
     * reference to [IAttachFileClickListener] interface used to notify the consumer with different attach file actions
     */
    private var iAttachFileClickListener: IAttachFileClickListener? = null

    /**
     * reference to [IPickFilesFactory] interface used to pick files
     */
    private var pickFilesFactory: IPickFilesFactory? = null

    /**
     * refers to [IPickFileStatus] handle the different picking file status
     */
    private var pickFileStatusListener: IPickFileStatus? = null

    /**
     * refers to the attachment type view
     */
    private var mAttachmentTypeView: Int? = null

    /**
     * attachment type image view
     */
    private var imgAttachmentTypeView: ImageView? = null

    /**
     * reference to [FileAttachmentPermissions] object
     */
    private lateinit var fileAttachmentPermissions: FileAttachmentPermissions

    /**
     * refers to the required to access permission tag
     */
    private var mPermissionTag: String? = null

    /**
     * refers to the show permission rational message
     */
    private var mShowRationalMessage: String? = null

    /**
     * refers to show permission positive action text
     */
    private var mShowRationalPositiveActionText: String? = null

    /**
     * refers to show permission rational negative action text
     */
    private var mShowRationalNegativeActionText: String? = null

    /**
     * refers to the permission not granted message
     */
    private var mPermissionNotGrantedMessage: String? = null

    /**
     * refers to the never asked again for permission message
     */
    private var mPermissionNeverAskedAgainMessage: String? = null

    /**
     * list of required to grant permissions
     */
    private var mPermissions: ArrayList<String> = arrayListOf()

    /**
     * reference to the [IRequestPermissions]
     */
    private var iRequestPermissions: IRequestPermissions? = null


    /**
     * Caller fragment / activity
     */
    private var mCaller: Any? = null

    /**
     * list of the mimeType that the consumer need to choose file with this list
     */
    private var mMimeTypeNameList: ArrayList<String> = arrayListOf()

    /**
     * the type of file that consumer need to attach it with default type pick file
     */
    private var mFileType: Int = DEFAULT_FILE_TYPE

    /**
     * refers to file selection mode with default value single selection
     */

    private var mFileSelectionMode: Int = DEFAULT_SELECTION_MODE

    /**
     * refers to the created folder name in the gallery if you need to save captured image on it
     */
    private var mGalleryFolderName: String? = null

    /**
     * refers if the consumer need to save captured image to gallery or not
     */
    private var mIsSavedToGallery: Boolean = DEFAULT_SAVE_TO_GALLERY

    /**
     * refers to the picking cancel message
     */
    private var mPickingFileCanceledMessage: String? = null

    /**
     * refers to the max file size with default value 5 mbs
     */
    private var mMaxFileSize: Int = DEFAULT_MAX_ATTACHMENT_SIZE


    /**
     * an array list of [FileExtensionsData]
     */
    private var mFileExtensionsDataList: ArrayList<FileExtensionsData>? = null

    /**
     * refers to delete attachment dialog title text
     */
    private var mDeleteAttachmentDialogTitle: String? = null

    /**
     * refers to delete attachment dialog message
     */
    private var mDeleteAttachmentDialogMessage: String? = null

    /**
     * refers to delete attachment dialog positive button text
     */
    private var mDeleteAttachmentDialogPositiveButtonText: String? = null

    /**
     * refers to delete attachment dialog negative button text
     */
    private var mDeleteAttachmentDialogNegativeButtonText: String? = null

    /**
     * refers to resource id contains the avatar of add attachment image
     */
    private var mAddAttachmentImageAvatar: Int? = null

    /**
     * refers to the attachment thumbnail width
     */
    private var mAttachmentThumbnailWidth: Int = DEFAULT_ATTACHMENT_THUMBNAIL_SIZE

    /**
     * refers to the attachment thumbnail height
     */
    private var mAttachmentThumbnailHeight: Int = DEFAULT_ATTACHMENT_THUMBNAIL_SIZE

    /**
     * refers to the add attachment description view
     */
    private var mAddAttachmentDescriptionView: Int? = null

    /**
     * refers to the add attachment description textView
     */
    private var txtAddAttachmentDescription: TextView? = null

    /**
     * refers to the container of the multiple selection view
     */
    private var mMultipleSelectionView: Int? = null

    /**
     * RecyclerView of multiple file selection
     */
    private var rcvMultipleSelection: RecyclerView? = null

    /**
     * reference to the multiple selection item
     */
    private var mMultipleSelectionItemView: Int? = null

    /**
     * reference to delete item  action
     */
    private var mDeleteMultipleSelectionItemView: Int? = null

    /**
     * reference to the multiple selection file name view
     */
    private var mMultipleSelectionItemNameView: Int? = null

    /**
     * reference to the multiple selection file Thumbnail or file extension
     */
    private var mMultipleSelectionItemThumbnail: Int? = null

    /**
     * reference to the error view of multiple file selection
     */
    private var mMultipleSelectionErrorView: Int? = null

    /**
     * reference to the [MultipleFileRecyclerViewAdapter]
     */
    private lateinit var multipleFileRecyclerViewAdapter: MultipleFileRecyclerViewAdapter

    /**
     * indicates if the picked file is image and consumer need to display it's thumbnail in case of multi file selected
     */
    private var mIsDisplayThumbnail: Boolean? = null

    /**
     * layout manger of multi selection file recyclerView
     */
    private var multiSelectionLayoutManager: RecyclerView.LayoutManager? = null

    /**
     * reference to [IAttachFileActionsCallback]
     */
    private var mIAttachFileActionsCallback: IAttachFileActionsCallback? = null

    /**
     * reference to string resource contains the file exceed max size message
     */
    private var mSizeErrorMessage: String? = null

    /**
     * reference to the style of size hint view
     */
    private var mSizeHintViewStyle: Int? = null

    /**
     * visibility of size hint view
     */
    private var sizeHintViewVisibility: Int? = null

    /**
     * initialize the view attributes
     */
    init {
        attrs?.let { initAttributes(it) }
    }

    /**
     * set the view different Attributes
     * [attrs] instance of  [AttributeSet] is A collection of attributes, as found associated with a tag in an XML document.
     */
    private fun initAttributes(attrs: AttributeSet) {
        val attachmentLayout: Int
        val sizeHintView: Int
        val deleteAttachmentView: Int
        val attachmentThumbnailImageView: Int
        val addAttachmentTextView: Int
        val addAttachmentText: String?
        val addAttachmentView: Int
        val addAttachmentIcon: Int
        val deleteAttachmentIcon: Int
        val sizeHintViewStyle: Int
        val sizeHintText: String?
        val attachmentNameView: Int?
        val attachmentNameViewVisibility: Int?
        val attachmentTypeIconVisibility: Int?
        val attachmentTypeView: Int?
        val permissionTag: String?
        val showRationalMessage: String?
        val showRationalPositiveActionText: String?
        val showRationalNegativeActionText: String?
        val permissionNotGrantedMessage: String?
        val permissionNeverAskedAgainMessage: String?
        val fileType: Int
        val fileSelectionMode: Int
        val galleryFolderName: String?
        val isSavedToGallery: Boolean?
        val pickingFileCanceledMessage: String?
        val maxFileSize: Int
        val deleteAttachmentDialogTitle: String?
        val deleteAttachmentDialogMessage: String?
        val deleteAttachmentDialogPositiveButtonText: String?
        val deleteAttachmentDialogNegativeButtonText: String?
        val addAttachmentImageAvatar: Int?
        val attachmentThumbnailWidth: Int
        val attachmentThumbnailHeight: Int
        val addAttachmentDescriptionView: Int?
        val addAttachmentDescriptionText: String?
        val multipleSelectionView: Int?
        val multipleSelectionItemView: Int?
        val deleteMultipleSelectionItemView: Int?
        val multipleSelectionItemNameView: Int?
        val multipleSelectionItemThumbnail: Int?
        val multipleSelectionErrorView: Int
        val isDisplayThumbnail: Boolean?
        val sizeErrorMessage: String?
        val typedArray = context.obtainStyledAttributes(attrs, R.styleable.FileAttachmentView)
        attachmentLayout = typedArray.getResourceId(
            R.styleable.FileAttachmentView_attachmentLayout,
            NO_RESOURCE
        )
        addAttachmentImageAvatar = typedArray.getResourceId(
            R.styleable.FileAttachmentView_addAttachmentImageAvatar,
            NO_RESOURCE
        )
        permissionTag = typedArray.getString(
            R.styleable.FileAttachmentView_permissionTag
        )
        showRationalMessage = typedArray.getString(
            R.styleable.FileAttachmentView_showRationalMessage
        )
        showRationalPositiveActionText = typedArray.getString(
            R.styleable.FileAttachmentView_showRationalPositiveActionText
        )
        showRationalNegativeActionText = typedArray.getString(
            R.styleable.FileAttachmentView_showRationalNegativeActionText
        )
        permissionNotGrantedMessage = typedArray.getString(
            R.styleable.FileAttachmentView_permissionNotGrantedMessage
        )
        permissionNeverAskedAgainMessage = typedArray.getString(
            R.styleable.FileAttachmentView_permissionNeverAskedAgainMessage
        )

        sizeHintView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_sizeHintView,
            NO_RESOURCE
        )
        deleteAttachmentView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_deleteAttachmentView,
            NO_RESOURCE
        )

        attachmentThumbnailImageView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_attachmentThumbnailImageView,
            NO_RESOURCE
        )
        addAttachmentTextView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_addAttachmentTextView,
            NO_RESOURCE
        )

        addAttachmentText = typedArray.getString(
            R.styleable.FileAttachmentView_addAttachmentText
        )
        addAttachmentView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_addAttachmentView,
            NO_RESOURCE
        )
        addAttachmentIcon = typedArray.getResourceId(
            R.styleable.FileAttachmentView_addAttachmentIcon,
            NO_RESOURCE
        )
        deleteAttachmentIcon = typedArray.getResourceId(
            R.styleable.FileAttachmentView_deleteAttachmentIcon,
            NO_RESOURCE
        )
        sizeHintViewVisibility = typedArray.getInteger(
            R.styleable.FileAttachmentView_sizeHintViewVisibility,
            DEFAULT_VIEW_VISIBILITY
        )
        sizeHintViewStyle = typedArray.getResourceId(
            R.styleable.FileAttachmentView_sizeHintViewStyle,
            NO_STYLE
        )
        sizeHintText = typedArray.getString(
            R.styleable.FileAttachmentView_sizeHintText
        )
        attachmentNameView =
            typedArray.getResourceId(R.styleable.FileAttachmentView_attachmentNameView, NO_RESOURCE)
        attachmentNameViewVisibility = typedArray.getInteger(
            R.styleable.FileAttachmentView_attachmentNameViewVisibility,
            DEFAULT_VIEW_VISIBILITY
        )
        attachmentTypeIconVisibility = typedArray.getInteger(
            R.styleable.FileAttachmentView_attachmentTypeIconVisibility,
            DEFAULT_VIEW_VISIBILITY
        )
        attachmentTypeView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_attachmentTypeView,
            NO_RESOURCE
        )
        fileType = typedArray.getInteger(R.styleable.FileAttachmentView_fileType, DEFAULT_FILE_TYPE)
        fileSelectionMode = typedArray.getInteger(
            R.styleable.FileAttachmentView_fileSelectionMode,
            DEFAULT_FILE_TYPE
        )
        galleryFolderName = typedArray.getString(R.styleable.FileAttachmentView_galleryFolderName)
        isSavedToGallery = typedArray.getBoolean(
            R.styleable.FileAttachmentView_isSavedToGallery,
            DEFAULT_SAVE_TO_GALLERY
        )
        pickingFileCanceledMessage = typedArray.getString(
            R.styleable.FileAttachmentView_pickingFileCanceledMessage
        )

        maxFileSize = typedArray.getInteger(
            R.styleable.FileAttachmentView_maxFileSize,
            DEFAULT_MAX_ATTACHMENT_SIZE
        )
        deleteAttachmentDialogTitle =
            typedArray.getString(
                R.styleable.FileAttachmentView_deleteAttachmentDialogTitle
            )
        deleteAttachmentDialogMessage =
            typedArray.getString(
                R.styleable.FileAttachmentView_deleteAttachmentDialogMessage
            )
        deleteAttachmentDialogPositiveButtonText =
            typedArray.getString(
                R.styleable.FileAttachmentView_deleteAttachmentDialogPositiveButtonText
            )
        deleteAttachmentDialogNegativeButtonText =
            typedArray.getString(
                R.styleable.FileAttachmentView_deleteAttachmentDialogNegativeButtonText
            )

        attachmentThumbnailWidth = typedArray.getInteger(
            R.styleable.FileAttachmentView_attachmentThumbnailWidth,
            DEFAULT_ATTACHMENT_THUMBNAIL_SIZE
        )
        attachmentThumbnailHeight = typedArray.getInteger(
            R.styleable.FileAttachmentView_attachmentThumbnailHeight,
            DEFAULT_ATTACHMENT_THUMBNAIL_SIZE
        )
        addAttachmentDescriptionView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_addAttachmentDescriptionView,
            NO_RESOURCE
        )
        addAttachmentDescriptionText =
            typedArray.getString(R.styleable.FileAttachmentView_addAttachmentDescriptionText)
        multipleSelectionView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_multipleSelectionView,
            NO_RESOURCE
        )
        multipleSelectionItemView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_multipleSelectionItemView,
            NO_RESOURCE
        )
        deleteMultipleSelectionItemView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_deleteMultipleSelectionItemView,
            NO_RESOURCE
        )
        multipleSelectionItemNameView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_multipleSelectionItemNameView,
            NO_RESOURCE
        )
        multipleSelectionItemThumbnail = typedArray.getResourceId(
            R.styleable.FileAttachmentView_multipleSelectionItemThumbnail,
            NO_RESOURCE
        )
        multipleSelectionErrorView = typedArray.getResourceId(
            R.styleable.FileAttachmentView_multipleSelectionErrorView,
            NO_RESOURCE
        )
        isDisplayThumbnail = typedArray.getBoolean(
            R.styleable.FileAttachmentView_isDisplayThumbnail,
            IS_DISPLAY_IMAGE_THUMBNAIL_IN_MULTI_SELECTED_FILES_DEFAULT
        )
        sizeErrorMessage = typedArray.getString(R.styleable.FileAttachmentView_sizeErrorMessage)
        setAttachmentLayout(attachmentLayout)
        inflateAttachmentView()
        setDeleteAttachmentDialogMessage(deleteAttachmentDialogMessage)
        setDeleteAttachmentDialogTitle(deleteAttachmentDialogTitle)
        setDeleteAttachmentDialogNegativeButtonText(deleteAttachmentDialogNegativeButtonText)
        setDeleteAttachmentDialogPositiveButtonText(deleteAttachmentDialogPositiveButtonText)
        setAttachmentSizeHintView(sizeHintView)
        setAddAttachmentDescriptionView(addAttachmentDescriptionView)
        setAttachmentDeleteView(deleteAttachmentView)
        setAttachmentThumbnailImageView(attachmentThumbnailImageView)
        setAddAttachmentTextView(addAttachmentTextView)
        setAddAttachmentView(addAttachmentView)
        setAttachmentNameView(attachmentNameView)
        setAttachmentTypeView(attachmentTypeView)
        initializeFileAttachment()
        setThumbnailWidth(attachmentThumbnailWidth)
        setThumbnailHeight(attachmentThumbnailHeight)
        setAddAttachmentImageAvatar(addAttachmentImageAvatar)
        setFileType(fileType)
        setMaxFileSize(maxFileSize)
        setSaveFileToGallery(isSavedToGallery)
        setGalleryFolderName(galleryFolderName)
        setFileSelectionMode(fileSelectionMode)
        setPickingCanceledMessage(pickingFileCanceledMessage)
        setMultipleSelectionItemView(multipleSelectionItemView)
        setMultipleSelectionView(multipleSelectionView)
        setDeleteMultipleSelectionItemView(deleteMultipleSelectionItemView)
        setMultipleSelectionItemNameView(multipleSelectionItemNameView)
        setMultipleSelectionItemThumbnail(multipleSelectionItemThumbnail)
        setMultipleSelectionErrorView(multipleSelectionErrorView)
        setDisplayImageThumbnailInMultiSelection(isDisplayThumbnail)
        initializeFileAttachment()
        setAddAttachmentDescriptionText(addAttachmentDescriptionText)
        setPermissionTag(permissionTag)
        setPermissionNeverAskedAgainMessage(permissionNeverAskedAgainMessage)
        setPermissionNotGrantedMessage(permissionNotGrantedMessage)
        setShowRationalMessage(showRationalMessage)
        setShowRationalNegativeActionText(showRationalNegativeActionText)
        setShowRationalPositiveActionText(showRationalPositiveActionText)
        setAttachmentTypeViewVisibility(attachmentTypeIconVisibility)
        setAttachmentNameViewVisibility(attachmentNameViewVisibility)
        setAttachmentNameView(attachmentNameView)
        setAddAttachmentText(addAttachmentText)
        setAddAttachmentIcon(addAttachmentIcon)
        setDeleteAttachmentIcon(deleteAttachmentIcon)
        setAttachmentSizeHintView(sizeHintView)
        setSizeHintViewVisibility(sizeHintViewVisibility)
        setSizeHintViewStyle(sizeHintViewStyle)
        setSizeHintViewText(sizeHintText)
        setSizeErrorMessage(sizeErrorMessage)
        typedArray.recycle()

    }

    /**
     * Sets [mSizeErrorMessage] using [sizeErrorMessage]
     * [sizeErrorMessage] reference to error message of exceeding max file size
     */
    fun setSizeErrorMessage(sizeErrorMessage: String?) {
        mSizeErrorMessage = sizeErrorMessage
    }

    /**
     * Sets [mIAttachFileActionsCallback] using iAttachFileActionsCallback
     * [iAttachFileActionsCallback] reference to [IAttachFileActionsCallback]
     */
    fun setAttachFileActionCallBack(iAttachFileActionsCallback: IAttachFileActionsCallback) {
        mIAttachFileActionsCallback = iAttachFileActionsCallback
    }

    /**
     * Sets [mIsDisplayThumbnail] using [isDisplay]
     * [isDisplay] indicates if the image thumbnail need to display or not
     */
    fun setDisplayImageThumbnailInMultiSelection(isDisplay: Boolean?) {
        mIsDisplayThumbnail = isDisplay
    }


    /**
     * Sets the [txtAddAttachmentDescription] text using [addAttachmentDescriptionText]
     * [addAttachmentDescriptionText] text of the add attachment description view
     */
    fun setAddAttachmentDescriptionText(addAttachmentDescriptionText: String?) {
        txtAddAttachmentDescription?.text = addAttachmentDescriptionText
    }

    /**
     * Sets the [mAddAttachmentDescriptionView] using [addAttachmentDescriptionView]
     * [addAttachmentDescriptionView] refers to the add attachment description view
     */
    private fun setAddAttachmentDescriptionView(addAttachmentDescriptionView: Int?) {
        mAddAttachmentDescriptionView = addAttachmentDescriptionView
    }

    /**
     * Sets[mAttachmentThumbnailWidth] using [width]
     * [width] refers to thumbnail width
     */
    fun setThumbnailWidth(width: Int) {
        mAttachmentThumbnailWidth = width
    }

    /**
     * Sets[mAttachmentThumbnailHeight] using [height]
     * [height] refers to thumbnail width
     */
    fun setThumbnailHeight(height: Int) {
        mAttachmentThumbnailHeight = height
    }

    /**
     * Sets [mDeleteAttachmentDialogTitle] using [deleteAttachmentDialogTitle]
     * [deleteAttachmentDialogTitle] refers to delete attachment dialog title
     */
    fun setDeleteAttachmentDialogTitle(deleteAttachmentDialogTitle: String?) {
        mDeleteAttachmentDialogTitle = deleteAttachmentDialogTitle
    }

    /**
     * Sets [mDeleteAttachmentDialogMessage] using [deleteAttachmentDialogMessage]
     * [deleteAttachmentDialogMessage] refers to delete attachment dialog message
     */
    fun setDeleteAttachmentDialogMessage(deleteAttachmentDialogMessage: String?) {
        mDeleteAttachmentDialogMessage = deleteAttachmentDialogMessage
    }

    /**
     * Sets [mDeleteAttachmentDialogPositiveButtonText] using [deleteAttachmentDialogPositiveButtonText]
     * [deleteAttachmentDialogPositiveButtonText] refers to delete attachment dialog positive button text
     */
    fun setDeleteAttachmentDialogPositiveButtonText(deleteAttachmentDialogPositiveButtonText: String?) {
        mDeleteAttachmentDialogPositiveButtonText = deleteAttachmentDialogPositiveButtonText
    }

    /**
     * Sets [mDeleteAttachmentDialogNegativeButtonText] using [deleteAttachmentDialogNegativeButtonText]
     * [deleteAttachmentDialogNegativeButtonText] refers to delete attachment dialog negative button text
     */
    fun setDeleteAttachmentDialogNegativeButtonText(deleteAttachmentDialogNegativeButtonText: String?) {
        mDeleteAttachmentDialogNegativeButtonText = deleteAttachmentDialogNegativeButtonText
    }

    /**
     * Sets the [mFileExtensionsDataList] using [fileExtensionsDataList]
     */
    fun setFileExtensionDataList(fileExtensionsDataList: ArrayList<FileExtensionsData>) {
        mFileExtensionsDataList = fileExtensionsDataList
    }

    /**
     * Sets the [mMaxFileSize] using [maxSize]
     * [maxSize] refers to the max file size
     */
    fun setMaxFileSize(maxSize: Int) {
        mMaxFileSize = maxSize
    }

    /**
     * Sets the [mPickingFileCanceledMessage] using [pickingCanceledMessage]
     * [pickingCanceledMessage] picking file cancellation message
     */
    fun setPickingCanceledMessage(pickingCanceledMessage: String?) {
        mPickingFileCanceledMessage = pickingCanceledMessage
    }

    /**
     * Sets the [mGalleryFolderName] using [folderName]
     * [folderName] refers to the name of the folder that the file will saved on it
     */
    fun setGalleryFolderName(folderName: String?) {
        mGalleryFolderName = folderName
    }

    /**
     * Sets the [mIsSavedToGallery] using [isSave]
     * [isSave] refers to save file to gallery or not
     */
    fun setSaveFileToGallery(isSave: Boolean) {
        mIsSavedToGallery = isSave
    }

    /**
     * Sets the [mFileSelectionMode] using [selectionMode]
     * [selectionMode] refers to the file selection mode
     */
    fun setFileSelectionMode(selectionMode: Int) {
        mFileSelectionMode = selectionMode
    }

    /**
     * Sets the [mFileType] using [fileType]
     * [fileType] the required to pick file type
     */
    fun setFileType(fileType: Int) {
        mFileType = fileType
    }

    /**
     * Sets the[mPermissionTag] using[permissionTag]
     */
    fun setPermissionTag(permissionTag: String?) {
        mPermissionTag = permissionTag
    }

    /**
     * Sets[mShowRationalMessage] using [showRationalMessage]
     */
    fun setShowRationalMessage(showRationalMessage: String?) {
        mShowRationalMessage = showRationalMessage
    }

    /**
     * Sets [mShowRationalPositiveActionText] using [showRationalPositiveActionText]
     */
    fun setShowRationalPositiveActionText(showRationalPositiveActionText: String?) {
        mShowRationalPositiveActionText = showRationalPositiveActionText
    }

    /**
     * Sets [mPermissionNotGrantedMessage] using [permissionNotGrantedMessage]
     */
    fun setPermissionNotGrantedMessage(permissionNotGrantedMessage: String?) {
        mPermissionNotGrantedMessage = permissionNotGrantedMessage
    }

    /**
     * Sets [mPermissionNeverAskedAgainMessage] using [permissionNeverAskedAgainMessage]
     */
    fun setPermissionNeverAskedAgainMessage(permissionNeverAskedAgainMessage: String?) {
        mPermissionNeverAskedAgainMessage = permissionNeverAskedAgainMessage
    }

    /**
     * Sets [mShowRationalNegativeActionText] using [showRationalNegativeActionText]
     */
    fun setShowRationalNegativeActionText(showRationalNegativeActionText: String?) {
        mShowRationalNegativeActionText = showRationalNegativeActionText
    }

    /**
     * Sets the [mMultipleSelectionErrorView] using [multipleSelectionErrorView]
     * [multipleSelectionErrorView] refers to the error view in multiple selection view
     */
    fun setMultipleSelectionErrorView(multipleSelectionErrorView: Int) {
        mMultipleSelectionErrorView = multipleSelectionErrorView
    }

    /**
     * Sets the [mDeleteMultipleSelectionItemView] using [deleteMultipleSelectionItemView]
     * [deleteMultipleSelectionItemView] refers to the view that perform the delete action in multiple selection view
     */
    fun setDeleteMultipleSelectionItemView(deleteMultipleSelectionItemView: Int?) {
        mDeleteMultipleSelectionItemView = deleteMultipleSelectionItemView
    }

    /**
     * Sets the [mMultipleSelectionItemThumbnail] using [multipleSelectionItemThumbnail]
     * [multipleSelectionItemThumbnail] refers to the view that display  file Thumbnail or the file extension icon in multiple selection view
     */
    fun setMultipleSelectionItemThumbnail(multipleSelectionItemThumbnail: Int?) {
        mMultipleSelectionItemThumbnail = multipleSelectionItemThumbnail
    }

    /**
     * Sets the [mMultipleSelectionItemNameView] using [multipleSelectionItemNameView]
     * [multipleSelectionItemNameView] refers to the view that display the file name in multiple selection view
     */
    fun setMultipleSelectionItemNameView(multipleSelectionItemNameView: Int?) {
        mMultipleSelectionItemNameView = multipleSelectionItemNameView
    }

    /**
     * Sets the [mMultipleSelectionView] using [multipleSelectionView]
     * [multipleSelectionView] refers to multiple selection view
     */
    fun setMultipleSelectionView(multipleSelectionView: Int?) {
        mMultipleSelectionView = multipleSelectionView
    }

    /**
     * Sets the [mMultipleSelectionItemView] using [multipleSelectionItemView]
     * [multipleSelectionItemView] refers to multiple selection item view
     */
    fun setMultipleSelectionItemView(multipleSelectionItemView: Int?) {
        mMultipleSelectionItemView = multipleSelectionItemView
    }

    /**
     * Sets the [mAttachmentNameView] using [attachmentNameView]
     * [attachmentNameView] refers to the attachment name view
     */
    private fun setAttachmentNameView(attachmentNameView: Int?) {
        mAttachmentNameView = attachmentNameView
    }

    /**
     * Sets the [mAttachmentTypeView] using [attachmentTypeView]
     * [attachmentTypeView] refers to the attachment type view
     */
    private fun setAttachmentTypeView(attachmentTypeView: Int?) {
        mAttachmentTypeView = attachmentTypeView
    }

    /**
     * Sets the visibility of [txtAttachmentNameView]
     */
    fun setAttachmentNameViewVisibility(visibility: Int) {
        txtAttachmentNameView?.visibility = visibility
    }

    /**
     * Sets the visibility of [imgAttachmentTypeView]
     */
    fun setAttachmentTypeViewVisibility(visibility: Int) {
        imgAttachmentTypeView?.visibility = visibility
    }

    /**
     * Sets the visibility of [txtAddAttachment]
     */
    fun setAddAttachmentTextVisibility(visibility: Int) {
        txtAddAttachment?.visibility = visibility
    }

    /**
     * Sets the text of [txtAttachmentSizeHint] using [hintMessage]
     * [hintMessage] refers to size hint message
     */
    fun setSizeHintViewText(hintMessage: String?) {
        if (!hintMessage.isNullOrEmpty())
            txtAttachmentSizeHint?.text = hintMessage
    }

    /**
     * Sets the [txtAttachmentSizeHint] style using the [sizeHintStyle]
     * [sizeHintStyle] refers to the size hint style resource
     */
    private fun setSizeHintViewStyle(sizeHintStyle: Int?) {
        mSizeHintViewStyle = sizeHintStyle

    }

    /**
     * apply style to the [txtAttachmentSizeHint] using style
     * [style] reference to style resource
     */
    fun applySizeHintViewStyle(style: Int?) {
        if (style != NO_STYLE && style != null)
            txtAttachmentSizeHint?.let {
                TextViewCompat.setTextAppearance(
                    it,
                    style
                )
            }

    }

    /**
     * Sets [mAttachmentLayout] using [attachmentLayout]
     * [attachmentLayout] reference to the attachment view
     */
    private fun setAttachmentLayout(attachmentLayout: Int?) {
        if (!isValidResource(attachmentLayout)) {
            Logger.log(
                context.getString(R.string.layout_error_message),
                logErrorMessage = context.getString(R.string.layout_error_message)
            )
        } else
            mAttachmentLayout = attachmentLayout

    }

    /**
     * Call this method to inflate the attachment layout using [mAttachmentLayout]
     */
    private fun inflateAttachmentView() {
        mAttachmentLayout?.let {
            inflate(context, it, this)
        }
    }


    /**
     * Sets the attachment [mSizeHintView] using[sizeHintView]
     * [sizeHintView] reference to the attachment sizeHintView
     */
    private fun setAttachmentSizeHintView(sizeHintView: Int?) {
        mSizeHintView = sizeHintView
    }

    /**
     * Sets the delete attachment View using[deleteView]
     * [deleteView] reference to the delete attachment View
     */
    private fun setAttachmentDeleteView(deleteView: Int?) {
        mDeleteAttachmentView = deleteView
    }

    /**
     * Sets the [mAttachmentThumbnailImageView] using [thumbnailImageView]
     * [thumbnailImageView] reference to the imageView that displays the the attachment thumbnail
     */
    private fun setAttachmentThumbnailImageView(thumbnailImageView: Int?) {
        mAttachmentThumbnailImageView = thumbnailImageView
    }

    /**
     * Sets the [mAddAttachmentTextView] using[addAttachmentTextView]
     * [addAttachmentTextView] reference to the textView that hold add attachment message
     */
    private fun setAddAttachmentTextView(addAttachmentTextView: Int?) {
        mAddAttachmentTextView = addAttachmentTextView
    }

    /**
     * Sets text for [txtAddAttachment] using [addAttachmentText]
     * [addAttachmentText] the add attachment text
     */
    fun setAddAttachmentText(addAttachmentText: String?) {
        if (!addAttachmentText.isNullOrEmpty())
            txtAddAttachment?.text = addAttachmentText
    }

    /**
     * Sets the[mAddAttachmentView] using [addAttachmentView]
     * [addAttachmentView] the add attachment View
     */
    private fun setAddAttachmentView(addAttachmentView: Int?) {
        mAddAttachmentView = addAttachmentView
    }

    /**
     * Sets the[viewAddAttachment] background resource using [addAttachmentIcon]
     * [addAttachmentIcon] the add attachment icon
     */
    fun setAddAttachmentIcon(addAttachmentIcon: Int?) {
        if (isValidResource(addAttachmentIcon))
            viewAddAttachment?.setBackgroundResource(addAttachmentIcon!!)
    }

    /**
     * Sets the[deleteAttachmentView] background using [deleteAttachmentIcon]
     * [deleteAttachmentIcon] the delete attachment icon
     */
    fun setDeleteAttachmentIcon(deleteAttachmentIcon: Int?) {
        if (isValidResource(deleteAttachmentIcon))
            deleteAttachmentView?.setBackgroundResource(deleteAttachmentIcon!!)
    }


    /**
     * Sets the[txtAttachmentSizeHint] using [sizeHintViewVisibility]
     * [sizeHintViewVisibility] the visibility of the file size hint view
     */
    fun setSizeHintViewVisibility(sizeHintViewVisibility: Int?) {
        if (sizeHintViewVisibility != null)
            txtAttachmentSizeHint?.visibility = sizeHintViewVisibility
    }

    /**
     * initialize the file attachment views , setAttachmentListeners
     */
    private fun initializeFileAttachment() {
        initAttachmentViews()
        setFileAttachmentListeners()
    }

    /**
     * Sets the click listeners of [FileAttachmentView]
     */
    private fun setFileAttachmentListeners() {
        viewAddAttachment?.setOnClickListener { onAddAttachmentClicked() }
        deleteAttachmentView?.setOnClickListener { onDeleteAttachmentClicked() }

    }

    /**
     * handle delete attachment
     */
    fun onDeleteAttachmentClicked() {
        getActiveAttachFileListener().onDeleteAttachmentClicked()

    }

    /**
     * show the delete attachment dialog and handle the delete action
     * [title] title of the delete dialog
     * [message] delete message of the dialog
     * [positiveButton] positive button string
     * [negativeButton] negative button string
     * [selectedFile] file data needed to delete
     */
    fun displayDeleteAttachmentDialog(
        title: String? = null,
        message: String? = null,
        positiveButton: String,
        negativeButton: String? = null,
        selectedFile: AttachFileData? = null
    ) {
        UIUtils.showBasicDialog(context, title, message,
            positiveButton, negativeButton, true,
            { _, _ ->
                onDeleteAttachmentPositiveActionClicked(selectedFile)

            },
            { _, _ ->

            })
    }


    /**
     * Call this method to set the [iAttachFileClickListener]
     * Set it if the consumer need to handle the callback by itSelf
     * [attachFileClickListener] reference to [IAttachFileClickListener]
     */
    fun setAttachFileListener(attachFileClickListener: IAttachFileClickListener) {
        iAttachFileClickListener = attachFileClickListener
    }

    /**
     * initialize the attachment different views
     */
    private fun initAttachmentViews() {
        if (isValidResource(mSizeHintView))
            txtAttachmentSizeHint = findViewById(mSizeHintView!!)
        if (isValidResource(mDeleteAttachmentView))
            deleteAttachmentView = findViewById(mDeleteAttachmentView!!)
        if (isValidResource(mAttachmentThumbnailImageView))
            imgAttachmentThumbnail = findViewById(mAttachmentThumbnailImageView!!)
        if (isValidResource(mAddAttachmentTextView))
            txtAddAttachment = findViewById(mAddAttachmentTextView!!)
        if (isValidResource(mAddAttachmentView))
            viewAddAttachment = findViewById(mAddAttachmentView!!)
        if (isValidResource(mAttachmentNameView))
            txtAttachmentNameView = findViewById(mAttachmentNameView!!)
        if (isValidResource(mAttachmentTypeView))
            imgAttachmentTypeView = findViewById(mAttachmentTypeView!!)
        if (isValidResource(mAddAttachmentDescriptionView))
            txtAddAttachmentDescription = findViewById(mAddAttachmentDescriptionView!!)
        if (isValidResource(mMultipleSelectionView))
            rcvMultipleSelection = findViewById(mMultipleSelectionView!!)
    }

    /**
     * Calls to pick an image after the access storage permission granted
     * [mimeTypeNameList] arrayList of different mime type names that's need to attach
     * [caller] host Fragment/Activity
     * [requestCode] used to handle [Fragment.onActivityResult]/[android.app.Activity.onActivityResult]
     * [fileType] refers to the attachment file type
     * [allowSyncWithGallery] boolean to check if should copy captured image the gallery
     * [galleryFolderName] for custom folder name to save camera files
     */
    fun pickFile(
        mimeTypeNameList: ArrayList<String>? = null,
        caller: Any,
        requestCode: Int,
        fileType: Int,
        galleryFolderName: String? = null,
        allowSyncWithGallery: Boolean = DEFAULT_SAVE_TO_GALLERY
    ) {
        initializePickFilesFactory(
            caller = caller,
            requestCode = requestCode,
            selectionMode = getCorrespondingFileSelectionMode(mFileSelectionMode),
            fileType = getCorrespondingFileType(fileType),
            galleryFolderName = galleryFolderName,
            allowSyncWithGallery = allowSyncWithGallery
        )

        if (mimeTypeNameList.isNullOrEmpty()) {
            // in case of capture photo no need for mime types
            pickFilesFactory?.pickFiles()
        } else {
            val mimeTypes = getCorrespondingMimeTypes(mimeTypeNameList)
            if (mimeTypes.isNotEmpty())
                pickFilesFactory?.pickFiles(
                    mimeTypeList = mimeTypes
                )
        }


    }

    /**
     * Calls to initialize the [pickFilesFactory]
     * [caller] host Fragment/Activity
     * [requestCode] used to handle [Fragment.onActivityResult]/[android.app.Activity.onActivityResult]
     * [selectionMode]refers to [SelectionMode] for [Intent.ACTION_OPEN_DOCUMENT] selection mode
     * [fileType] refers to the attachment file type
     * [allowSyncWithGallery] boolean to check if should copy captured image the gallery
     * [galleryFolderName] for custom folder name to save camera files
     */
    private fun initializePickFilesFactory(
        caller: Any,
        requestCode: Int,
        selectionMode: SelectionMode,
        fileType: FileTypes,
        galleryFolderName: String? = null,
        allowSyncWithGallery: Boolean = false
    ) {
        pickFilesFactory = PickFilesFactory(
            caller = caller,
            requestCode = requestCode,
            selectionMode = selectionMode,
            galleryFolderName = galleryFolderName,
            allowSyncWithGallery = allowSyncWithGallery
        ).getInstance(fileTypes = fileType)
    }

    /**
     * used to handle Activity result called on the host view [Fragment.onActivityResult]/[Activity.onActivityResult]
     * [mRequestCode]  to identify who this result came from
     * [resultCode] to identify if operation succeeded or canceled
     * [data] return result data to the caller
     * [callback] handle file status
     */
    fun handleFileAttachmentActivityResult(
        mRequestCode: Int,
        resultCode: Int,
        data: Intent?,
        callback: IPickFileStatus? = null
    ) {
        setPickFileStatusListener(callback)
        // check if the result is from handle permission
        if (mRequestCode == Constants.RequestCodes.REQUEST_APP_SETTINGS) {
            fileAttachmentPermissions.onHandleActivityResult(mRequestCode, resultCode, data)
        } else {
            pickFilesFactory?.handleActivityResult(
                mRequestCode,
                resultCode,
                data,
                pickFilesStatusCallback
            )
        }
    }

    /**
     * Sets the [pickFileStatusListener] using [iPickFileStatus]
     * [iPickFileStatus] reference to [IPickFileStatus]
     */
    private fun setPickFileStatusListener(iPickFileStatus: IPickFileStatus?) {
        pickFileStatusListener = iPickFileStatus
    }

    private val mPickFilesStatus = object : IPickFileStatus {
        override fun onPickFileCanceled() {
            showToast(context, mPickingFileCanceledMessage)
        }

        override fun onPickFileError(errorMessage: Int) {
            showToast(context, context.getString(errorMessage))
        }

        override fun onFilePicked(attachFileDataList: ArrayList<AttachFileData>) {
            if (mFileSelectionMode == SINGLE) {
                onSingleFilePicked(attachFileDataList[0])
            } else
                onMultipleFilesPicked(attachFileDataList)
        }

    }

    /**
     * handle the success file picking in case of multi file picked
     * [attachFileDataList] array list of multi selected file data
     */
    private fun onMultipleFilesPicked(attachFileDataList: ArrayList<AttachFileData>) {
        setUpMultipleFileSelectionRecyclerView(
            context,
            multiSelectionLayoutManager ?: LinearLayoutManager(context),
            checkFileSizes(attachFileDataList),
            isDisplayImageThumbnail = mIsDisplayThumbnail!!,
            imageThumbnailHeight = mAttachmentThumbnailHeight,
            imageThumbnailWidth = mAttachmentThumbnailWidth,
            fileExtensionsDataList = mFileExtensionsDataList,
        )
        if (validateAttachFileActionsCallbackListener())
            mIAttachFileActionsCallback?.onAttachmentLoaded(attachFileDataList)
    }

    /**
     * validate that [mIAttachFileActionsCallback] not equal null otherwise throw exception
     */
    private fun validateAttachFileActionsCallbackListener(): Boolean {
        if (mIAttachFileActionsCallback != null)
            return true
        else
            Logger.log(
                context.getString(R.string.attach_file_callback_error_message),
                logErrorMessage = context.getString(R.string.attach_file_callback_error_message)
            )
    }

    /**
     * handle the success file picking in case of single file picked
     */
    private fun onSingleFilePicked(pickedFile: AttachFileData) {
        if (isValidAttachmentSize(pickedFile.fileSize ?: 0.0, mMaxFileSize)) {
            if (validateAttachFileActionsCallbackListener())
                mIAttachFileActionsCallback?.onAttachmentLoaded(arrayListOf(pickedFile))
            // return to the default visibility
            if (sizeHintViewVisibility == View.GONE || sizeHintViewVisibility == View.INVISIBLE)
                setSizeHintViewVisibility(View.GONE)
            setAttachmentTypeViewVisibility(View.VISIBLE)
            setAddAttachmentTextVisibility(View.GONE)
            setAttachmentNameViewVisibility(View.VISIBLE)
            setDeleteAttachmentViewVisibility(View.VISIBLE)
            displayPickedFileName(attachFileData = pickedFile)
            mFileExtensionsDataList?.let {
                displayAttachmentType(
                    it,
                    fileData = pickedFile
                )
            }
            displayedPickedImageThumbnail(
                pickedFile,
                Size(mAttachmentThumbnailWidth, mAttachmentThumbnailHeight),
                imgAttachmentThumbnail
            )
        } else {
            applySizeHintViewStyle(mSizeHintViewStyle)
            if (sizeHintViewVisibility == View.GONE || sizeHintViewVisibility == View.INVISIBLE)
                setSizeHintViewVisibility(View.VISIBLE)
            mSizeErrorMessage?.let { showToast(context, it) }
        }
    }


    /**
     * implements the PickFilesStatusCallback interface
     */
    private val pickFilesStatusCallback = object : PickFilesStatusCallback {
        override fun onFilePicked(fileData: ArrayList<FileData>) {
            getActivePickFileStatusListener().onFilePicked(getFileAttachmentDataList(fileData))
        }

        override fun onPickFileCanceled() {
            getActivePickFileStatusListener().onPickFileCanceled()
        }

        override fun onPickFileError(errorModel: ErrorModel) {
            when (errorModel.errorStatus) {
                ErrorStatus.DATA_ERROR -> getActivePickFileStatusListener().onPickFileError(
                    errorModel.errorMessage
                )
                ErrorStatus.FILE_ERROR -> getActivePickFileStatusListener().onPickFileError(
                    errorModel.errorMessage
                )
                ErrorStatus.PICK_ERROR -> getActivePickFileStatusListener().onPickFileError(
                    errorModel.errorMessage
                )
            }
        }

    }

    /**
     * Gets list of [AttachFileData] using [fileDataList]
     * [fileDataList] list of [FileData] specific file data
     */
    private fun getFileAttachmentDataList(fileDataList: ArrayList<FileData>): ArrayList<AttachFileData> {
        val attachFileDataList = arrayListOf<AttachFileData>()
        for (fileData in fileDataList)
            attachFileDataList.add(convertFileDataToAttachFileData(fileData))
        return attachFileDataList
    }

    /**
     * display the Thumbnail of the attachment
     * [attachFileData] object contains the file specific information
     * [thumbnailSize] desired size of thumbnail
     */
    fun displayedPickedImageThumbnail(
        attachFileData: AttachFileData,
        thumbnailSize: Size,
        imageView: ImageView? = null
    ) {
        val thumbnailBitmap =
            attachFileData.convertAttachFileDataToFileData()
                .getThumbnail(context, thumbnailSize)
        imageView?.setImageBitmap(thumbnailBitmap)
    }

    /**
     * display the attached file name
     * [attachFileData] object contains the file specific information
     */
    fun displayPickedFileName(attachFileData: AttachFileData) {
        txtAttachmentNameView?.text = attachFileData.fileName
    }

    /**
     * Sets the visibility of add attachment view
     */
    fun setAddAttachmentViewVisibility(visibility: Int) {
        viewAddAttachment?.visibility = visibility
    }

    /**
     * Sets the visibility of delete attachment view
     */
    fun setDeleteAttachmentViewVisibility(visibility: Int) {
        deleteAttachmentView?.visibility = visibility
    }

    /**
     * display the avatar of [imgAttachmentThumbnail]
     * [avatar] refernce to drawable resource
     */
    fun displayAttachmentImageAvatar(avatar: Int?) {
        if (isValidResource(avatar))
            imgAttachmentThumbnail?.setImageResource(avatar!!)
    }

    /**
     * Sets [mAddAttachmentImageAvatar] using [avatar]
     * [avatar] refers to add attachment avatar
     */
    private fun setAddAttachmentImageAvatar(avatar: Int?) {
        mAddAttachmentImageAvatar = avatar
    }

    /**
     * Calls to display the attachment type icon
     * [fileExtensionsDataList] list of [FileExtensionsData] data of available file extension data
     * [fileData] the attached file data
     */
    fun displayAttachmentType(
        fileExtensionsDataList: ArrayList<FileExtensionsData>,
        fileData: AttachFileData
    ) {
        setAttachmentTypeIcon(
            getCorrespondingAttachmentTypeIcon(
                fileExtensionsDataList,
                fileData
            )
        )
    }


    /**
     * Sets the attachment type icon
     * [icon] refers to file extension drawable icon
     */
    fun setAttachmentTypeIcon(icon: Int?) {
        icon?.let { imgAttachmentTypeView?.setImageResource(it) }
    }

    /**
     * Calls to setup the [rcvMultipleSelection]
     *  [context] refers to the current context
     *  [layoutManager] layout manger of the recyclerView
     * [attachFileDataList] array list of the selected files data [AttachFileData]
     * [isDisplayImageThumbnail] boolean indicates if the consumer need to display the attached file thumbnail or not in case of the attached files are images
     * [fileExtensionsDataList] array list of available extensions data
     * [imageThumbnailWidth] width of the Thumbnail
     * [imageThumbnailHeight] height of the thumbnail
     */

    fun setUpMultipleFileSelectionRecyclerView(
        context: Context,
        layoutManager: RecyclerView.LayoutManager,
        attachFileDataList: ArrayList<AttachFileData>,
        isDisplayImageThumbnail: Boolean = false,
        fileExtensionsDataList: ArrayList<FileExtensionsData>? = null,
        imageThumbnailWidth: Int = DEFAULT_ATTACHMENT_THUMBNAIL_SIZE,
        imageThumbnailHeight: Int = DEFAULT_ATTACHMENT_THUMBNAIL_SIZE
    ) {

        rcvMultipleSelection?.layoutManager = layoutManager
        mMultipleSelectionItemView?.let {
            multipleFileRecyclerViewAdapter = MultipleFileRecyclerViewAdapter(
                context = context,
                fileDataList = attachFileDataList,
                adapterLayout = mMultipleSelectionItemView!!,
                deleteMultipleSelectionItemView = mDeleteMultipleSelectionItemView,
                multipleSelectionItemNameView = mMultipleSelectionItemNameView,
                multipleSelectionItemThumbnail = mMultipleSelectionItemThumbnail,
                multipleSelectionErrorView = mMultipleSelectionErrorView,
                isDisplayImageThumbnail = isDisplayImageThumbnail,
                fileExtensionsDataList = fileExtensionsDataList,
                imageThumbnailWidth = imageThumbnailWidth,
                imageThumbnailHeight = imageThumbnailHeight,
                onFileClickListener = onFileClickListener

            )
            rcvMultipleSelection?.adapter = multipleFileRecyclerViewAdapter
        } ?: kotlin.run {
            Logger.log(
                context.getString(R.string.multi_selection_layout_error_message),
                logErrorMessage = context.getString(R.string.multi_selection_layout_error_message)
            )
        }

    }

    /**
     * reference to [OnFileClickListener]
     */
    private val onFileClickListener = object : OnFileClickListener {
        override fun onDeleteAttachmentClicked(
            selectedFile: AttachFileData
        ) {
            getActiveAttachFileListener().onDeleteAttachmentClicked(selectedFile)
        }

    }

    /**
     * Calls when on file deleted from multi selected files
     * [selectedFile] file data needed to delete
     */
    private fun onMultipleSelectionFileDeleted(
        selectedFile: AttachFileData?
    ) {
        if (selectedFile != null)
            multipleFileRecyclerViewAdapter.deleteFile(selectedFile)
    }

    /**
     * initialize [fileAttachmentPermissions] object
     */
    fun initializeFileAttachmentPermission() {
        if (mCaller != null)
            fileAttachmentPermissions = FileAttachmentPermissions(
                getCallerFragment(caller = mCaller!!),
                getCallerActivity(caller = mCaller!!),
                getActiveRequestPermissionListener()
            )
        else {
            Logger.log(
                context.getString(R.string.caller_error_message),
                logErrorMessage = context.getString(R.string.caller_error_message)
            )
        }
    }

    /**
     * calls to check if the required permissions granted or not
     */
    fun checkPermissions() {
        fileAttachmentPermissions.checkPermissions(
            context,
            message = mShowRationalMessage ?: "",
            showRationalNegativeActionText = mShowRationalNegativeActionText,
            showRationalPositiveActionText = mShowRationalPositiveActionText,
            permissions = mPermissions,
            tag = mPermissionTag
        )
    }

    /**
     * Handle permissions requested result
     * [requestCode] the requested code
     * [permissions] the requested permissions
     * [grantResults] grant result for the permissions
     */
    fun handleOnRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {
        fileAttachmentPermissions.onRequestPermissionsResult(
            requestCode,
            permissions,
            grantResults
        )
    }


    /**
     * Calls to prepare file picking process
     */
    private fun prepareFilePicking() {
        mCaller?.let {
            pickFile(
                mimeTypeNameList = mMimeTypeNameList,
                caller = mCaller!!,
                requestCode = getCorrespondingRequestCode(mFileType),
                fileType = mFileType,
                galleryFolderName = mGalleryFolderName,
                allowSyncWithGallery = mIsSavedToGallery
            )
        }
    }

    /**
     * reference to [IRequestPermissions] contains callback to handle different permissions state
     */
    private val mIRequestPermissions = object : IRequestPermissions {
        override fun onPermissionGranted(permission: String) {
            prepareFilePicking()
        }

        override fun onNeverAskAgainChecked(permission: String) {
            showToast(context, mPermissionNeverAskedAgainMessage)
        }

        override fun onPermissionDenied(permission: String) {
            showToast(context, mPermissionNotGrantedMessage)
        }


    }

    /**
     * Sets [mPermissions] using [permissions]
     * [permissions] required to grant permissions
     */
    private fun setRequiredPermissions(permissions: ArrayList<String>) {
        mPermissions = permissions
    }

    /**
     * Sets [iRequestPermissions] using [iRequestPermissions]
     * set it if the consumer fragment/ activity need to handle the different permissions callbacks
     * [iRequestPermissions] reference to [IRequestPermissions]
     */
    fun setRequestPermissionsListener(iRequestPermissions: IRequestPermissions?) {
        this.iRequestPermissions = iRequestPermissions
    }

    /**
     * this method called when click on [viewAddAttachment] to add attachment
     */
    fun onAddAttachmentClicked() {
        getActiveAttachFileListener().onAddAttachmentClicked()
    }

    /**
     * reference to [IAttachFileClickListener]
     */
    private val mIAttachFileListener = object : IAttachFileClickListener {

        override fun onAddAttachmentClicked() {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                initializeFileAttachmentPermission()
                checkPermissions()
            } else {
                prepareFilePicking()
            }
        }

        override fun onDeleteAttachmentClicked(selectedFile: AttachFileData?) {

            displayDeleteAttachmentDialog(
                title = mDeleteAttachmentDialogTitle,
                message = mDeleteAttachmentDialogMessage,
                positiveButton = mDeleteAttachmentDialogPositiveButtonText ?: "",
                negativeButton = mDeleteAttachmentDialogNegativeButtonText,
                selectedFile
            )

        }


    }

    /**
     * handle delete action when the delete dialog positive action clicked
     * [selectedFile] data of the view that needed to delete
     */
    private fun onDeleteAttachmentPositiveActionClicked(selectedFile: AttachFileData?) {
        if (validateAttachFileActionsCallbackListener())
            mIAttachFileActionsCallback?.onAttachmentDeleted(selectedFile)
        if (mFileSelectionMode == SINGLE)
            onSingleSelectionFileDeleted()
        else
            onMultipleSelectionFileDeleted(selectedFile)
    }

    /**
     * handle delete single attachment action
     */
    private fun onSingleSelectionFileDeleted() {
        setAddAttachmentTextVisibility(View.VISIBLE)
        setAttachmentNameViewVisibility(View.GONE)
        setDeleteAttachmentViewVisibility(View.GONE)
        setAttachmentTypeViewVisibility(View.GONE)
        setAddAttachmentViewVisibility(View.VISIBLE)
        displayAttachmentImageAvatar(mAddAttachmentImageAvatar)
    }

    /**
     * Sets the caller [mCaller] with the caller fragment / activity
     *
     */
    private fun setCaller(caller: Any) {
        mCaller = caller
    }

    /**
     * Sets the [mMimeTypeNameList] with[mimeTypeNameList]
     * [mimeTypeNameList] list of provided mime types names
     */
    fun setMimeTypeList(mimeTypeNameList: ArrayList<String> = arrayListOf(MimeType.ALL_FILES.mimeTypeName)) {
        mMimeTypeNameList = mimeTypeNameList
    }

    /**
     * Gets the active [IPickFileStatus] listener
     */
    private fun getActivePickFileStatusListener(): IPickFileStatus {
        return pickFileStatusListener ?: mPickFilesStatus
    }

    /**
     * Gets the active [IRequestPermissions] listener
     */
    private fun getActiveRequestPermissionListener(): IRequestPermissions {
        return iRequestPermissions ?: mIRequestPermissions
    }

    /**
     * Gets the active [IAttachFileClickListener]
     */
    private fun getActiveAttachFileListener(): IAttachFileClickListener {
        return iAttachFileClickListener ?: mIAttachFileListener
    }

    /**
     * check the size of each attached file in case of multi file selected
     * [attachFileDataList] array list contains data of each attached file
     */
    fun checkFileSizes(attachFileDataList: ArrayList<AttachFileData>): ArrayList<AttachFileData> {
        val sizesAttachedFileList: ArrayList<AttachFileData> = arrayListOf()
        for (fileData in attachFileDataList) {
            fileData.isExceedMaxFile = !isValidAttachmentSize(fileData.fileSize!!, mMaxFileSize)
            sizesAttachedFileList.add(fileData)
        }
        return sizesAttachedFileList
    }

    /**
     * Sets the [multiSelectionLayoutManager]
     */
    fun setLayoutManger(layoutManager: RecyclerView.LayoutManager) {
        multiSelectionLayoutManager = layoutManager
    }

    /**
     * initialize the required fields of [FileAttachmentView]
     * [caller] the caller fragment/ activity
     * [requiredPermissions] list of run time permissions need to access
     */
    fun initializeFileAttachmentView(
        caller: Any,
        requiredPermissions: ArrayList<String>? = null
    ) {
        setCaller(caller)
        requiredPermissions?.let { setRequiredPermissions(it) }
    }

    /**
     * check request code in the existing requests code
     */
    fun ifExistingRequestCode(requestCode: Int): Boolean {
        return (requestCode == Constants.RequestCodes.PICK_IMAGE ||
                requestCode == Constants.RequestCodes.PICK_FILE ||
                requestCode == Constants.RequestCodes.REQUEST_APP_SETTINGS ||
                requestCode == Constants.RequestCodes.CAPTURE_IMAGE_REQUEST_CODE
                )
    }

}