/*
 * Copyright (C) 2020 Link Development
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.linkdev.fileattachmentview.view

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import androidx.core.app.ActivityCompat
import androidx.core.app.ActivityCompat.requestPermissions
import androidx.core.app.ActivityCompat.shouldShowRequestPermissionRationale
import androidx.core.content.ContextCompat
import androidx.fragment.app.Fragment
import com.linkdev.fileattachmentview.R
import com.linkdev.fileattachmentview.utils.Constants
import com.linkdev.fileattachmentview.utils.UIUtils
import com.linkdev.fileattachmentview.view.listeners.IRequestPermissions

/**
 * handle all permissions of [FileAttachmentView]
 * [fragment] the host fragment
 * [activity] the host activity
 * [iRequestPermissions reference to [IRequestPermissions]
 */
internal class FileAttachmentPermissions(
    private val fragment: Fragment?,
    private val activity: Activity?,
    private val iRequestPermissions: IRequestPermissions
) {

    /**
     * reference to the requested permission tag
     */
    private var mPermissionsTag: String? = null

    /**
     * context of current consumer
     */
    lateinit var mContext: Context

    companion object {

        private const val MY_PERMISSIONS_REQUEST = 1
    }

    /**
     * ArrayList of not granted permission
     */
    private var notGrantedPermissions = ArrayList<String>()


    /**
     * Calls to check if the passed permissions is granted or not
     * [context] context of current consumer
     * [tag] the tag of requested permission / permissions
     * [message] rational message explaining why your app needs the user to grant a particular runtime permission
     * [showRationalPositiveActionText] text of the positive action of rational dialog
     * [showRationalNegativeActionText] text of the negative action of rational dialog
     */
    fun checkPermissions(
        context: Context, tag: String? = null, message: String = "",
        permissions: ArrayList<String>,
        showRationalPositiveActionText: String? = null,
        showRationalNegativeActionText: String? = null,

        ) {
        mContext = context
        if (permissions.isEmpty())
            throw Exception(context.getString(R.string.no_permissions_provided))
        mPermissionsTag = tag
        notGrantedPermissions = ArrayList() // reset

        for (permission in permissions) {
            if (ContextCompat.checkSelfPermission(
                    context,
                    permission
                ) != PackageManager.PERMISSION_GRANTED
            ) {
                notGrantedPermissions.add(permission)
            }
        }

        if (notGrantedPermissions.isEmpty()) {
            iRequestPermissions.onPermissionGranted(tag ?: permissions[0])
        } else {
            if (shouldShowRationale(notGrantedPermissions)) {
                showCustomDialog(
                    message,
                    showRationalPositiveActionText,
                    showRationalNegativeActionText
                )
            } else {
                if (fragment != null) {
                    fragment.requestPermissions(
                        notGrantedPermissions.toTypedArray(),
                        MY_PERMISSIONS_REQUEST
                    )
                } else if (activity != null) {
                    requestPermissions(
                        activity,
                        notGrantedPermissions.toTypedArray(),
                        MY_PERMISSIONS_REQUEST
                    )
                }
            }
        }
    }

    /**
     * used to handle Activity result called on the host view [Fragment.onActivityResult]/[Activity.onActivityResult]
     * [requestCode]  to identify who this result came from
     * [resultCode] to identify if operation succeeded or canceled
     * [data] return result data to the caller
     */
    fun onHandleActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        if (requestCode == Constants.RequestCodes.REQUEST_APP_SETTINGS) {
            var allPermissionsGranted = true

            for (permission in notGrantedPermissions) {
                if (!isPermissionGranted(permission)) {
                    allPermissionsGranted = false
                }
            }

            when {
                allPermissionsGranted -> iRequestPermissions.onPermissionGranted(
                    mPermissionsTag
                        ?: notGrantedPermissions[0]
                )
                else -> iRequestPermissions.onPermissionDenied(
                    mPermissionsTag
                        ?: notGrantedPermissions[0]
                )
            }
        }
    }

    /**
     * Handle permissions requested by looping on the permissions and
     * setting two flags one for when all permissions are granted and one for when never ask again checked in any of the permissions.
     * [requestCode]  to identify who this result came from
     * [permissions] list of requested permissions
     * [grantResults] list of grant result
     */
    fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>, grantResults: IntArray
    ) {
        when (requestCode) {
            MY_PERMISSIONS_REQUEST -> {
                if (grantResults.isNotEmpty()) {
                    var allPermissionsGranted = true
                    var shouldShowRationale = true

                    for (grantResult in grantResults) {
                        if (grantResult == PackageManager.PERMISSION_DENIED) {
                            allPermissionsGranted = false
                            val showRationale = fragment?.shouldShowRequestPermissionRationale(
                                permissions[grantResults.indexOf(grantResult)]
                            )
                                ?: if (activity != null) {
                                    shouldShowRequestPermissionRationale(
                                        activity,
                                        permissions[grantResults.indexOf(grantResult)]
                                    )
                                } else {
                                    false
                                }
                            if (!showRationale) {
                                shouldShowRationale = false
                                break
                            }
                        }
                    }

                    when {
                        !shouldShowRationale -> iRequestPermissions.onNeverAskAgainChecked(
                            mPermissionsTag
                                ?: permissions[0]
                        )
                        allPermissionsGranted -> iRequestPermissions.onPermissionGranted(
                            mPermissionsTag
                                ?: permissions[0]
                        )
                        else -> iRequestPermissions.onPermissionDenied(
                            mPermissionsTag
                                ?: permissions[0]
                        )
                    }
                }
            }
        }
    }

    /**
     * check if permission granted or not
     * [permission] requested permission
     */
    private fun isPermissionGranted(permission: String): Boolean {
        return ActivityCompat.checkSelfPermission(
            mContext,
            permission
        ) == PackageManager.PERMISSION_GRANTED
    }

    /**
     * check if permission need to show rational or not
     * [permissions] array list of requested permissions
     */
    private fun shouldShowRationale(permissions: ArrayList<String>): Boolean {
        var isNeed = false
        for (i in permissions.indices) {
            if (fragment != null) {
                if (fragment.shouldShowRequestPermissionRationale(permissions[i])) {
                    isNeed = true
                    break
                }
            } else if (activity != null) {
                if (shouldShowRequestPermissionRationale(activity, permissions[i])) {
                    isNeed = true
                    break
                }
            }

        }
        return isNeed
    }


    /**
     * display custom dialog for rational message
     * [message] rational message explaining why your app needs the user to grant a particular runtime permission
     * [showRationalPositiveActionText] text of the positive action of rational dialog
     * [showRationalNegativeActionText] text of the negative action of rational dialog
     */
    private fun showCustomDialog(
        message: String, showRationalPositiveActionText: String? = null,
        showRationalNegativeActionText: String? = null
    ) {

        UIUtils.showBasicDialog(mContext,
            null,
            message,
            showRationalPositiveActionText ?: "",
            negativeButton = showRationalNegativeActionText,
            false,
            { _, _ ->
                if (fragment != null) {
                    fragment.requestPermissions(
                        notGrantedPermissions.toTypedArray(),
                        MY_PERMISSIONS_REQUEST
                    )
                } else if (activity != null) {
                    requestPermissions(
                        activity,
                        notGrantedPermissions.toTypedArray(),
                        MY_PERMISSIONS_REQUEST
                    )
                }
            },
            { _, _ ->
                iRequestPermissions.onPermissionDenied(mPermissionsTag ?: notGrantedPermissions[0])
            })

    }


}