/*
 * Copyright (C) 2020 Link Development
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.linkdev.fileattachmentview.utils

import com.linkdev.fileattachmentview.models.AttachFileData
import com.linkdev.fileattachmentview.models.FileExtensions
import com.linkdev.fileattachmentview.models.FileExtensionsData
import com.linkdev.fileattachmentview.utils.Constants.FileTypes.CAPTURE_IMAGE
import com.linkdev.fileattachmentview.utils.Constants.FileTypes.PICK_FILE
import com.linkdev.fileattachmentview.utils.Constants.FileTypes.PICK_IMAGE
import com.linkdev.fileattachmentview.utils.Constants.SelectionMode.MULTIPLE
import com.linkdev.fileattachmentview.utils.Constants.SelectionMode.SINGLE
import com.linkdev.fileattachmentview.view.FileAttachmentView.Companion.NO_RESOURCE
import com.linkdev.filepicker.models.FileData
import com.linkdev.filepicker.models.FileTypes
import com.linkdev.filepicker.models.MimeType
import com.linkdev.filepicker.models.SelectionMode


/**
 * utils class  contains helper methods
 */
object Utils {

    /**
     * check if the view id is validate or not
     * [viewId] the view id
     */
    fun isValidResource(viewId: Int?): Boolean {
        return viewId != null && viewId != NO_RESOURCE
    }

    /**
     * Gets the file mime types
     * [mimeTypeNames] array list of mime types name
     */
    fun getCorrespondingMimeTypes(mimeTypeNames: ArrayList<String>): ArrayList<MimeType> {
        val mMimeTypeList = ArrayList<MimeType>()
        for (mimeTypeName in mimeTypeNames) {
            for (mimeType in getMimeTypesList())
                if (mimeType.mimeTypeName.contains(mimeTypeName, ignoreCase = true))
                    mMimeTypeList.add(mimeType)
        }
        return mMimeTypeList
    }


    /**
     * Gets the file Types
     * [fileType] an integer refers to file type
     */
    fun getCorrespondingFileType(fileType: Int): FileTypes {
        return when (fileType) {
            PICK_IMAGE, PICK_FILE -> FileTypes.PICK_FILES
            CAPTURE_IMAGE -> FileTypes.CAPTURE_IMAGE
            else -> FileTypes.PICK_FILES
        }
    }

    /**
     * returns the corresponding request code
     */
    fun getCorrespondingRequestCode(fileType: Int): Int {
        return when (fileType) {
            PICK_FILE -> Constants.RequestCodes.PICK_FILE
            PICK_IMAGE -> Constants.RequestCodes.PICK_IMAGE
            CAPTURE_IMAGE -> Constants.RequestCodes.CAPTURE_IMAGE_REQUEST_CODE
            else -> Constants.RequestCodes.PICK_FILE
        }
    }


    /**
     * Gets the file selection mode
     * [selectionMode] an integer refers to file selection mode
     */
    fun getCorrespondingFileSelectionMode(selectionMode: Int): SelectionMode {
        return when (selectionMode) {
            SINGLE -> SelectionMode.SINGLE
            MULTIPLE -> SelectionMode.MULTIPLE
            else -> SelectionMode.SINGLE
        }
    }

    /**
     * convert the [FileData] object to [AttachFileData]
     * [fileData] an object of [FileData]
     */
    fun convertFileDataToAttachFileData(fileData: FileData): AttachFileData {
        return AttachFileData(
            uri = fileData.uri,
            filePath = fileData.filePath,
            file = fileData.file,
            fileName = fileData.fileName,
            mimeType = fileData.mimeType,
            fileSize = fileData.fileSize
        )
    }

    /**
     * gets an array list of all available mime types
     */
    fun getMimeTypesList(): ArrayList<MimeType> {
        return arrayListOf(
            MimeType.ALL_IMAGES,
            MimeType.JPEG,
            MimeType.PNG,
            MimeType.GIF,
            MimeType.BMP,
            MimeType.WEBP,
            MimeType.ALL_VIDEOS,
            MimeType.MPEG,
            MimeType.MP4,
            MimeType.GPP,
            MimeType.GPP2,
            MimeType.AVI,
            MimeType.TXT,
            MimeType.PDF,
            MimeType.DOC,
            MimeType.DOCX,
            MimeType.XLS,
            MimeType.XLSX,
            MimeType.PTT,
            MimeType.ALL_AUDIO,
            MimeType.PTTX,
            MimeType.MP3,
            MimeType.AUDIO_3GPP,
            MimeType.AUDIO_MP4,
            MimeType.AUDIO_AMR,
            MimeType.AUDIO_MIDI,
            MimeType.AUDIO_X_MIDI,
            MimeType.AUDIO_OGG,
            MimeType.AUDIO_WAV,
            MimeType.ALL_FILES
        )
    }

    /**
     * convert bytes to mega bytes
     */
    fun bytesToMegaBytes(bytes: Double?): Double {
        return if (bytes != null)
            bytes / 1000000
        else
            0.0
    }

    /**
     * check if the attachment size is valid or not
     * [attachmentSize] the attachment size
     * [maxFileSize] the max file size
     */
    fun isValidAttachmentSize(attachmentSize: Double, maxFileSize: Int): Boolean {
        return bytesToMegaBytes(attachmentSize) <= maxFileSize
    }
    /**
     * get the attached file Corresponding extension icon
     * [fileExtensionsDataList] list of [FileExtensionsData] data of available file extension data
     * [fileData] the attached file data
     */
     fun getCorrespondingAttachmentTypeIcon(
        fileExtensionsDataList: ArrayList<FileExtensionsData>,
        fileData: AttachFileData
    ): Int? {
        for (mimeType in getMimeTypesList()) {
            if (fileData.mimeType?.let { mimeType.mimeTypeName.contains(it) } == true) {
                if (mimeType == MimeType.ALL_IMAGES
                    || mimeType == MimeType.JPEG
                    || mimeType == MimeType.PNG
                    || mimeType == MimeType.GIF
                    || mimeType == MimeType.BMP
                    || mimeType == MimeType.WEBP
                ) {
                    for (fileExtensionsData in fileExtensionsDataList) {
                        if (fileExtensionsData.fileExtensions?.name?.equals(FileExtensions.IMAGE.name) == true) {
                            return fileExtensionsData.extensionsIcon
                        }
                    }

                } else if (mimeType == MimeType.MPEG
                    || mimeType == MimeType.MP4
                    || mimeType == MimeType.GPP
                    || mimeType == MimeType.GPP2
                    || mimeType == MimeType.AVI
                    || mimeType == MimeType.ALL_VIDEOS
                ) {
                    for (fileExtensionsData in fileExtensionsDataList) {
                        if (fileExtensionsData.fileExtensions?.name?.equals(FileExtensions.VIDEO.name) == true) {
                            return fileExtensionsData.extensionsIcon
                        }
                    }
                } else if (mimeType == MimeType.MP3
                    || mimeType == MimeType.AUDIO_3GPP
                    || mimeType == MimeType.AUDIO_MP4
                    || mimeType == MimeType.AUDIO_AMR
                    || mimeType == MimeType.AUDIO_MIDI
                    || mimeType == MimeType.AUDIO_X_MIDI
                    || mimeType == MimeType.AUDIO_OGG
                    || mimeType == MimeType.AUDIO_WAV
                    || mimeType == MimeType.ALL_AUDIO
                ) {
                    for (fileExtensionsData in fileExtensionsDataList) {
                        if (fileExtensionsData.fileExtensions?.name?.equals(FileExtensions.AUDIO.name) == true) {
                            return fileExtensionsData.extensionsIcon
                        }
                    }

                } else if (mimeType == MimeType.DOC
                    || mimeType == MimeType.DOCX
                ) {
                    for (fileExtensionsData in fileExtensionsDataList) {
                        if (fileExtensionsData.fileExtensions?.name?.equals(FileExtensions.DOCS.name) == true) {
                            return fileExtensionsData.extensionsIcon
                        }
                    }
                } else if (mimeType == MimeType.PDF) {
                    for (fileExtensionsData in fileExtensionsDataList) {
                        if (fileExtensionsData.fileExtensions?.name?.equals(FileExtensions.PDF.name) == true) {
                            return fileExtensionsData.extensionsIcon
                        }
                    }
                } else if (mimeType == MimeType.TXT) {
                    for (fileExtensionsData in fileExtensionsDataList) {
                        if (fileExtensionsData.fileExtensions?.name?.equals(FileExtensions.PLAIN_TEXT.name) == true) {
                            return fileExtensionsData.extensionsIcon
                        }
                    }
                    break
                } else if (mimeType == MimeType.PTT
                    || mimeType == MimeType.PTTX
                ) {
                    for (fileExtensionsData in fileExtensionsDataList) {
                        if (fileExtensionsData.fileExtensions?.name?.equals(FileExtensions.POWER_POINT.name) == true) {
                          return fileExtensionsData.extensionsIcon
                        }
                    }
                    break
                } else if (mimeType == MimeType.XLS
                    || mimeType == MimeType.XLSX
                ) {
                    for (fileExtensionsData in fileExtensionsDataList) {
                        if (fileExtensionsData.fileExtensions?.name?.equals(FileExtensions.EXCEL.name) == true) {
                            return fileExtensionsData.extensionsIcon
                        }
                    }

                } else if (mimeType == MimeType.ALL_FILES) {
                    for (fileExtensionsData in fileExtensionsDataList) {
                        if (fileExtensionsData.fileExtensions?.name?.equals(FileExtensions.ALL_FILES.name) == true) {
                            return fileExtensionsData.extensionsIcon
                        }
                    }
                }
            }
        }
        return null
    }

}