/*
 * Decompiled with CFR 0.152.
 */
package com.linecorp.centraldogma.server.storage.encryption;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.linecorp.armeria.internal.common.util.ReentrantShortLock;
import com.linecorp.centraldogma.internal.Jackson;
import com.linecorp.centraldogma.internal.shaded.guava.collect.ImmutableList;
import com.linecorp.centraldogma.internal.shaded.guava.primitives.Ints;
import com.linecorp.centraldogma.server.internal.storage.AesGcmSivCipher;
import com.linecorp.centraldogma.server.internal.storage.repository.git.rocksdb.GitObjectMetadata;
import com.linecorp.centraldogma.server.storage.encryption.EncryptionEntryExistsException;
import com.linecorp.centraldogma.server.storage.encryption.EncryptionStorageException;
import com.linecorp.centraldogma.server.storage.encryption.KeyWrapper;
import com.linecorp.centraldogma.server.storage.encryption.RocksDBStorage;
import com.linecorp.centraldogma.server.storage.encryption.SecretKeyWithVersion;
import com.linecorp.centraldogma.server.storage.encryption.WrappedDekDetails;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ReadOptions;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.Snapshot;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class RepositoryEncryptionStorage {
    private static final Logger logger = LoggerFactory.getLogger(RepositoryEncryptionStorage.class);
    private static final int BATCH_WRITE_SIZE = 1000;
    private final RocksDBStorage rocksDbStorage;
    private final KeyWrapper keyWrapper;
    private final String kekId;
    private final ReentrantShortLock lock = new ReentrantShortLock();
    @GuardedBy(value="lock")
    private final Map<String, Consumer<SecretKeyWithVersion>> currentDekListeners = new HashMap<String, Consumer<SecretKeyWithVersion>>();

    RepositoryEncryptionStorage(RocksDBStorage rocksDbStorage, KeyWrapper keyWrapper, String kekId) {
        this.rocksDbStorage = Objects.requireNonNull(rocksDbStorage, "rocksDbStorage");
        this.keyWrapper = Objects.requireNonNull(keyWrapper, "keyWrapper");
        this.kekId = Objects.requireNonNull(kekId, "kekId");
    }

    CompletableFuture<String> generateWdek() {
        byte[] dek = AesGcmSivCipher.generateAes256Key();
        return this.keyWrapper.wrap(dek, this.kekId);
    }

    List<WrappedDekDetails> wdeks() {
        ColumnFamilyHandle wdekCf = this.rocksDbStorage.getColumnFamilyHandle("wdek");
        try (RocksIterator iterator = this.rocksDbStorage.newIterator(wdekCf);){
            iterator.seekToFirst();
            ImmutableList.Builder wdekDetailsBuilder = ImmutableList.builder();
            while (iterator.isValid()) {
                String key = new String(iterator.key(), StandardCharsets.UTF_8);
                if (key.startsWith("wdeks/") && !key.endsWith("/current")) {
                    try {
                        WrappedDekDetails wdekDetails = (WrappedDekDetails)Jackson.readValue((byte[])iterator.value(), WrappedDekDetails.class);
                        wdekDetailsBuilder.add((Object)wdekDetails);
                    }
                    catch (JsonParseException | JsonMappingException e) {
                        logger.warn("Failed to read WDEK for key: {}", (Object)key, (Object)e);
                    }
                }
                iterator.next();
            }
            ImmutableList immutableList = wdekDetailsBuilder.build();
            return immutableList;
        }
    }

    SecretKey getDek(String projectName, String repoName, int version) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        return this.getDek0(projectName, repoName, version);
    }

    private SecretKeySpec getDek0(String projectName, String repoName, int version) {
        WrappedDekDetails wdekDetails;
        byte[] wdekDetailsBytes;
        try {
            wdekDetailsBytes = this.rocksDbStorage.get("wdek", RepositoryEncryptionStorage.repoWdekDbKey(projectName, repoName, version));
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get WDEK of " + RepositoryEncryptionStorage.projectRepoVersion(projectName, repoName, version), e);
        }
        if (wdekDetailsBytes == null) {
            throw new EncryptionStorageException("WDEK of " + RepositoryEncryptionStorage.projectRepoVersion(projectName, repoName, version) + " does not exist");
        }
        try {
            wdekDetails = (WrappedDekDetails)Jackson.readValue((byte[])wdekDetailsBytes, WrappedDekDetails.class);
        }
        catch (JsonParseException | JsonMappingException e) {
            throw new EncryptionStorageException("Failed to read WDEK of " + RepositoryEncryptionStorage.projectRepoVersion(projectName, repoName, version), e);
        }
        return this.blockingUnwrap(projectName, repoName, version, wdekDetails);
    }

    private SecretKeySpec blockingUnwrap(String projectName, String repoName, int version, WrappedDekDetails wdekDetails) {
        byte[] key;
        try {
            key = this.keyWrapper.unwrap(wdekDetails.wrappedDek(), wdekDetails.kekId()).get(10L, TimeUnit.SECONDS);
        }
        catch (Throwable t) {
            throw new EncryptionStorageException("Failed to unwrap WDEK of " + RepositoryEncryptionStorage.projectRepoVersion(projectName, repoName, version), t);
        }
        return AesGcmSivCipher.aesSecretKey(key);
    }

    SecretKeyWithVersion getCurrentDek(String projectName, String repoName) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        return this.getCurrentDek0(projectName, repoName);
    }

    private SecretKeyWithVersion getCurrentDek0(String projectName, String repoName) {
        int version;
        try {
            byte[] versionBytes = this.rocksDbStorage.get("wdek", RepositoryEncryptionStorage.repoCurrentWdekDbKey(projectName, repoName));
            if (versionBytes == null) {
                throw new EncryptionStorageException("Current WDEK of " + RepositoryEncryptionStorage.projectRepo(projectName, repoName) + " does not exist");
            }
            version = Ints.fromByteArray((byte[])versionBytes);
        }
        catch (IllegalArgumentException e) {
            throw new EncryptionStorageException("Failed to parse the current WDEK version of " + RepositoryEncryptionStorage.projectRepo(projectName, repoName), e);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get the current WDEK of " + RepositoryEncryptionStorage.projectRepo(projectName, repoName), e);
        }
        return new SecretKeyWithVersion(this.getDek(projectName, repoName, version), version);
    }

    void storeWdek(WrappedDekDetails wdekDetails) {
        this.storeWdek(wdekDetails, false);
    }

    private void storeWdek(WrappedDekDetails wdekDetails, boolean rotate) {
        byte[] wdekBytes;
        Objects.requireNonNull(wdekDetails, "wdekDetails");
        String projectName = wdekDetails.projectName();
        String repoName = wdekDetails.repoName();
        int version = wdekDetails.dekVersion();
        if (rotate) {
            SecretKeyWithVersion currentDek = this.getCurrentDek(projectName, repoName);
            if (wdekDetails.dekVersion() != currentDek.version() + 1) {
                throw new EncryptionStorageException("The WDEK version to rotate must be exactly one greater than the current version. Current version: " + currentDek.version() + ", rotated version: " + wdekDetails.dekVersion());
            }
        }
        byte[] wdekKeyBytes = RepositoryEncryptionStorage.repoWdekDbKey(projectName, repoName, version);
        try {
            byte[] existingWdekBytes = this.rocksDbStorage.get("wdek", wdekKeyBytes);
            if (existingWdekBytes != null) {
                throw new EncryptionEntryExistsException("WDEK of " + RepositoryEncryptionStorage.projectRepoVersion(projectName, repoName, version) + " already exists");
            }
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to check the existence of WDEK for " + RepositoryEncryptionStorage.projectRepoVersion(projectName, repoName, version), e);
        }
        SecretKeySpec unwrap = this.blockingUnwrap(projectName, repoName, version, wdekDetails);
        try {
            wdekBytes = Jackson.writeValueAsBytes((Object)wdekDetails);
        }
        catch (JsonProcessingException e) {
            throw new EncryptionStorageException("Failed to serialize WDEK of " + RepositoryEncryptionStorage.projectRepoVersion(projectName, repoName, version), e);
        }
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            ColumnFamilyHandle wdekCf = this.rocksDbStorage.getColumnFamilyHandle("wdek");
            writeBatch.put(wdekCf, wdekKeyBytes, wdekBytes);
            writeBatch.put(wdekCf, RepositoryEncryptionStorage.repoCurrentWdekDbKey(projectName, repoName), Ints.toByteArray((int)version));
            this.rocksDbStorage.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to store WDEK of " + RepositoryEncryptionStorage.projectRepoVersion(projectName, repoName, version), e);
        }
        this.notifyCurrentDekListener(projectName, repoName, new SecretKeyWithVersion(unwrap, version));
    }

    void rotateWdek(WrappedDekDetails wdekDetails) {
        Objects.requireNonNull(wdekDetails, "wdekDetails");
        this.storeWdek(wdekDetails, true);
    }

    void removeWdek(String projectName, String repoName, int version, boolean removeCurrent) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            ColumnFamilyHandle wdekCf = this.rocksDbStorage.getColumnFamilyHandle("wdek");
            writeBatch.delete(wdekCf, RepositoryEncryptionStorage.repoWdekDbKey(projectName, repoName, version));
            if (removeCurrent) {
                writeBatch.delete(wdekCf, RepositoryEncryptionStorage.repoCurrentWdekDbKey(projectName, repoName));
            }
            this.rocksDbStorage.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to remove WDEK of " + RepositoryEncryptionStorage.projectRepoVersion(projectName, repoName, version), e);
        }
        if (removeCurrent) {
            this.removeCurrentDekListener(projectName, repoName);
        }
    }

    @Nullable
    byte[] getObject(byte[] key, byte[] metadataKey) {
        Objects.requireNonNull(key, "key");
        Objects.requireNonNull(metadataKey, "metadataKey");
        try {
            return this.rocksDbStorage.get("encrypted_object", key);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get object. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    @Nullable
    byte[] getObjectId(byte[] key, byte[] metadataKey) {
        Objects.requireNonNull(key, "key");
        Objects.requireNonNull(metadataKey, "metadataKey");
        try {
            return this.rocksDbStorage.get("encrypted_object_id", key);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get object ID. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    @Nullable
    byte[] getMetadata(byte[] metadataKey) {
        Objects.requireNonNull(metadataKey, "metadataKey");
        try {
            return this.rocksDbStorage.get("encryption_metadata", metadataKey);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get metadata. key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    void putObject(byte[] metadataKey, byte[] metadataValue, byte[] key, byte[] value) {
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            writeBatch.put(this.rocksDbStorage.getColumnFamilyHandle("encryption_metadata"), metadataKey, metadataValue);
            writeBatch.put(this.rocksDbStorage.getColumnFamilyHandle("encrypted_object"), key, value);
            this.rocksDbStorage.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to write object key-value with metadata. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    void putObjectId(byte[] metadataKey, byte[] metadataValue, byte[] key, byte[] value, @Nullable byte[] previousKeyToRemove) {
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            writeBatch.put(this.rocksDbStorage.getColumnFamilyHandle("encryption_metadata"), metadataKey, metadataValue);
            ColumnFamilyHandle objectIdcolumnFamilyHandle = this.rocksDbStorage.getColumnFamilyHandle("encrypted_object_id");
            writeBatch.put(objectIdcolumnFamilyHandle, key, value);
            if (previousKeyToRemove != null) {
                writeBatch.delete(objectIdcolumnFamilyHandle, previousKeyToRemove);
            }
            this.rocksDbStorage.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to write object key-value with metadata. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    boolean containsMetadata(byte[] key) {
        Objects.requireNonNull(key, "key");
        try {
            byte[] value = this.rocksDbStorage.get("encryption_metadata", key);
            return value != null;
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to check existence of metadata. key: " + new String(key, StandardCharsets.UTF_8), e);
        }
    }

    void deleteObjectId(byte[] metadataKey, byte[] key) {
        Objects.requireNonNull(metadataKey, "metadataKey");
        Objects.requireNonNull(key, "key");
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            writeBatch.delete(this.rocksDbStorage.getColumnFamilyHandle("encryption_metadata"), metadataKey);
            writeBatch.delete(this.rocksDbStorage.getColumnFamilyHandle("encrypted_object_id"), key);
            this.rocksDbStorage.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to delete object ID key-value with metadata. metadata key: " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
    }

    void reencryptRepositoryData(String projectName, String repoName) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        logger.info("Re-encrypting data for repository: {}/{}", (Object)projectName, (Object)repoName);
        SecretKeyWithVersion currentDek = this.getCurrentDek(projectName, repoName);
        int newKeyVersion = currentDek.version();
        SecretKey newDek = currentDek.secretKey();
        String projectRepoPrefix = RepositoryEncryptionStorage.projectRepo(projectName, repoName) + "/";
        byte[] projectRepoPrefixBytes = projectRepoPrefix.getBytes(StandardCharsets.UTF_8);
        byte[] objectKeyPrefixBytes = (projectRepoPrefix + "objs/").getBytes(StandardCharsets.UTF_8);
        byte[] refsKeyPrefixBytes = (projectRepoPrefix + "refs/").getBytes(StandardCharsets.UTF_8);
        byte[] headKeyBytes = (projectRepoPrefix + "HEAD").getBytes(StandardCharsets.UTF_8);
        byte[] rev2ShaPrefixBytes = (projectRepoPrefix + "rev2sha/").getBytes(StandardCharsets.UTF_8);
        int totalReencryptedCount = 0;
        int operationsInCurrentBatch = 0;
        ColumnFamilyHandle metadataColumnFamilyHandle = this.rocksDbStorage.getColumnFamilyHandle("encryption_metadata");
        ColumnFamilyHandle encryptedObjectIdHandle = this.rocksDbStorage.getColumnFamilyHandle("encrypted_object_id");
        HashMap<Integer, SecretKey> deks = new HashMap<Integer, SecretKey>();
        Snapshot snapshot = this.rocksDbStorage.getSnapshot();
        try (ReadOptions readOptions = new ReadOptions().setSnapshot(snapshot);
             WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();
             RocksIterator iterator = this.rocksDbStorage.newIterator(metadataColumnFamilyHandle, readOptions);){
            byte[] metadataKey;
            iterator.seek(projectRepoPrefixBytes);
            while (iterator.isValid() && RepositoryEncryptionStorage.startsWith(metadataKey = iterator.key(), projectRepoPrefixBytes)) {
                byte[] idPart;
                byte[] metadataValue = iterator.value();
                if (RepositoryEncryptionStorage.startsWith(metadataKey, objectKeyPrefixBytes)) {
                    if (metadataKey.length == objectKeyPrefixBytes.length + 20) {
                        idPart = null;
                        if (metadataValue != null) {
                            byte[] objectDekBytes;
                            GitObjectMetadata gitObjectMetadata = GitObjectMetadata.fromBytes(metadataValue);
                            if (gitObjectMetadata.keyVersion() == newKeyVersion) {
                                iterator.next();
                                continue;
                            }
                            SecretKey oldRepoDek = this.getCachedDek(projectName, repoName, deks, gitObjectMetadata.keyVersion());
                            try {
                                objectDekBytes = AesGcmSivCipher.decrypt(oldRepoDek, gitObjectMetadata.nonce(), gitObjectMetadata.objectWdek());
                            }
                            catch (Exception e) {
                                throw new EncryptionStorageException("Failed to unwrap object DEK for " + new String(metadataKey, StandardCharsets.UTF_8), e);
                            }
                            byte[] newWrappedObjectDek = AesGcmSivCipher.encrypt(newDek, gitObjectMetadata.nonce(), objectDekBytes);
                            GitObjectMetadata newGitObjectMetadata = GitObjectMetadata.of(newKeyVersion, gitObjectMetadata.nonce(), gitObjectMetadata.type(), newWrappedObjectDek);
                            byte[] newMetadataValue = newGitObjectMetadata.toBytes();
                            writeBatch.put(metadataColumnFamilyHandle, metadataKey, newMetadataValue);
                            ++operationsInCurrentBatch;
                            ++totalReencryptedCount;
                        } else {
                            logger.warn("Invalid metadata value for object key: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                        }
                    } else {
                        logger.warn("Invalid object metadata key length: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                    }
                } else if (RepositoryEncryptionStorage.startsWith(metadataKey, rev2ShaPrefixBytes)) {
                    if (metadataKey.length == rev2ShaPrefixBytes.length + 4) {
                        idPart = Arrays.copyOfRange(metadataKey, rev2ShaPrefixBytes.length, metadataKey.length);
                        if (this.reencryptObjectIdEntry(projectName, repoName, metadataKey, metadataValue, idPart, newKeyVersion, newDek, deks, metadataColumnFamilyHandle, encryptedObjectIdHandle, writeBatch, "rev2sha")) {
                            ++operationsInCurrentBatch;
                            ++totalReencryptedCount;
                        }
                    } else {
                        logger.warn("Invalid rev2sha metadata key length: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                    }
                } else if (RepositoryEncryptionStorage.startsWith(metadataKey, headKeyBytes) || RepositoryEncryptionStorage.startsWith(metadataKey, refsKeyPrefixBytes)) {
                    idPart = Arrays.copyOfRange(metadataKey, projectRepoPrefixBytes.length, metadataKey.length);
                    if (this.reencryptObjectIdEntry(projectName, repoName, metadataKey, metadataValue, idPart, newKeyVersion, newDek, deks, metadataColumnFamilyHandle, encryptedObjectIdHandle, writeBatch, "ref")) {
                        ++operationsInCurrentBatch;
                        ++totalReencryptedCount;
                    }
                } else {
                    logger.warn("Unknown metadata key pattern for prefix {}: {}", (Object)projectRepoPrefix, (Object)new String(metadataKey, StandardCharsets.UTF_8));
                }
                operationsInCurrentBatch = this.executeBatchIfNeeded(writeBatch, writeOptions, operationsInCurrentBatch, totalReencryptedCount, projectName, repoName, "Re-encrypted");
                iterator.next();
            }
            this.executeFinalBatch(writeBatch, writeOptions, operationsInCurrentBatch, totalReencryptedCount, projectName, repoName, "Re-encrypted");
            if (totalReencryptedCount > 0) {
                logger.info("Successfully re-encrypted a total of {} entries for repository {}/{} with DEK version {}", new Object[]{totalReencryptedCount, projectName, repoName, newKeyVersion});
            } else {
                logger.info("No data needed re-encryption for repository {}/{}", (Object)projectName, (Object)repoName);
            }
        }
        catch (EncryptionStorageException | RocksDBException e) {
            throw new EncryptionStorageException("Failed to re-encrypt repository data for " + RepositoryEncryptionStorage.projectRepo(projectName, repoName), e);
        }
        catch (Exception e) {
            throw new EncryptionStorageException("Unexpected error during repository data re-encryption for " + RepositoryEncryptionStorage.projectRepo(projectName, repoName), e);
        }
        finally {
            this.rocksDbStorage.releaseSnapshot(snapshot);
        }
    }

    private SecretKey getCachedDek(String projectName, String repoName, HashMap<Integer, SecretKey> deks, int keyVersion) {
        return deks.computeIfAbsent(keyVersion, k -> this.getDek(projectName, repoName, (int)k));
    }

    void deleteRepositoryData(String projectName, String repoName) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        logger.info("Deleting encrypted data for repository: {}/{}", (Object)projectName, (Object)repoName);
        String projectRepoPrefix = RepositoryEncryptionStorage.projectRepo(projectName, repoName) + "/";
        byte[] projectRepoPrefixBytes = projectRepoPrefix.getBytes(StandardCharsets.UTF_8);
        byte[] objectKeyPrefixBytes = (projectRepoPrefix + "objs/").getBytes(StandardCharsets.UTF_8);
        byte[] refsKeyPrefixBytes = (projectRepoPrefix + "refs/").getBytes(StandardCharsets.UTF_8);
        byte[] headKeyBytes = (projectRepoPrefix + "HEAD").getBytes(StandardCharsets.UTF_8);
        byte[] rev2ShaPrefixBytes = (projectRepoPrefix + "rev2sha/").getBytes(StandardCharsets.UTF_8);
        int totalDeletedCount = 0;
        int operationsInCurrentBatch = 0;
        ColumnFamilyHandle metadataColumnFamilyHandle = this.rocksDbStorage.getColumnFamilyHandle("encryption_metadata");
        ColumnFamilyHandle encryptedObjectHandle = this.rocksDbStorage.getColumnFamilyHandle("encrypted_object");
        ColumnFamilyHandle encryptedObjectIdHandle = this.rocksDbStorage.getColumnFamilyHandle("encrypted_object_id");
        HashMap<Integer, SecretKey> deks = new HashMap<Integer, SecretKey>();
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();
             RocksIterator iterator = this.rocksDbStorage.newIterator(metadataColumnFamilyHandle);){
            byte[] metadataKey;
            iterator.seek(projectRepoPrefixBytes);
            while (iterator.isValid() && RepositoryEncryptionStorage.startsWith(metadataKey = iterator.key(), projectRepoPrefixBytes)) {
                SecretKey dek;
                byte[] nonce;
                byte[] key;
                byte[] idPart;
                byte[] metadataValue = iterator.value();
                if (RepositoryEncryptionStorage.startsWith(metadataKey, objectKeyPrefixBytes)) {
                    if (metadataKey.length == objectKeyPrefixBytes.length + 20) {
                        idPart = Arrays.copyOfRange(metadataKey, objectKeyPrefixBytes.length, metadataKey.length);
                        if (metadataValue != null) {
                            SecretKeySpec objectDek;
                            GitObjectMetadata gitObjectMetadata = GitObjectMetadata.fromBytes(metadataValue);
                            try {
                                objectDek = gitObjectMetadata.objectDek(this.getCachedDek(projectName, repoName, deks, gitObjectMetadata.keyVersion()));
                            }
                            catch (Exception e) {
                                throw new EncryptionStorageException("Failed to get object dek for " + new String(metadataKey, StandardCharsets.UTF_8), e);
                            }
                            key = AesGcmSivCipher.encrypt(objectDek, gitObjectMetadata.nonce(), idPart);
                            writeBatch.delete(encryptedObjectHandle, key);
                            ++operationsInCurrentBatch;
                            ++totalDeletedCount;
                        } else {
                            logger.warn("Invalid metadata value for object key: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                        }
                    } else {
                        logger.warn("Invalid object metadata key length: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                    }
                } else if (RepositoryEncryptionStorage.startsWith(metadataKey, rev2ShaPrefixBytes)) {
                    if (metadataKey.length == rev2ShaPrefixBytes.length + 4) {
                        idPart = Arrays.copyOfRange(metadataKey, rev2ShaPrefixBytes.length, metadataKey.length);
                        if (metadataValue != null && metadataValue.length == 16) {
                            nonce = new byte[12];
                            System.arraycopy(metadataValue, 4, nonce, 0, 12);
                            dek = this.getCachedDek(projectName, repoName, deks, Ints.fromByteArray((byte[])metadataValue));
                            key = AesGcmSivCipher.encrypt(dek, nonce, idPart);
                            writeBatch.delete(encryptedObjectIdHandle, key);
                            ++operationsInCurrentBatch;
                            ++totalDeletedCount;
                        } else {
                            logger.warn("Invalid nonce (metadata value) for rev2sha key: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                        }
                    } else {
                        logger.warn("Invalid rev2sha metadata key length: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                    }
                } else if (RepositoryEncryptionStorage.startsWith(metadataKey, headKeyBytes) || RepositoryEncryptionStorage.startsWith(metadataKey, refsKeyPrefixBytes)) {
                    idPart = Arrays.copyOfRange(metadataKey, projectRepoPrefixBytes.length, metadataKey.length);
                    if (metadataValue != null && metadataValue.length == 16) {
                        nonce = new byte[12];
                        System.arraycopy(metadataValue, 4, nonce, 0, 12);
                        dek = this.getCachedDek(projectName, repoName, deks, Ints.fromByteArray((byte[])metadataValue));
                        key = AesGcmSivCipher.encrypt(dek, nonce, idPart);
                        writeBatch.delete(encryptedObjectIdHandle, key);
                        ++operationsInCurrentBatch;
                        ++totalDeletedCount;
                    } else {
                        logger.warn("Invalid nonce (metadata value) for ref key: {}", (Object)new String(metadataKey, StandardCharsets.UTF_8));
                    }
                } else {
                    logger.warn("Unknown metadata key pattern for prefix {}: {}", (Object)projectRepoPrefix, (Object)new String(metadataKey, StandardCharsets.UTF_8));
                }
                writeBatch.delete(metadataColumnFamilyHandle, metadataKey);
                ++operationsInCurrentBatch;
                operationsInCurrentBatch = this.executeBatchIfNeeded(writeBatch, writeOptions, operationsInCurrentBatch, ++totalDeletedCount, projectName, repoName, "Deleted");
                iterator.next();
            }
            this.executeFinalBatch(writeBatch, writeOptions, operationsInCurrentBatch, totalDeletedCount, projectName, repoName, "Deleted");
            if (totalDeletedCount > 0) {
                logger.info("Successfully deleted a total of {} entries for repository {}/{}", new Object[]{totalDeletedCount, projectName, repoName});
            } else {
                logger.info("No data found for repository {}/{}", (Object)projectName, (Object)repoName);
            }
        }
        catch (EncryptionStorageException | RocksDBException e) {
            throw new EncryptionStorageException("Failed to delete repository data for " + RepositoryEncryptionStorage.projectRepo(projectName, repoName), e);
        }
        catch (Exception e) {
            throw new EncryptionStorageException("Unexpected error during repository data deletion for " + RepositoryEncryptionStorage.projectRepo(projectName, repoName), e);
        }
        logger.info("Removing WDEKs for repository: {}/{}", (Object)projectName, (Object)repoName);
        ColumnFamilyHandle wdekColumnFamilyHandle = this.rocksDbStorage.getColumnFamilyHandle("wdek");
        byte[] wdekPrefixBytes = ("wdeks/" + projectName + "/" + repoName + "/").getBytes(StandardCharsets.UTF_8);
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();
             RocksIterator iterator = this.rocksDbStorage.newIterator(wdekColumnFamilyHandle);){
            byte[] wdekKey;
            iterator.seek(wdekPrefixBytes);
            while (iterator.isValid() && RepositoryEncryptionStorage.startsWith(wdekKey = iterator.key(), wdekPrefixBytes)) {
                writeBatch.delete(wdekColumnFamilyHandle, wdekKey);
                iterator.next();
            }
            writeOptions.setSync(true);
            this.rocksDbStorage.write(writeOptions, writeBatch);
            logger.info("Deleted WDEKs for repository {}/{}", (Object)projectName, (Object)repoName);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to remove WDEKs for repository " + RepositoryEncryptionStorage.projectRepo(projectName, repoName), e);
        }
        this.removeCurrentDekListener(projectName, repoName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addCurrentDekListener(String projectName, String repoName, Consumer<SecretKeyWithVersion> listener) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        Objects.requireNonNull(listener, "listener");
        String projectRepoKey = RepositoryEncryptionStorage.projectRepo(projectName, repoName);
        this.lock.lock();
        try {
            if (this.currentDekListeners.containsKey(projectRepoKey)) {
                throw new IllegalStateException("A current DEK listener is already registered for " + projectRepoKey);
            }
            this.currentDekListeners.put(projectRepoKey, listener);
            SecretKeyWithVersion currentDek = this.getCurrentDek0(projectName, repoName);
            listener.accept(currentDek);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeCurrentDekListener(String projectName, String repoName) {
        Objects.requireNonNull(projectName, "projectName");
        Objects.requireNonNull(repoName, "repoName");
        String projectRepoKey = RepositoryEncryptionStorage.projectRepo(projectName, repoName);
        this.lock.lock();
        try {
            this.currentDekListeners.remove(projectRepoKey);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void notifyCurrentDekListener(String projectName, String repoName, SecretKeyWithVersion newDek) {
        String projectRepoKey = RepositoryEncryptionStorage.projectRepo(projectName, repoName);
        this.lock.lock();
        try {
            Consumer<SecretKeyWithVersion> listener = this.currentDekListeners.get(projectRepoKey);
            if (listener != null) {
                listener.accept(newDek);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private static String projectRepo(String projectName, String repoName) {
        return projectName + "/" + repoName;
    }

    private static String projectRepoVersion(String projectName, String repoName, int version) {
        return projectName + "/" + repoName + "/" + version;
    }

    private static byte[] repoWdekDbKey(String projectName, String repoName, int version) {
        return ("wdeks/" + projectName + "/" + repoName + "/" + version).getBytes(StandardCharsets.UTF_8);
    }

    private static byte[] repoCurrentWdekDbKey(String projectName, String repoName) {
        return ("wdeks/" + projectName + "/" + repoName + "/current").getBytes(StandardCharsets.UTF_8);
    }

    CompletableFuture<Void> rewrapAllWdeks(Executor executor) {
        List<WrappedDekDetails> allWdeks = this.wdeks();
        if (allWdeks.isEmpty()) {
            logger.info("No WDEKs to rewrap");
            return CompletableFuture.completedFuture(null);
        }
        logger.info("Starting to rewrap WDEKs...");
        ArrayList<CompletionStage> rewrapFutures = new ArrayList<CompletionStage>();
        for (WrappedDekDetails wdekDetails : allWdeks) {
            String oldKekId = wdekDetails.kekId();
            logger.info("Re-wrapping WDEK for {}/{} version {} from KEK {} to {}", new Object[]{wdekDetails.projectName(), wdekDetails.repoName(), wdekDetails.dekVersion(), oldKekId, this.kekId});
            CompletionStage rewrapFuture = this.keyWrapper.rewrap(wdekDetails.wrappedDek(), oldKekId, this.kekId).handle((wrappedDek, cause) -> {
                if (cause != null) {
                    logger.warn("Failed to rewrap WDEK for {}", (Object)RepositoryEncryptionStorage.projectRepoVersion(wdekDetails.projectName(), wdekDetails.repoName(), wdekDetails.dekVersion()), cause);
                    return null;
                }
                if (oldKekId.equals(this.kekId) && wdekDetails.wrappedDek().equals(wrappedDek)) {
                    logger.info("WDEK for {} is already wrapped with the target KEK {}. Skipping rewrap.", (Object)RepositoryEncryptionStorage.projectRepoVersion(wdekDetails.projectName(), wdekDetails.repoName(), wdekDetails.dekVersion()), (Object)this.kekId);
                    return null;
                }
                return new WrappedDekDetails((String)wrappedDek, wdekDetails.dekVersion(), this.kekId, wdekDetails.creationInstant(), wdekDetails.projectName(), wdekDetails.repoName());
            });
            rewrapFutures.add(rewrapFuture);
        }
        return CompletableFuture.allOf(rewrapFutures.toArray(new CompletableFuture[0])).thenAcceptAsync(unused -> {
            List collected = (List)rewrapFutures.stream().map(CompletableFuture::join).filter(Objects::nonNull).collect(ImmutableList.toImmutableList());
            logger.info("Storing {} re-wrapped WDEKs", (Object)collected.size());
            ColumnFamilyHandle wdekCf = this.rocksDbStorage.getColumnFamilyHandle("wdek");
            int failedCount = 0;
            try (WriteBatch writeBatch = new WriteBatch();
                 WriteOptions writeOptions = new WriteOptions();){
                writeOptions.setSync(true);
                for (WrappedDekDetails newWdekDetails : collected) {
                    byte[] wdekBytes;
                    try {
                        wdekBytes = Jackson.writeValueAsBytes((Object)newWdekDetails);
                    }
                    catch (JsonProcessingException e) {
                        ++failedCount;
                        logger.warn("Failed to serialize re-wrapped WDEK for {}", (Object)RepositoryEncryptionStorage.projectRepoVersion(newWdekDetails.projectName(), newWdekDetails.repoName(), newWdekDetails.dekVersion()), (Object)e);
                        continue;
                    }
                    byte[] wdekKeyBytes = RepositoryEncryptionStorage.repoWdekDbKey(newWdekDetails.projectName(), newWdekDetails.repoName(), newWdekDetails.dekVersion());
                    writeBatch.put(wdekCf, wdekKeyBytes, wdekBytes);
                }
                this.rocksDbStorage.write(writeOptions, writeBatch);
                logger.info("Successfully re-wrapped {} WDEKs", (Object)(collected.size() - failedCount));
            }
            catch (RocksDBException e) {
                throw new EncryptionStorageException("Failed to store re-wrapped WDEKs", e);
            }
        }, executor);
    }

    private int executeBatchIfNeeded(WriteBatch writeBatch, WriteOptions writeOptions, int currentBatchSize, int totalCount, String projectName, String repoName, String operation) {
        if (currentBatchSize >= 1000 && writeBatch.count() > 0) {
            try {
                writeOptions.setSync(true);
                this.rocksDbStorage.write(writeOptions, writeBatch);
                logger.info("{} {} entries for repository {}/{}. Total entries processed so far: {}.", new Object[]{operation, currentBatchSize, projectName, repoName, totalCount});
                writeBatch.clear();
                return 0;
            }
            catch (RocksDBException e) {
                throw new EncryptionStorageException("Failed to write batch for " + RepositoryEncryptionStorage.projectRepo(projectName, repoName), e);
            }
        }
        return currentBatchSize;
    }

    private void executeFinalBatch(WriteBatch writeBatch, WriteOptions writeOptions, int currentBatchSize, int totalCount, String projectName, String repoName, String operation) {
        if (currentBatchSize > 0) {
            try {
                writeOptions.setSync(true);
                this.rocksDbStorage.write(writeOptions, writeBatch);
                logger.info("{} {} entries for repository {}/{}. Total entries processed so far: {}.", new Object[]{operation, currentBatchSize, projectName, repoName, totalCount});
            }
            catch (RocksDBException e) {
                throw new EncryptionStorageException("Failed to write final batch for " + RepositoryEncryptionStorage.projectRepo(projectName, repoName), e);
            }
        }
    }

    private boolean reencryptObjectIdEntry(String projectName, String repoName, byte[] metadataKey, @Nullable byte[] metadataValue, byte[] idPart, int newKeyVersion, SecretKey newDek, HashMap<Integer, SecretKey> deks, ColumnFamilyHandle metadataColumnFamilyHandle, ColumnFamilyHandle encryptedObjectIdHandle, WriteBatch writeBatch, String entryType) {
        byte[] newEncryptedData;
        byte[] newEncryptedKey;
        byte[] decryptedData;
        byte[] encryptedData;
        byte[] oldEncryptedKey;
        if (metadataValue == null || metadataValue.length != 16) {
            logger.warn("Invalid metadata value for {} key: {}", (Object)entryType, (Object)new String(metadataKey, StandardCharsets.UTF_8));
            return false;
        }
        int oldKeyVersion = Ints.fromByteArray((byte[])metadataValue);
        if (oldKeyVersion == newKeyVersion) {
            return false;
        }
        byte[] oldNonce = new byte[12];
        System.arraycopy(metadataValue, 4, oldNonce, 0, 12);
        SecretKey oldDek = this.getCachedDek(projectName, repoName, deks, oldKeyVersion);
        try {
            oldEncryptedKey = AesGcmSivCipher.encrypt(oldDek, oldNonce, idPart);
        }
        catch (Exception e) {
            throw new EncryptionStorageException("Failed to encrypt old key for " + entryType + " " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
        try {
            encryptedData = this.rocksDbStorage.get("encrypted_object_id", oldEncryptedKey);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get encrypted " + entryType + " for " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
        if (encryptedData == null) {
            logger.warn("Encrypted {} data not found for: {}", (Object)entryType, (Object)new String(metadataKey, StandardCharsets.UTF_8));
            return false;
        }
        try {
            decryptedData = AesGcmSivCipher.decrypt(oldDek, oldNonce, encryptedData);
        }
        catch (Exception e) {
            throw new EncryptionStorageException("Failed to decrypt " + entryType + " data for " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
        byte[] newNonce = AesGcmSivCipher.generateNonce();
        byte[] newMetadataValue = new byte[16];
        System.arraycopy(Ints.toByteArray((int)newKeyVersion), 0, newMetadataValue, 0, 4);
        System.arraycopy(newNonce, 0, newMetadataValue, 4, 12);
        try {
            newEncryptedKey = AesGcmSivCipher.encrypt(newDek, newNonce, idPart);
        }
        catch (Exception e) {
            throw new EncryptionStorageException("Failed to encrypt new key for " + entryType + " " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
        try {
            newEncryptedData = AesGcmSivCipher.encrypt(newDek, newNonce, decryptedData);
        }
        catch (Exception e) {
            throw new EncryptionStorageException("Failed to encrypt new " + entryType + " data for " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
        try {
            writeBatch.put(metadataColumnFamilyHandle, metadataKey, newMetadataValue);
            writeBatch.delete(encryptedObjectIdHandle, oldEncryptedKey);
            writeBatch.put(encryptedObjectIdHandle, newEncryptedKey, newEncryptedData);
            if (logger.isTraceEnabled()) {
                logger.trace("Re-encrypted {} entry: {} (oldVersion={}, newVersion={})", new Object[]{entryType, new String(metadataKey, StandardCharsets.UTF_8), oldKeyVersion, newKeyVersion});
            }
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to update encrypted " + entryType + " for " + new String(metadataKey, StandardCharsets.UTF_8), e);
        }
        return true;
    }

    private static boolean startsWith(byte[] array, byte[] prefix) {
        if (array.length < prefix.length) {
            return false;
        }
        for (int i = 0; i < prefix.length; ++i) {
            if (array[i] == prefix[i]) continue;
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Map<String, Map<String, byte[]>> getAllData() {
        HashMap<String, Map<String, byte[]>> allData = new HashMap<String, Map<String, byte[]>>();
        Snapshot snapshot = this.rocksDbStorage.getSnapshot();
        try (ReadOptions readOptions = new ReadOptions().setSnapshot(snapshot);){
            for (Map.Entry<String, ColumnFamilyHandle> entry : this.rocksDbStorage.getAllColumnFamilyHandles().entrySet()) {
                String cfName = entry.getKey();
                ColumnFamilyHandle cfHandle = entry.getValue();
                HashMap<String, byte[]> cfData = new HashMap<String, byte[]>();
                try (RocksIterator iterator = this.rocksDbStorage.newIterator(cfHandle, readOptions);){
                    iterator.seekToFirst();
                    while (iterator.isValid()) {
                        String key = new String(iterator.key(), StandardCharsets.UTF_8);
                        cfData.put(key, iterator.value());
                        iterator.next();
                    }
                }
                allData.put(cfName, cfData);
            }
        }
        finally {
            this.rocksDbStorage.releaseSnapshot(snapshot);
        }
        return allData;
    }
}

