/*
 * Decompiled with CFR 0.152.
 */
package com.linecorp.centraldogma.server.storage.encryption;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.linecorp.armeria.internal.common.util.ReentrantShortLock;
import com.linecorp.centraldogma.internal.Jackson;
import com.linecorp.centraldogma.internal.shaded.guava.collect.ImmutableList;
import com.linecorp.centraldogma.internal.shaded.guava.primitives.Ints;
import com.linecorp.centraldogma.server.auth.SessionKey;
import com.linecorp.centraldogma.server.auth.SessionMasterKey;
import com.linecorp.centraldogma.server.internal.storage.AesGcmSivCipher;
import com.linecorp.centraldogma.server.storage.encryption.EncryptionEntryExistsException;
import com.linecorp.centraldogma.server.storage.encryption.EncryptionEntryNoExistException;
import com.linecorp.centraldogma.server.storage.encryption.EncryptionStorageException;
import com.linecorp.centraldogma.server.storage.encryption.KeyWrapper;
import com.linecorp.centraldogma.server.storage.encryption.RocksDBStorage;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class SessionKeyStorage {
    private static final Logger logger = LoggerFactory.getLogger(SessionKeyStorage.class);
    private final RocksDBStorage rocksDbStorage;
    private final KeyWrapper keyWrapper;
    private final String kekId;
    private final ReentrantShortLock lock = new ReentrantShortLock();
    @Nullable
    @GuardedBy(value="lock")
    private SessionKey currentSessionKey;
    private final Map<Integer, CompletableFuture<SessionKey>> sessionKeys = new ConcurrentHashMap<Integer, CompletableFuture<SessionKey>>();
    @GuardedBy(value="lock")
    private final List<Consumer<SessionKey>> listeners = new ArrayList<Consumer<SessionKey>>();

    SessionKeyStorage(RocksDBStorage rocksDbStorage, KeyWrapper keyWrapper, String kekId) {
        this.rocksDbStorage = Objects.requireNonNull(rocksDbStorage, "rocksDbStorage");
        this.keyWrapper = Objects.requireNonNull(keyWrapper, "keyWrapper");
        this.kekId = Objects.requireNonNull(kekId, "kekId");
    }

    CompletableFuture<SessionMasterKey> generateSessionMasterKey(int version) {
        byte[] dek = AesGcmSivCipher.generateAes256Key();
        return this.keyWrapper.wrap(dek, this.kekId).thenApply(wrappedMasterKey -> {
            byte[] salt = AesGcmSivCipher.generateAes256Key();
            return new SessionMasterKey((String)wrappedMasterKey, version, Base64.getEncoder().encodeToString(salt), this.kekId, Instant.now());
        });
    }

    void storeSessionMasterKey(SessionMasterKey sessionMasterKey) {
        this.storeSessionMasterKey(sessionMasterKey, false);
    }

    void storeSessionMasterKey(SessionMasterKey sessionMasterKey, boolean rotate) {
        byte[] sessionMasterKeyBytes;
        SessionMasterKey currentSessionMasterKey;
        Objects.requireNonNull(sessionMasterKey, "sessionMasterKey");
        int version = sessionMasterKey.version();
        if (rotate && (currentSessionMasterKey = this.getCurrentSessionMasterKey()).version() + 1 != version) {
            throw new IllegalArgumentException("The version of the new session master key (" + version + ") must be exactly one greater than the current version (" + currentSessionMasterKey.version() + ")");
        }
        byte[] masterKeyKey = SessionKeyStorage.sessionMasterKeyKey(version);
        try {
            byte[] existing = this.rocksDbStorage.get("wdek", masterKeyKey);
            if (existing != null) {
                throw new EncryptionEntryExistsException("Session master key of version " + version + " already exists");
            }
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to check the existence of session master key of version " + version, e);
        }
        try {
            sessionMasterKeyBytes = Jackson.writeValueAsBytes((Object)sessionMasterKey);
        }
        catch (JsonProcessingException e) {
            throw new EncryptionStorageException("Failed to serialize wrapped session master key of version " + version, e);
        }
        try (WriteBatch writeBatch = new WriteBatch();
             WriteOptions writeOptions = new WriteOptions();){
            writeOptions.setSync(true);
            writeBatch.put(this.rocksDbStorage.getColumnFamilyHandle("wdek"), masterKeyKey, sessionMasterKeyBytes);
            writeBatch.put(this.rocksDbStorage.getColumnFamilyHandle("wdek"), SessionKeyStorage.currentSessionMasterKeyVersionKey(), Ints.toByteArray((int)version));
            this.rocksDbStorage.write(writeOptions, writeBatch);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to store session master key of version " + version, e);
        }
        this.maybeSetCurrentSessionKey(sessionMasterKey);
    }

    SessionMasterKey getCurrentSessionMasterKey() {
        int version;
        try {
            byte[] versionBytes = this.rocksDbStorage.get("wdek", SessionKeyStorage.currentSessionMasterKeyVersionKey());
            if (versionBytes == null) {
                throw new EncryptionEntryNoExistException("Current session master key does not exist");
            }
            version = Ints.fromByteArray((byte[])versionBytes);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get the current session master key version", e);
        }
        return this.getSessionMasterKey(version);
    }

    private SessionMasterKey getSessionMasterKey(int version) {
        try {
            byte[] wrappedMasterKeyBytes = this.rocksDbStorage.get("wdek", SessionKeyStorage.sessionMasterKeyKey(version));
            if (wrappedMasterKeyBytes == null) {
                throw new EncryptionStorageException("Session master key of version " + version + " does not exist");
            }
            return (SessionMasterKey)Jackson.readValue((byte[])wrappedMasterKeyBytes, SessionMasterKey.class);
        }
        catch (RocksDBException e) {
            throw new EncryptionStorageException("Failed to get the session master key of version " + version, e);
        }
        catch (JsonParseException | JsonMappingException e) {
            throw new EncryptionStorageException("Failed to read the session master key of version " + version, e);
        }
    }

    CompletableFuture<SessionKey> getCurrentSessionKey() {
        this.lock.lock();
        try {
            if (this.currentSessionKey != null) {
                CompletableFuture<SessionKey> completableFuture = CompletableFuture.completedFuture(this.currentSessionKey);
                return completableFuture;
            }
        }
        finally {
            this.lock.unlock();
        }
        SessionMasterKey sessionMasterKey = this.getCurrentSessionMasterKey();
        return this.maybeSetCurrentSessionKey(sessionMasterKey);
    }

    private CompletableFuture<SessionKey> maybeSetCurrentSessionKey(SessionMasterKey sessionMasterKey) {
        return this.keyWrapper.unwrap(sessionMasterKey.wrappedMasterKey(), sessionMasterKey.kekId()).thenApply(unwrapped -> {
            SessionKey sessionKey = SessionKey.of(unwrapped, sessionMasterKey);
            this.lock.lock();
            try {
                if (this.currentSessionKey != null && this.currentSessionKey.version() >= sessionKey.version()) {
                    SessionKey sessionKey2 = this.currentSessionKey;
                    return sessionKey2;
                }
                this.currentSessionKey = sessionKey;
                for (Consumer<SessionKey> listener : this.listeners) {
                    listener.accept(sessionKey);
                }
                SessionKey sessionKey3 = sessionKey;
                return sessionKey3;
            }
            finally {
                this.lock.unlock();
            }
        });
    }

    public CompletableFuture<SessionKey> getSessionKey(int version) {
        CompletableFuture result = this.sessionKeys.computeIfAbsent(version, v -> {
            SessionMasterKey sessionMasterKey = this.getSessionMasterKey((int)v);
            CompletableFuture future = new CompletableFuture();
            this.keyWrapper.unwrap(sessionMasterKey.wrappedMasterKey(), sessionMasterKey.kekId()).handle((unwrapped, cause) -> {
                if (cause != null) {
                    future.completeExceptionally((Throwable)cause);
                    return null;
                }
                future.complete(SessionKey.of(unwrapped, sessionMasterKey));
                return null;
            });
            return future;
        });
        result.exceptionally(unused -> {
            this.sessionKeys.remove(version);
            return null;
        });
        return result;
    }

    void rotateSessionMasterKey(SessionMasterKey sessionMasterKey) {
        this.storeSessionMasterKey(sessionMasterKey, true);
    }

    private static byte[] sessionMasterKeyKey(int version) {
        return ("session/master/" + version).getBytes(StandardCharsets.UTF_8);
    }

    private static byte[] currentSessionMasterKeyVersionKey() {
        return "session/master/current".getBytes(StandardCharsets.UTF_8);
    }

    void addSessionKeyListener(Consumer<SessionKey> listener) {
        Objects.requireNonNull(listener, "listener");
        this.lock.lock();
        try {
            this.listeners.add(listener);
            if (this.currentSessionKey != null) {
                listener.accept(this.currentSessionKey);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    CompletableFuture<Void> rewrapAllSessionMasterKeys(Executor executor) {
        ArrayList<SessionMasterKey> allSessionMasterKeys = new ArrayList<SessionMasterKey>();
        byte[] prefix = "session/master/".getBytes(StandardCharsets.UTF_8);
        try (RocksIterator iterator = this.rocksDbStorage.newIterator(this.rocksDbStorage.getColumnFamilyHandle("wdek"));){
            iterator.seek(prefix);
            while (iterator.isValid()) {
                byte[] key = iterator.key();
                String keyStr = new String(key, StandardCharsets.UTF_8);
                if (!keyStr.startsWith("session/master/")) {
                    break;
                }
                if ("session/master/current".equals(keyStr)) {
                    iterator.next();
                    continue;
                }
                try {
                    SessionMasterKey sessionMasterKey = (SessionMasterKey)Jackson.readValue((byte[])iterator.value(), SessionMasterKey.class);
                    allSessionMasterKeys.add(sessionMasterKey);
                }
                catch (JsonParseException | JsonMappingException e) {
                    throw new EncryptionStorageException("Failed to read session master key: " + keyStr, e);
                }
                iterator.next();
            }
        }
        ArrayList<CompletionStage> rewrapFutures = new ArrayList<CompletionStage>();
        for (SessionMasterKey sessionMasterKey : allSessionMasterKeys) {
            String oldKekId = sessionMasterKey.kekId();
            CompletionStage rewrapFuture = this.keyWrapper.rewrap(sessionMasterKey.wrappedMasterKey(), oldKekId, this.kekId).handle((newWrappedKey, cause) -> {
                if (cause != null) {
                    logger.warn("Failed to rewrap session master key. version: {}", (Object)sessionMasterKey.version(), cause);
                    return null;
                }
                if (oldKekId.equals(this.kekId) && sessionMasterKey.wrappedMasterKey().equals(newWrappedKey)) {
                    return null;
                }
                return new SessionMasterKey((String)newWrappedKey, sessionMasterKey.version(), sessionMasterKey.salt(), this.kekId, sessionMasterKey.creationInstant());
            });
            rewrapFutures.add(rewrapFuture);
        }
        return CompletableFuture.allOf(rewrapFutures.toArray(new CompletableFuture[0])).thenAcceptAsync(unused -> {
            List collected = (List)rewrapFutures.stream().map(CompletableFuture::join).filter(Objects::nonNull).collect(ImmutableList.toImmutableList());
            if (collected.isEmpty()) {
                logger.info("All session master keys are already wrapped with the current KEK. {}", (Object)this.kekId);
                return;
            }
            try (WriteBatch writeBatch = new WriteBatch();
                 WriteOptions writeOptions = new WriteOptions();){
                writeOptions.setSync(true);
                for (SessionMasterKey newSessionMasterKey : collected) {
                    byte[] sessionMasterKeyBytes;
                    try {
                        sessionMasterKeyBytes = Jackson.writeValueAsBytes((Object)newSessionMasterKey);
                    }
                    catch (JsonProcessingException e) {
                        logger.warn("Failed to serialize re-wrapped session master key. version: {}", (Object)newSessionMasterKey.version(), (Object)e);
                        continue;
                    }
                    byte[] keyBytes = SessionKeyStorage.sessionMasterKeyKey(newSessionMasterKey.version());
                    writeBatch.put(this.rocksDbStorage.getColumnFamilyHandle("wdek"), keyBytes, sessionMasterKeyBytes);
                }
                this.rocksDbStorage.write(writeOptions, writeBatch);
            }
            catch (RocksDBException e) {
                throw new EncryptionStorageException("Failed to store re-wrapped session master keys", e);
            }
        }, executor);
    }
}

