/*
 * Decompiled with CFR 0.152.
 */
package com.linecorp.centraldogma.server.internal.storage;

import com.linecorp.centraldogma.common.Author;
import com.linecorp.centraldogma.common.CentralDogmaException;
import com.linecorp.centraldogma.internal.Util;
import com.linecorp.centraldogma.internal.shaded.guava.base.Preconditions;
import com.linecorp.centraldogma.internal.shaded.guava.collect.ImmutableMap;
import com.linecorp.centraldogma.server.storage.StorageException;
import com.linecorp.centraldogma.server.storage.StorageManager;
import com.linecorp.centraldogma.server.storage.encryption.EncryptionStorageManager;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class DirectoryBasedStorageManager<T>
implements StorageManager<T> {
    private static final Logger logger = LoggerFactory.getLogger(DirectoryBasedStorageManager.class);
    public static final String SUFFIX_REMOVED = ".removed";
    public static final String SUFFIX_PURGED = ".purged";
    private static final String GIT_EXTENSION = ".git";
    private final String childTypeName;
    private final File rootDir;
    private final StorageRemovalManager storageRemovalManager = new StorageRemovalManager();
    private final ConcurrentMap<String, T> children = new ConcurrentHashMap<String, T>();
    private final AtomicReference<Supplier<CentralDogmaException>> closed = new AtomicReference();
    private final Executor purgeWorker;
    private final EncryptionStorageManager encryptionStorageManager;
    private boolean initialized;

    protected DirectoryBasedStorageManager(File rootDir, Class<? extends T> childType, Executor purgeWorker, EncryptionStorageManager encryptionStorageManager) {
        Objects.requireNonNull(rootDir, "rootDir");
        this.purgeWorker = Objects.requireNonNull(purgeWorker, "purgeWorker");
        this.encryptionStorageManager = Objects.requireNonNull(encryptionStorageManager, "encryptionStorageManager");
        if (!rootDir.exists() && !rootDir.mkdirs()) {
            throw new StorageException("failed to create root directory at " + rootDir);
        }
        try {
            rootDir = rootDir.getCanonicalFile();
        }
        catch (IOException e) {
            throw new StorageException("failed to get the canonical path of: " + rootDir, e);
        }
        if (!rootDir.isDirectory()) {
            throw new StorageException("not a directory: " + rootDir);
        }
        this.rootDir = rootDir;
        this.childTypeName = Util.simpleTypeName(Objects.requireNonNull(childType, "childTypeName"), (boolean)true);
    }

    protected Executor purgeWorker() {
        return this.purgeWorker;
    }

    protected EncryptionStorageManager encryptionStorageManager() {
        return this.encryptionStorageManager;
    }

    protected final void init() {
        Preconditions.checkState((!this.initialized ? 1 : 0) != 0, (Object)"initialized already");
        Throwable cause = null;
        try {
            File[] childFiles = this.rootDir.listFiles();
            if (childFiles != null) {
                for (File f : childFiles) {
                    this.loadChild(f);
                }
            }
            this.initialized = true;
        }
        catch (Throwable t) {
            cause = t;
        }
        if (cause != null) {
            CentralDogmaException finalCause = cause instanceof CentralDogmaException ? (CentralDogmaException)cause : new CentralDogmaException("Failed to load a child: " + cause, cause);
            this.close(() -> finalCause);
            throw finalCause;
        }
    }

    @Nullable
    private T loadChild(File f) {
        String name = f.getName();
        if (!DirectoryBasedStorageManager.isValidChildName(name)) {
            return null;
        }
        if (!f.isDirectory()) {
            return null;
        }
        if (new File(f + SUFFIX_REMOVED).exists()) {
            return null;
        }
        try {
            T child = this.openChild(f);
            this.children.put(name, child);
            return child;
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new StorageException("failed to open " + this.childTypeName + ": " + f, e);
        }
    }

    protected boolean replaceChild(String name, T oldChild, T newChild) {
        return this.children.replace(name, oldChild, newChild);
    }

    protected abstract T openChild(File var1) throws Exception;

    protected abstract T createChild(File var1, Author var2, long var3, boolean var5) throws Exception;

    private void closeChild(String name, T child, Supplier<CentralDogmaException> failureCauseSupplier) {
        this.closeChild(new File(this.rootDir, name), child, failureCauseSupplier);
    }

    protected void closeChild(File childDir, T child, Supplier<CentralDogmaException> failureCauseSupplier) {
    }

    protected abstract CentralDogmaException newStorageExistsException(String var1);

    protected abstract CentralDogmaException newStorageNotFoundException(String var1);

    protected abstract void deletePurged(File var1);

    @Override
    public void close(Supplier<CentralDogmaException> failureCauseSupplier) {
        Objects.requireNonNull(failureCauseSupplier, "failureCauseSupplier");
        if (!this.closed.compareAndSet(null, failureCauseSupplier)) {
            return;
        }
        for (Map.Entry e : this.children.entrySet()) {
            this.closeChild((String)e.getKey(), e.getValue(), failureCauseSupplier);
        }
    }

    @Override
    public boolean exists(String name) {
        this.ensureOpen();
        return this.children.containsKey(this.validateChildName(name));
    }

    @Override
    public T get(String name) {
        this.ensureOpen();
        Object child = this.children.get(this.validateChildName(name));
        if (child == null) {
            throw this.newStorageNotFoundException(name);
        }
        return (T)child;
    }

    @Override
    public T create(String name, long creationTimeMillis, Author author, boolean encrypt) {
        this.ensureOpen();
        Objects.requireNonNull(author, "author");
        this.validateChildName(name);
        AtomicBoolean created = new AtomicBoolean();
        Object child = this.children.computeIfAbsent(name, n -> {
            T c = this.create0(author, (String)n, creationTimeMillis, encrypt);
            created.set(true);
            return c;
        });
        if (created.get()) {
            return (T)child;
        }
        throw this.newStorageExistsException(name);
    }

    private T create0(Author author, String name, long creationTimeMillis, boolean encrypt) {
        if (new File(this.rootDir, name + SUFFIX_REMOVED).exists()) {
            throw this.newStorageExistsException(name + " (removed)");
        }
        File f = new File(this.rootDir, name);
        boolean success = false;
        try {
            T newChild = this.createChild(f, author, creationTimeMillis, encrypt);
            success = true;
            T t = newChild;
            return t;
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new StorageException("failed to create a new " + this.childTypeName + ": " + f, e);
        }
        finally {
            if (!success && f.exists()) {
                try {
                    Util.deleteFileTree((File)f);
                }
                catch (IOException e) {
                    logger.warn("Failed to delete a partially created project: {}", (Object)f, (Object)e);
                }
            }
        }
    }

    @Override
    public Map<String, T> list() {
        this.ensureOpen();
        int estimatedSize = this.children.size();
        Object[] names = this.children.keySet().toArray(new String[estimatedSize]);
        Arrays.sort(names);
        LinkedHashMap ret = new LinkedHashMap(estimatedSize);
        for (Object k : names) {
            Object v = this.children.get(k);
            if (v == null) continue;
            ret.put(k, v);
        }
        return Collections.unmodifiableMap(ret);
    }

    @Override
    public Map<String, Instant> listRemoved() {
        this.ensureOpen();
        Object[] files = this.rootDir.listFiles();
        if (files == null) {
            return ImmutableMap.of();
        }
        Arrays.sort(files);
        ImmutableMap.Builder builder = ImmutableMap.builder();
        for (Object f : files) {
            String name;
            if (!((File)f).isDirectory() || !(name = ((File)f).getName()).endsWith(SUFFIX_REMOVED) || !DirectoryBasedStorageManager.isValidChildName(name = name.substring(0, name.length() - SUFFIX_REMOVED.length())) || this.children.containsKey(name)) continue;
            builder.put((Object)name, (Object)this.storageRemovalManager.readRemoval((File)f));
        }
        return builder.build();
    }

    @Override
    public void remove(String name) {
        this.ensureOpen();
        Object child = this.children.remove(this.validateChildName(name));
        if (child == null) {
            throw this.newStorageNotFoundException(name);
        }
        this.closeChild(name, child, () -> this.newStorageNotFoundException(name));
        File file = new File(this.rootDir, name);
        this.storageRemovalManager.mark(file);
        if (!file.renameTo(new File(this.rootDir, name + SUFFIX_REMOVED))) {
            throw new StorageException("failed to mark " + this.childTypeName + " as removed: " + name);
        }
    }

    @Override
    public T unremove(String name) {
        this.ensureOpen();
        this.validateChildName(name);
        File removed = new File(this.rootDir, name + SUFFIX_REMOVED);
        if (!removed.isDirectory()) {
            throw this.newStorageNotFoundException(name);
        }
        File unremoved = new File(this.rootDir, name);
        if (!removed.renameTo(unremoved)) {
            throw new StorageException("failed to mark " + this.childTypeName + " as unremoved: " + name);
        }
        this.storageRemovalManager.unmark(unremoved);
        T unremovedChild = this.loadChild(unremoved);
        if (unremovedChild == null) {
            throw this.newStorageNotFoundException(name);
        }
        return unremovedChild;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void markForPurge(String name) {
        File marked;
        this.ensureOpen();
        this.validateChildName(name);
        File removed = new File(this.rootDir, name + SUFFIX_REMOVED);
        Supplier<File> newMarkedFile = () -> {
            String interfix = '.' + Long.toHexString(ThreadLocalRandom.current().nextLong());
            return new File(this.rootDir, name + interfix + SUFFIX_PURGED);
        };
        DirectoryBasedStorageManager directoryBasedStorageManager = this;
        synchronized (directoryBasedStorageManager) {
            if (!removed.exists()) {
                logger.warn("Tried to purge {}, but it's non-existent.", (Object)removed);
                return;
            }
            if (!removed.isDirectory()) {
                throw new StorageException("not a directory: " + removed);
            }
            marked = newMarkedFile.get();
            boolean moved = false;
            while (!moved) {
                try {
                    Files.move(removed.toPath(), marked.toPath(), new CopyOption[0]);
                    moved = true;
                }
                catch (FileAlreadyExistsException e) {
                    marked = newMarkedFile.get();
                }
                catch (IOException e) {
                    throw new StorageException("failed to mark " + this.childTypeName + " for purge: " + removed, e);
                }
            }
        }
        File purged = marked;
        try {
            this.purgeWorker().execute(() -> this.deletePurged(purged));
        }
        catch (Exception e) {
            logger.warn("Failed to schedule a purge task for {}:", (Object)purged, (Object)e);
        }
    }

    @Override
    public void purgeMarked() {
        this.ensureOpen();
        File[] files = this.rootDir.listFiles();
        if (files == null) {
            return;
        }
        for (File f : files) {
            String name;
            if (!f.isDirectory() || !(name = f.getName()).endsWith(SUFFIX_PURGED)) continue;
            this.deletePurged(f);
        }
    }

    @Override
    public void ensureOpen() {
        Preconditions.checkState((boolean)this.initialized, (Object)"not initialized yet");
        if (this.closed.get() != null) {
            throw this.closed.get().get();
        }
    }

    private String validateChildName(String name) {
        if (!DirectoryBasedStorageManager.isValidChildName(Objects.requireNonNull(name, "name"))) {
            throw new IllegalArgumentException("invalid " + this.childTypeName + " name: " + name);
        }
        return name;
    }

    private static boolean isValidChildName(String name) {
        if (name == null) {
            return false;
        }
        if (!Util.PROJECT_AND_REPO_NAME_PATTERN.matcher(name).matches()) {
            return false;
        }
        return !name.endsWith(SUFFIX_REMOVED) && !name.endsWith(SUFFIX_PURGED) && !name.endsWith(GIT_EXTENSION);
    }

    public String toString() {
        return Util.simpleTypeName(this.getClass()) + '(' + this.rootDir + ')';
    }

    private final class StorageRemovalManager {
        private static final String REMOVAL_TIMESTAMP_NAME = "removal.timestamp";

        private StorageRemovalManager() {
        }

        void mark(File file) {
            File removal = new File(file, REMOVAL_TIMESTAMP_NAME);
            String timestamp = DateTimeFormatter.ISO_INSTANT.format(Instant.now());
            try {
                Files.write(removal.toPath(), timestamp.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
            }
            catch (IOException e) {
                throw new StorageException("failed to write a removal timestamp for " + DirectoryBasedStorageManager.this.childTypeName + ": " + removal);
            }
        }

        void unmark(File file) {
            File removal = new File(file, REMOVAL_TIMESTAMP_NAME);
            if (removal.exists() && !removal.delete()) {
                logger.warn("Failed to delete a removal timestamp for {}: {}", (Object)DirectoryBasedStorageManager.this.childTypeName, (Object)removal);
            }
        }

        Instant readRemoval(File file) {
            File removal = new File(file, REMOVAL_TIMESTAMP_NAME);
            if (!removal.exists()) {
                return Instant.ofEpochMilli(file.lastModified());
            }
            try {
                String timestamp = new String(Files.readAllBytes(removal.toPath()), StandardCharsets.UTF_8);
                return Instant.from(DateTimeFormatter.ISO_INSTANT.parse(timestamp));
            }
            catch (Exception e) {
                logger.warn("Failed to read a removal timestamp for {}: {}", new Object[]{DirectoryBasedStorageManager.this.childTypeName, removal, e});
                return Instant.ofEpochMilli(file.lastModified());
            }
        }
    }
}

