/*
 * Decompiled with CFR 0.152.
 */
package com.linecorp.centraldogma.server.internal.mirror;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.linecorp.centraldogma.common.Author;
import com.linecorp.centraldogma.common.Change;
import com.linecorp.centraldogma.common.Entry;
import com.linecorp.centraldogma.common.Markup;
import com.linecorp.centraldogma.common.Revision;
import com.linecorp.centraldogma.internal.Jackson;
import com.linecorp.centraldogma.internal.shaded.guava.annotations.VisibleForTesting;
import com.linecorp.centraldogma.internal.shaded.guava.base.Stopwatch;
import com.linecorp.centraldogma.internal.shaded.guava.collect.ImmutableList;
import com.linecorp.centraldogma.internal.shaded.guava.collect.ImmutableMap;
import com.linecorp.centraldogma.server.command.Command;
import com.linecorp.centraldogma.server.command.CommandExecutor;
import com.linecorp.centraldogma.server.command.CommitResult;
import com.linecorp.centraldogma.server.internal.replication.ZooKeeperCommandExecutor;
import com.linecorp.centraldogma.server.internal.storage.repository.DefaultMetaRepository;
import com.linecorp.centraldogma.server.internal.storage.repository.MirrorConfig;
import com.linecorp.centraldogma.server.internal.storage.repository.RepositoryMetadataException;
import com.linecorp.centraldogma.server.management.ServerStatus;
import com.linecorp.centraldogma.server.mirror.MirrorCredential;
import com.linecorp.centraldogma.server.storage.project.Project;
import com.linecorp.centraldogma.server.storage.project.ProjectManager;
import com.linecorp.centraldogma.server.storage.repository.FindOption;
import com.linecorp.centraldogma.server.storage.repository.MetaRepository;
import com.linecorp.centraldogma.server.storage.repository.Repository;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.net.URI;
import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class MirroringMigrationService {
    private static final Logger logger = LoggerFactory.getLogger(MirroringMigrationService.class);
    @VisibleForTesting
    static final String PATH_LEGACY_MIRRORS = "/mirrors.json";
    @VisibleForTesting
    static final String PATH_LEGACY_CREDENTIALS = "/credentials.json";
    @VisibleForTesting
    static final String PATH_LEGACY_MIRRORS_BACKUP = "/mirrors.json.bak";
    @VisibleForTesting
    static final String PATH_LEGACY_CREDENTIALS_BACKUP = "/credentials.json.bak";
    @VisibleForTesting
    static final String MIRROR_MIGRATION_JOB_LOG = "/mirror-migration-job.json";
    private final ProjectManager projectManager;
    private final CommandExecutor commandExecutor;
    @Nullable
    private List<String> shortWords;

    MirroringMigrationService(ProjectManager projectManager, CommandExecutor commandExecutor) {
        this.projectManager = projectManager;
        this.commandExecutor = commandExecutor;
    }

    void migrate() throws Exception {
        if (this.hasMigrationLog()) {
            logger.debug("Mirrors and credentials have already been migrated. Skipping auto migration...");
            return;
        }
        this.commandExecutor.execute(Command.updateServerStatus(ServerStatus.REPLICATION_ONLY)).get(1L, TimeUnit.MINUTES);
        logger.info("Starting Mirrors and credentials migration ...");
        if (this.commandExecutor instanceof ZooKeeperCommandExecutor) {
            logger.debug("Waiting for 30 seconds to make sure that all cluster have been notified of the read-only mode ...");
            Thread.sleep(30000L);
        }
        Stopwatch stopwatch = Stopwatch.createStarted();
        int numMigratedProjects = 0;
        try {
            for (Project project : this.projectManager.list().values()) {
                logger.info("Migrating mirrors and credentials in the project: {} ...", (Object)project.name());
                boolean processed = false;
                MetaRepository repository = project.metaRepo();
                processed |= this.migrateCredentials(repository);
                if (processed |= this.migrateMirrors(repository)) {
                    ++numMigratedProjects;
                    logger.info("Mirrors and credentials in the project: {} have been migrated.", (Object)project.name());
                    continue;
                }
                logger.info("No legacy configurations of mirrors and credentials found in the project: {}.", (Object)project.name());
            }
            this.logMigrationJob(numMigratedProjects);
        }
        catch (Exception ex) {
            MirrorMigrationException mirrorException = new MirrorMigrationException("Failed to migrate mirrors and credentials. Rollback to the legacy configurations", ex);
            try {
                this.rollbackMigration();
            }
            catch (Exception ex0) {
                ex0.addSuppressed(mirrorException);
                throw new MirrorMigrationException("Failed to rollback the mirror migration:", ex0);
            }
            throw mirrorException;
        }
        this.commandExecutor.execute(Command.updateServerStatus(ServerStatus.WRITABLE)).get(1L, TimeUnit.MINUTES);
        logger.info("Mirrors and credentials migration has been completed. (took: {} ms.)", (Object)stopwatch.elapsed().toMillis());
        this.shortWords = null;
    }

    private void logMigrationJob(int numMigratedProjects) throws Exception {
        ImmutableMap data = ImmutableMap.of((Object)"timestamp", (Object)Instant.now(), (Object)"projects", (Object)numMigratedProjects);
        Change change = Change.ofJsonUpsert((String)MIRROR_MIGRATION_JOB_LOG, (String)Jackson.writeValueAsString((Object)data));
        Command<CommitResult> command = Command.push(Author.SYSTEM, "dogma", "dogma", Revision.HEAD, "Migration of mirrors and credentials has been done", "", Markup.PLAINTEXT, change);
        this.executeCommand(command);
    }

    private void removeMigrationJobLog() throws Exception {
        if (!this.hasMigrationLog()) {
            return;
        }
        Change change = Change.ofRemoval((String)MIRROR_MIGRATION_JOB_LOG);
        Command<CommitResult> command = Command.push(Author.SYSTEM, "dogma", "dogma", Revision.HEAD, "Remove the migration job log", "", Markup.PLAINTEXT, change);
        this.executeCommand(command);
    }

    private boolean hasMigrationLog() throws Exception {
        Project internalProj = (Project)this.projectManager.get("dogma");
        Repository repository = (Repository)internalProj.repos().get("dogma");
        Map<String, Entry<?>> entries = repository.find(Revision.HEAD, MIRROR_MIGRATION_JOB_LOG).get();
        Entry<?> entry = entries.get(MIRROR_MIGRATION_JOB_LOG);
        return entry != null;
    }

    private boolean migrateMirrors(MetaRepository repository) throws Exception {
        ArrayNode mirrors = MirroringMigrationService.getLegacyMetaData(repository, PATH_LEGACY_MIRRORS);
        if (mirrors == null) {
            return false;
        }
        List<MirrorCredential> credentials = repository.credentials().get(30L, TimeUnit.SECONDS);
        HashSet<String> mirrorIds = new HashSet<String>();
        for (JsonNode mirror : mirrors) {
            if (!mirror.isObject()) {
                logger.warn("A mirror config must be an object: {} (project: {})", (Object)mirror, (Object)repository.parent().name());
                continue;
            }
            try {
                this.migrateMirror(repository, (ObjectNode)mirror, mirrorIds, credentials);
            }
            catch (Exception e) {
                logger.warn("Failed to migrate a mirror config: {} (project: {})", new Object[]{mirror, repository.parent().name(), e});
                throw e;
            }
        }
        this.rename(repository, PATH_LEGACY_MIRRORS, PATH_LEGACY_MIRRORS_BACKUP, false);
        return true;
    }

    private void migrateMirror(MetaRepository repository, ObjectNode mirror, Set<String> mirrorIds, List<MirrorCredential> credentials) throws Exception {
        JsonNode idNode = mirror.get("id");
        String id = idNode == null ? MirroringMigrationService.generateIdForMirror(repository.parent().name(), mirror) : idNode.asText();
        id = this.uniquify(id, mirrorIds);
        mirror.put("id", id);
        MirroringMigrationService.fillCredentialId(repository, mirror, credentials);
        if (mirror.get("schedule") == null) {
            mirror.put("schedule", "0 * * * * ?");
        }
        mirrorIds.add(id);
        String jsonFile = DefaultMetaRepository.mirrorFile(id);
        Command<CommitResult> command = Command.push(Author.SYSTEM, repository.parent().name(), repository.name(), Revision.HEAD, "Migrate the mirror " + id + " in '" + PATH_LEGACY_MIRRORS + "' into '" + jsonFile + "'.", "", Markup.PLAINTEXT, Change.ofJsonUpsert((String)jsonFile, (JsonNode)mirror));
        this.executeCommand(command);
    }

    private void rollbackMigration() throws Exception {
        for (Project project : this.projectManager.list().values()) {
            logger.info("Rolling back the migration of mirrors and credentials in the project: {} ...", (Object)project.name());
            MetaRepository metaRepository = project.metaRepo();
            this.rollbackMigration(metaRepository, "/mirrors/", PATH_LEGACY_MIRRORS, PATH_LEGACY_MIRRORS_BACKUP);
            this.rollbackMigration(metaRepository, "/credentials/", PATH_LEGACY_CREDENTIALS, PATH_LEGACY_CREDENTIALS_BACKUP);
            this.removeMigrationJobLog();
        }
    }

    private void rollbackMigration(MetaRepository repository, String targetDirectory, String originalFile, String backupFile) throws Exception {
        Map<String, Entry<?>> entries = repository.find(Revision.HEAD, targetDirectory + "**").get();
        List changes = (List)entries.keySet().stream().map(Change::ofRemoval).collect(ImmutableList.toImmutableList());
        Command<CommitResult> command = Command.push(Author.SYSTEM, repository.parent().name(), repository.name(), Revision.HEAD, "Rollback the migration of " + targetDirectory, "", Markup.PLAINTEXT, changes);
        try {
            this.executeCommand(command);
        }
        catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw new MirrorMigrationException("Failed to rollback the migration of " + targetDirectory, e);
        }
        Entry<?> backup = repository.getOrNull(Revision.HEAD, backupFile).get();
        if (backup != null) {
            this.rename(repository, backupFile, originalFile, true);
        }
    }

    private CommitResult executeCommand(Command<CommitResult> command) throws InterruptedException, ExecutionException, TimeoutException {
        return this.commandExecutor.execute(Command.forcePush(command)).get(1L, TimeUnit.MINUTES);
    }

    private static void fillCredentialId(MetaRepository repository, ObjectNode mirror, List<MirrorCredential> credentials) {
        JsonNode credentialId = mirror.get("credentialId");
        if (credentialId != null) {
            return;
        }
        JsonNode remoteUri = mirror.get("remoteUri");
        if (remoteUri == null) {
            return;
        }
        String remoteUriText = remoteUri.asText();
        MirrorCredential credential = MirrorConfig.findCredential(credentials, URI.create(remoteUriText), null);
        if (credential == MirrorCredential.FALLBACK) {
            logger.warn("Failed to find a credential for the mirror: {}, project: {}. Using the fallback credential.", (Object)mirror, (Object)repository.parent().name());
        }
        mirror.put("credentialId", credential.id());
    }

    private boolean migrateCredentials(MetaRepository repository) throws Exception {
        ArrayNode credentials = MirroringMigrationService.getLegacyMetaData(repository, PATH_LEGACY_CREDENTIALS);
        if (credentials == null) {
            return false;
        }
        HashSet<String> credentialIds = new HashSet<String>();
        int index = 0;
        for (JsonNode credential : credentials) {
            if (!credential.isObject()) {
                logger.warn("A credential config at {} must be an object: {} (project: {})", new Object[]{index, credential.getNodeType(), repository.parent().name()});
            } else {
                try {
                    this.migrateCredential(repository, (ObjectNode)credential, credentialIds);
                }
                catch (Exception e) {
                    logger.warn("Failed to migrate the credential config in project {}", (Object)repository.parent().name(), (Object)e);
                    throw e;
                }
            }
            ++index;
        }
        this.rename(repository, PATH_LEGACY_CREDENTIALS, PATH_LEGACY_CREDENTIALS_BACKUP, false);
        return true;
    }

    private void migrateCredential(MetaRepository repository, ObjectNode credential, Set<String> credentialIds) throws Exception {
        String id;
        JsonNode idNode = credential.get("id");
        String projectName = repository.parent().name();
        if (idNode == null) {
            JsonNode typeNode = credential.get("type");
            String type = typeNode.isTextual() ? typeNode.asText() : "";
            id = MirroringMigrationService.generateIdForCredential(projectName, type);
        } else {
            id = idNode.asText();
        }
        id = this.uniquify(id, credentialIds);
        credential.put("id", id);
        credentialIds.add(id);
        String jsonFile = DefaultMetaRepository.credentialFile(id);
        Command<CommitResult> command = Command.push(Author.SYSTEM, projectName, repository.name(), Revision.HEAD, "Migrate the credential '" + id + "' in '" + PATH_LEGACY_CREDENTIALS + "' into '" + jsonFile + "'.", "", Markup.PLAINTEXT, Change.ofJsonUpsert((String)jsonFile, (JsonNode)credential));
        this.executeCommand(command);
    }

    @Nullable
    private static ArrayNode getLegacyMetaData(MetaRepository repository, String path) throws InterruptedException, ExecutionException {
        Map<String, Entry<?>> entries = repository.find(Revision.HEAD, path, (Map<FindOption<?>, ?>)ImmutableMap.of()).get();
        Entry<?> entry = entries.get(path);
        if (entry == null) {
            return null;
        }
        JsonNode content = (JsonNode)entry.content();
        if (!content.isArray()) {
            throw new RepositoryMetadataException(path + " must be an array: " + content.getNodeType());
        }
        return (ArrayNode)content;
    }

    private CommitResult rename(MetaRepository repository, String oldPath, String newPath, boolean rollback) throws Exception {
        String summary = rollback ? "Rollback the migration of " + newPath : "Back up the legacy " + oldPath + " into " + newPath;
        Command<CommitResult> command = Command.push(Author.SYSTEM, repository.parent().name(), repository.name(), Revision.HEAD, summary, "", Markup.PLAINTEXT, Change.ofRename((String)oldPath, (String)newPath));
        return this.executeCommand(command);
    }

    private static String generateIdForMirror(String projectName, ObjectNode mirror) {
        return "mirror-" + projectName + '-' + mirror.get("localRepo").asText();
    }

    private String getShortWord(String id) {
        if (this.shortWords == null) {
            this.shortWords = MirroringMigrationService.buildShortWords();
        }
        int index = Math.abs(id.hashCode()) % this.shortWords.size();
        return this.shortWords.get(index);
    }

    private static String generateIdForCredential(String projectName, String type) {
        String id = "credential-" + projectName;
        if (!type.isEmpty()) {
            id = id + '-' + type;
        }
        return id;
    }

    private String uniquify(String id, Set<String> existingIds) {
        String maybeUnique = id;
        while (existingIds.contains(maybeUnique)) {
            maybeUnique = id + '-' + this.getShortWord(maybeUnique);
        }
        return maybeUnique;
    }

    @VisibleForTesting
    static List<String> buildShortWords() {
        List list;
        InputStream is = MirroringMigrationService.class.getResourceAsStream("short_wordlist.txt");
        BufferedReader reader = new BufferedReader(new InputStreamReader(is));
        try {
            list = (List)reader.lines().collect(ImmutableList.toImmutableList());
        }
        catch (Throwable throwable) {
            try {
                try {
                    reader.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        reader.close();
        return list;
    }

    private static class MirrorMigrationException
    extends RuntimeException {
        private static final long serialVersionUID = -3924318204193024460L;

        MirrorMigrationException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

