/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.commerce.pricing.service.impl;

import com.liferay.asset.kernel.model.AssetCategory;
import com.liferay.asset.kernel.model.AssetEntry;
import com.liferay.asset.kernel.service.AssetEntryLocalService;
import com.liferay.commerce.pricing.constants.CommercePriceModifierConstants;
import com.liferay.commerce.pricing.exception.CommercePriceModifierAmountException;
import com.liferay.commerce.pricing.exception.CommercePriceModifierDisplayDateException;
import com.liferay.commerce.pricing.exception.CommercePriceModifierExpirationDateException;
import com.liferay.commerce.pricing.exception.CommercePriceModifierTargetException;
import com.liferay.commerce.pricing.exception.CommercePriceModifierTitleException;
import com.liferay.commerce.pricing.exception.CommercePriceModifierTypeException;
import com.liferay.commerce.pricing.exception.DuplicateCommercePriceModifierException;
import com.liferay.commerce.pricing.exception.NoSuchPriceModifierException;
import com.liferay.commerce.pricing.model.CommercePriceModifier;
import com.liferay.commerce.pricing.service.CommercePricingClassLocalService;
import com.liferay.commerce.pricing.service.base.CommercePriceModifierLocalServiceBaseImpl;
import com.liferay.commerce.pricing.type.CommercePriceModifierType;
import com.liferay.commerce.pricing.type.CommercePriceModifierTypeRegistry;
import com.liferay.commerce.product.model.CPDefinition;
import com.liferay.portal.kernel.bean.BeanReference;
import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.model.SystemEventConstants;
import com.liferay.portal.kernel.model.User;
import com.liferay.portal.kernel.service.ServiceContext;
import com.liferay.portal.kernel.systemevent.SystemEvent;
import com.liferay.portal.kernel.util.OrderByComparator;
import com.liferay.portal.kernel.util.PortalUtil;
import com.liferay.portal.kernel.util.Validator;
import com.liferay.portal.kernel.workflow.WorkflowConstants;
import com.liferay.portal.kernel.workflow.WorkflowHandlerRegistryUtil;
import com.liferay.portal.spring.extender.service.ServiceReference;

import java.io.Serializable;

import java.math.BigDecimal;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.LongStream;
import java.util.stream.Stream;

/**
 * @author Riccardo Alberti
 * @see CommercePriceModifierLocalServiceBaseImpl
 */
public class CommercePriceModifierLocalServiceImpl
	extends CommercePriceModifierLocalServiceBaseImpl {

	@Override
	public CommercePriceModifier addCommercePriceModifier(
			long groupId, String title, long commercePriceListId,
			String modifierType, BigDecimal modifierAmount, double priority,
			boolean active, int displayDateMonth, int displayDateDay,
			int displayDateYear, int displayDateHour, int displayDateMinute,
			int expirationDateMonth, int expirationDateDay,
			int expirationDateYear, int expirationDateHour,
			int expirationDateMinute, boolean neverExpire,
			ServiceContext serviceContext)
		throws PortalException {

		return addCommercePriceModifier(
			groupId, title, CommercePriceModifierConstants.TARGET_CATALOG,
			commercePriceListId, modifierType, modifierAmount, priority, active,
			displayDateMonth, displayDateDay, displayDateYear, displayDateHour,
			displayDateMinute, expirationDateMonth, expirationDateDay,
			expirationDateYear, expirationDateHour, expirationDateMinute,
			neverExpire, serviceContext);
	}

	@Override
	public CommercePriceModifier addCommercePriceModifier(
			long groupId, String title, String target, long commercePriceListId,
			String modifierType, BigDecimal modifierAmount, double priority,
			boolean active, int displayDateMonth, int displayDateDay,
			int displayDateYear, int displayDateHour, int displayDateMinute,
			int expirationDateMonth, int expirationDateDay,
			int expirationDateYear, int expirationDateHour,
			int expirationDateMinute, boolean neverExpire,
			ServiceContext serviceContext)
		throws PortalException {

		return addCommercePriceModifier(
			groupId, title, target, commercePriceListId, modifierType,
			modifierAmount, priority, active, displayDateMonth, displayDateDay,
			displayDateYear, displayDateHour, displayDateMinute,
			expirationDateMonth, expirationDateDay, expirationDateYear,
			expirationDateHour, expirationDateMinute, null, neverExpire,
			serviceContext);
	}

	@Override
	public CommercePriceModifier addCommercePriceModifier(
			long groupId, String title, String target, long commercePriceListId,
			String modifierType, BigDecimal modifierAmount, double priority,
			boolean active, int displayDateMonth, int displayDateDay,
			int displayDateYear, int displayDateHour, int displayDateMinute,
			int expirationDateMonth, int expirationDateDay,
			int expirationDateYear, int expirationDateHour,
			int expirationDateMinute, String externalReferenceCode,
			boolean neverExpire, ServiceContext serviceContext)
		throws PortalException {

		if (Validator.isBlank(externalReferenceCode)) {
			externalReferenceCode = null;
		}

		validateExternalReferenceCode(
			serviceContext.getCompanyId(), externalReferenceCode);

		// Commerce price modifier

		User user = userLocalService.getUser(serviceContext.getUserId());

		Date now = new Date();

		Date displayDate = PortalUtil.getDate(
			displayDateMonth, displayDateDay, displayDateYear, displayDateHour,
			displayDateMinute, user.getTimeZone(),
			CommercePriceModifierDisplayDateException.class);

		Date expirationDate = null;

		if (!neverExpire) {
			expirationDate = PortalUtil.getDate(
				expirationDateMonth, expirationDateDay, expirationDateYear,
				expirationDateHour, expirationDateMinute, user.getTimeZone(),
				CommercePriceModifierExpirationDateException.class);
		}

		long commercePriceModifierId = counterLocalService.increment();

		validate(title, target, modifierType, modifierAmount);

		CommercePriceModifier commercePriceModifier =
			commercePriceModifierPersistence.create(commercePriceModifierId);

		commercePriceModifier.setGroupId(groupId);
		commercePriceModifier.setCompanyId(user.getCompanyId());
		commercePriceModifier.setUserId(user.getUserId());
		commercePriceModifier.setUserName(user.getFullName());
		commercePriceModifier.setCommercePriceListId(commercePriceListId);
		commercePriceModifier.setTitle(title);
		commercePriceModifier.setTarget(target);
		commercePriceModifier.setModifierAmount(modifierAmount);
		commercePriceModifier.setModifierType(modifierType);
		commercePriceModifier.setPriority(priority);
		commercePriceModifier.setActive(active);
		commercePriceModifier.setDisplayDate(displayDate);
		commercePriceModifier.setExpirationDate(expirationDate);
		commercePriceModifier.setExternalReferenceCode(externalReferenceCode);

		if ((expirationDate == null) || expirationDate.after(now)) {
			commercePriceModifier.setStatus(WorkflowConstants.STATUS_DRAFT);
		}
		else {
			commercePriceModifier.setStatus(WorkflowConstants.STATUS_EXPIRED);
		}

		commercePriceModifier.setStatusByUserId(user.getUserId());
		commercePriceModifier.setStatusDate(
			serviceContext.getModifiedDate(now));
		commercePriceModifier.setExpandoBridgeAttributes(serviceContext);

		commercePriceModifier = commercePriceModifierPersistence.update(
			commercePriceModifier);

		// Workflow

		return startWorkflowInstance(
			user.getUserId(), commercePriceModifier, serviceContext);
	}

	@Override
	@SystemEvent(type = SystemEventConstants.TYPE_DELETE)
	public CommercePriceModifier deleteCommercePriceModifier(
			CommercePriceModifier commercePriceModifier)
		throws PortalException {

		// Commerce price modifier rels

		commercePriceModifierRelLocalService.deleteCommercePriceModifierRels(
			commercePriceModifier.getCommercePriceModifierId());

		// Commerce price modifier

		commercePriceModifierPersistence.remove(commercePriceModifier);

		// Expando

		expandoRowLocalService.deleteRows(
			commercePriceModifier.getCommercePriceModifierId());

		// Workflow

		workflowInstanceLinkLocalService.deleteWorkflowInstanceLinks(
			commercePriceModifier.getCompanyId(), 0L,
			CommercePriceModifier.class.getName(),
			commercePriceModifier.getCommercePriceModifierId());

		return commercePriceModifier;
	}

	@Override
	public CommercePriceModifier deleteCommercePriceModifier(
			long commercePriceModifierId)
		throws PortalException {

		CommercePriceModifier commercePriceModifier =
			commercePriceModifierPersistence.findByPrimaryKey(
				commercePriceModifierId);

		return commercePriceModifierLocalService.deleteCommercePriceModifier(
			commercePriceModifier);
	}

	@Override
	public void deleteCommercePriceModifiers(long companyId)
		throws PortalException {

		List<CommercePriceModifier> commercePriceModifiers =
			commercePriceModifierPersistence.findByCompanyId(companyId);

		for (CommercePriceModifier commercePriceModifier :
				commercePriceModifiers) {

			commercePriceModifierLocalService.deleteCommercePriceModifier(
				commercePriceModifier);
		}
	}

	@Override
	public void deleteCommercePriceModifiersByCommercePriceListId(
			long commercePriceListId)
		throws PortalException {

		List<CommercePriceModifier> commercePriceModifiers =
			commercePriceModifierPersistence.findByCommercePriceListId(
				commercePriceListId);

		for (CommercePriceModifier commercePriceModifier :
				commercePriceModifiers) {

			commercePriceModifierLocalService.deleteCommercePriceModifier(
				commercePriceModifier);
		}
	}

	@Override
	public CommercePriceModifier fetchByExternalReferenceCode(
		long companyId, String externalReferenceCode) {

		if (Validator.isBlank(externalReferenceCode)) {
			return null;
		}

		return commercePriceModifierPersistence.fetchByC_ERC(
			companyId, externalReferenceCode);
	}

	@Override
	public List<CommercePriceModifier> getCommercePriceModifiers(
		long commercePriceListId) {

		return commercePriceModifierPersistence.findByCommercePriceListId(
			commercePriceListId);
	}

	@Override
	public List<CommercePriceModifier> getCommercePriceModifiers(
		long commercePriceListId, int start, int end,
		OrderByComparator<CommercePriceModifier> orderByComparator) {

		return commercePriceModifierPersistence.findByCommercePriceListId(
			commercePriceListId, start, end, orderByComparator);
	}

	@Override
	public List<CommercePriceModifier> getCommercePriceModifiers(
		long companyId, String target) {

		return commercePriceModifierPersistence.findByC_T(companyId, target);
	}

	@Override
	public int getCommercePriceModifiersCount(long commercePriceListId) {
		return commercePriceModifierPersistence.countByCommercePriceListId(
			commercePriceListId);
	}

	@Override
	public List<CommercePriceModifier> getQualifiedCommercePriceModifiers(
		long commercePriceListId, long cpDefinitionId) {

		return commercePriceModifierFinder.findByC_C_C_P(
			commercePriceListId, cpDefinitionId,
			_getAssetCategoryIds(cpDefinitionId),
			_commercePricingClassLocalService.
				getCommercePricingClassByCPDefinition(cpDefinitionId));
	}

	@Override
	public CommercePriceModifier updateCommercePriceModifier(
			long commercePriceModifierId, long groupId, String title,
			String target, long commercePriceListId, String modifierType,
			BigDecimal modifierAmount, double priority, boolean active,
			int displayDateMonth, int displayDateDay, int displayDateYear,
			int displayDateHour, int displayDateMinute, int expirationDateMonth,
			int expirationDateDay, int expirationDateYear,
			int expirationDateHour, int expirationDateMinute,
			boolean neverExpire, ServiceContext serviceContext)
		throws PortalException {

		User user = userLocalService.getUser(serviceContext.getUserId());

		CommercePriceModifier commercePriceModifier =
			commercePriceModifierPersistence.findByPrimaryKey(
				commercePriceModifierId);

		validate(title, target, modifierType, modifierAmount);

		String currentTarget = commercePriceModifier.getTarget();

		if (!currentTarget.equals(target)) {
			commercePriceModifierRelLocalService.
				deleteCommercePriceModifierRels(
					commercePriceModifier.getCommercePriceModifierId());
		}

		Date now = new Date();

		Date displayDate = PortalUtil.getDate(
			displayDateMonth, displayDateDay, displayDateYear, displayDateHour,
			displayDateMinute, user.getTimeZone(),
			CommercePriceModifierDisplayDateException.class);

		Date expirationDate = null;

		if (!neverExpire) {
			expirationDate = PortalUtil.getDate(
				expirationDateMonth, expirationDateDay, expirationDateYear,
				expirationDateHour, expirationDateMinute, user.getTimeZone(),
				CommercePriceModifierExpirationDateException.class);
		}

		commercePriceModifier.setGroupId(groupId);
		commercePriceModifier.setCommercePriceListId(commercePriceListId);
		commercePriceModifier.setTitle(title);
		commercePriceModifier.setTarget(target);
		commercePriceModifier.setModifierAmount(modifierAmount);
		commercePriceModifier.setModifierType(modifierType);
		commercePriceModifier.setPriority(priority);
		commercePriceModifier.setActive(active);
		commercePriceModifier.setDisplayDate(displayDate);
		commercePriceModifier.setExpirationDate(expirationDate);

		if ((expirationDate == null) || expirationDate.after(now)) {
			commercePriceModifier.setStatus(WorkflowConstants.STATUS_DRAFT);
		}
		else {
			commercePriceModifier.setStatus(WorkflowConstants.STATUS_EXPIRED);
		}

		commercePriceModifier.setStatusByUserId(user.getUserId());
		commercePriceModifier.setStatusDate(
			serviceContext.getModifiedDate(now));
		commercePriceModifier.setExpandoBridgeAttributes(serviceContext);

		commercePriceModifier = commercePriceModifierPersistence.update(
			commercePriceModifier);

		return startWorkflowInstance(
			user.getUserId(), commercePriceModifier, serviceContext);
	}

	@Override
	public CommercePriceModifier updateStatus(
			long userId, long commercePriceModifierId, int status,
			ServiceContext serviceContext,
			Map<String, Serializable> workflowContext)
		throws PortalException {

		User user = userLocalService.getUser(userId);
		Date now = new Date();

		CommercePriceModifier commercePriceModifier =
			commercePriceModifierPersistence.findByPrimaryKey(
				commercePriceModifierId);

		if ((status == WorkflowConstants.STATUS_APPROVED) &&
			(commercePriceModifier.getDisplayDate() != null) &&
			now.before(commercePriceModifier.getDisplayDate())) {

			status = WorkflowConstants.STATUS_SCHEDULED;
		}

		if (status == WorkflowConstants.STATUS_APPROVED) {
			Date expirationDate = commercePriceModifier.getExpirationDate();

			if ((expirationDate != null) && expirationDate.before(now)) {
				commercePriceModifier.setExpirationDate(null);
			}
		}

		if (status == WorkflowConstants.STATUS_EXPIRED) {
			commercePriceModifier.setExpirationDate(now);
		}

		commercePriceModifier.setStatus(status);
		commercePriceModifier.setStatusByUserId(user.getUserId());
		commercePriceModifier.setStatusByUserName(user.getFullName());
		commercePriceModifier.setStatusDate(
			serviceContext.getModifiedDate(now));

		return commercePriceModifierPersistence.update(commercePriceModifier);
	}

	@Override
	public CommercePriceModifier upsertCommercePriceModifier(
			long userId, long commercePriceModifierId, long groupId,
			String title, String target, long commercePriceListId,
			String modifierType, BigDecimal modifierAmount, double priority,
			boolean active, int displayDateMonth, int displayDateDay,
			int displayDateYear, int displayDateHour, int displayDateMinute,
			int expirationDateMonth, int expirationDateDay,
			int expirationDateYear, int expirationDateHour,
			int expirationDateMinute, String externalReferenceCode,
			boolean neverExpire, ServiceContext serviceContext)
		throws PortalException {

		// Update

		if (commercePriceModifierId > 0) {
			try {
				return commercePriceModifierLocalService.
					updateCommercePriceModifier(
						commercePriceModifierId, groupId, title, target,
						commercePriceListId, modifierType, modifierAmount,
						priority, active, displayDateMonth, displayDateDay,
						displayDateYear, displayDateHour, displayDateMinute,
						expirationDateMonth, expirationDateDay,
						expirationDateYear, expirationDateHour,
						expirationDateMinute, neverExpire, serviceContext);
			}
			catch (NoSuchPriceModifierException nspme) {
				if (_log.isDebugEnabled()) {
					_log.debug(
						"Unable to find price modifier with ID: " +
							commercePriceModifierId);
				}
			}
		}

		if (!Validator.isBlank(externalReferenceCode)) {
			CommercePriceModifier commercePriceModifier =
				commercePriceModifierPersistence.fetchByC_ERC(
					serviceContext.getCompanyId(), externalReferenceCode);

			if (commercePriceModifier != null) {
				return commercePriceModifierLocalService.
					updateCommercePriceModifier(
						commercePriceModifierId, groupId, title, target,
						commercePriceListId, modifierType, modifierAmount,
						priority, active, displayDateMonth, displayDateDay,
						displayDateYear, displayDateHour, displayDateMinute,
						expirationDateMonth, expirationDateDay,
						expirationDateYear, expirationDateHour,
						expirationDateMinute, neverExpire, serviceContext);
			}
		}

		// Add

		return commercePriceModifierLocalService.addCommercePriceModifier(
			groupId, title, target, commercePriceListId, modifierType,
			modifierAmount, priority, active, displayDateMonth, displayDateDay,
			displayDateYear, displayDateHour, displayDateMinute,
			expirationDateMonth, expirationDateDay, expirationDateYear,
			expirationDateHour, expirationDateMinute, externalReferenceCode,
			neverExpire, serviceContext);
	}

	protected CommercePriceModifier startWorkflowInstance(
			long userId, CommercePriceModifier commercePriceModifier,
			ServiceContext serviceContext)
		throws PortalException {

		Map<String, Serializable> workflowContext = new HashMap<>();

		return WorkflowHandlerRegistryUtil.startWorkflowInstance(
			commercePriceModifier.getCompanyId(), 0L, userId,
			CommercePriceModifier.class.getName(),
			commercePriceModifier.getCommercePriceModifierId(),
			commercePriceModifier, serviceContext, workflowContext);
	}

	protected void validate(
			String title, String target, String modifierType,
			BigDecimal modifierAmount)
		throws PortalException {

		if (Validator.isNull(title)) {
			throw new CommercePriceModifierTitleException();
		}

		if (!CommercePriceModifierConstants.TARGET_CATALOG.equals(target) &&
			!CommercePriceModifierConstants.TARGET_CATEGORIES.equals(target) &&
			!CommercePriceModifierConstants.TARGET_PRODUCT_GROUPS.equals(
				target) &&
			!CommercePriceModifierConstants.TARGET_PRODUCTS.equals(target)) {

			throw new CommercePriceModifierTargetException();
		}

		CommercePriceModifierType commercePriceModifierType =
			_commercePriceModifierTypeRegistry.getCommercePriceModifierType(
				modifierType);

		if (commercePriceModifierType == null) {
			throw new CommercePriceModifierTypeException();
		}

		if (modifierAmount == null) {
			throw new CommercePriceModifierAmountException();
		}
	}

	protected void validateExternalReferenceCode(
			long companyId, String externalReferenceCode)
		throws PortalException {

		if (Validator.isNull(externalReferenceCode)) {
			return;
		}

		CommercePriceModifier commercePriceModifier =
			commercePriceModifierPersistence.fetchByC_ERC(
				companyId, externalReferenceCode);

		if (commercePriceModifier != null) {
			throw new DuplicateCommercePriceModifierException(
				"There is another commerce price modifier with external " +
					"reference code " + externalReferenceCode);
		}
	}

	private long[] _getAssetCategoryIds(long cpDefinitionId) {
		try {
			AssetEntry assetEntry = _assetEntryLocalService.getEntry(
				CPDefinition.class.getName(), cpDefinitionId);

			Set<AssetCategory> assetCategories = new HashSet<>();

			for (AssetCategory assetCategory : assetEntry.getCategories()) {
				assetCategories.add(assetCategory);
				assetCategories.addAll(assetCategory.getAncestors());
			}

			Stream<AssetCategory> stream = assetCategories.stream();

			LongStream longStream = stream.mapToLong(
				AssetCategory::getCategoryId);

			return longStream.toArray();
		}
		catch (PortalException pe) {
			_log.error(pe, pe);
		}

		return new long[0];
	}

	private static final Log _log = LogFactoryUtil.getLog(
		CommercePriceModifierLocalServiceImpl.class);

	@ServiceReference(type = AssetEntryLocalService.class)
	private AssetEntryLocalService _assetEntryLocalService;

	@ServiceReference(type = CommercePriceModifierTypeRegistry.class)
	private CommercePriceModifierTypeRegistry
		_commercePriceModifierTypeRegistry;

	@BeanReference(type = CommercePricingClassLocalService.class)
	private CommercePricingClassLocalService _commercePricingClassLocalService;

}