/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.checks;

import com.liferay.portal.kernel.util.StringUtil;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Hugo Huijser
 */
public class XMLEmptyLinesCheck extends EmptyLinesCheck {

	@Override
	public boolean isPortalCheck() {
		return true;
	}

	@Override
	protected String doProcess(
		String fileName, String absolutePath, String content) {

		if (fileName.matches(".*\\.(action|function|macro|testcase)") ||
			fileName.endsWith("/content.xml")) {

			return content;
		}

		content = fixEmptyLinesInMultiLineTags(content);

		content = fixEmptyLinesInNestedTags(content);

		content = _fixEmptyLinesBetweenTags(fileName, content);

		content = _fixMissingEmptyLinesAroundComments(content);

		Matcher matcher = _missingEmptyLinePattern.matcher(content);

		if (matcher.find()) {
			return StringUtil.replaceFirst(
				content, "\n", "\n\n", matcher.start());
		}

		matcher = _redundantEmptyLinePattern.matcher(content);

		if (matcher.find()) {
			return StringUtil.replaceFirst(
				content, "\n\n", "\n", matcher.start());
		}

		return content;
	}

	private String _fixEmptyLinesBetweenTags(String fileName, String content) {
		if (fileName.startsWith(getBaseDirName() + "build") ||
			fileName.matches(".*/(build|tools/).*")) {

			return content;
		}

		if (fileName.endsWith("-log4j-ext.xml") ||
			fileName.endsWith("-log4j.xml") ||
			fileName.endsWith("-logback.xml") ||
			fileName.endsWith("/ivy.xml") ||
			fileName.endsWith("/struts-config.xml") ||
			fileName.endsWith("/tiles-defs.xml")) {

			return fixEmptyLinesBetweenTags(content);
		}

		Matcher matcher = _emptyLineBetweenTagsPattern.matcher(content);

		if (matcher.find()) {
			return StringUtil.replaceFirst(
				content, "\n\n", "\n", matcher.end(1));
		}

		return content;
	}

	private String _fixMissingEmptyLinesAroundComments(String content) {
		Matcher matcher = _missingEmptyLineAfterCommentPattern.matcher(content);

		if (matcher.find()) {
			return StringUtil.replaceFirst(
				content, "-->\n", "-->\n\n", matcher.start());
		}

		matcher = _missingEmptyLineBeforeCommentPattern.matcher(content);

		if (matcher.find()) {
			return StringUtil.replaceFirst(
				content, ">\n", ">\n\n", matcher.start());
		}

		return content;
	}

	private final Pattern _emptyLineBetweenTagsPattern = Pattern.compile(
		"\n(\t*)<[\\w/].*[^-]>(\n\n)(\t*)<(\\w)");
	private final Pattern _missingEmptyLineAfterCommentPattern =
		Pattern.compile("[\t ]-->\n[\t<]");
	private final Pattern _missingEmptyLineBeforeCommentPattern =
		Pattern.compile(">\n\t+<!--[\n ]");
	private final Pattern _missingEmptyLinePattern = Pattern.compile(
		"<\\?xml .*\\?>\n<\\w");
	private final Pattern _redundantEmptyLinePattern = Pattern.compile(
		"<\\?xml .*\\?>\n\n<\\!DOCTYPE");

}