/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.checks;

import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.util.StringUtil;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Hugo Huijser
 */
public class JSPStylingCheck extends BaseFileCheck {

	@Override
	protected String doProcess(
		String fileName, String absolutePath, String content) {

		_checkChaining(fileName, content);

		_checkLineBreak(fileName, content);

		content = _fixEmptyJavaSourceTag(content);

		content = _fixIncorrectClosingTag(content);

		content = StringUtil.replace(
			content,
			new String[] {
				"alert('<%= LanguageUtil.", "alert(\"<%= LanguageUtil.",
				"confirm('<%= LanguageUtil.", "confirm(\"<%= LanguageUtil.",
				";;\n"
			},
			new String[] {
				"alert('<%= UnicodeLanguageUtil.",
				"alert(\"<%= UnicodeLanguageUtil.",
				"confirm('<%= UnicodeLanguageUtil.",
				"confirm(\"<%= UnicodeLanguageUtil.", ";\n"
			});

		int pos = content.indexOf("debugger.");

		if (pos != -1) {
			addMessage(
				fileName, "Do not use debugger", getLineCount(content, pos));
		}

		if (!fileName.endsWith("test.jsp")) {
			pos = content.indexOf("System.out.print");

			if (pos != -1) {
				addMessage(
					fileName, "Do not call 'System.out.print'",
					getLineCount(content, pos));
			}
		}

		return content;
	}

	private void _checkChaining(String fileName, String content) {
		Matcher matcher = _chainingPattern.matcher(content);

		if (matcher.find()) {
			addMessage(
				fileName, "Avoid chaining on 'getClass'", "chaining.markdown",
				getLineCount(content, matcher.start()));
		}
	}

	private void _checkLineBreak(String fileName, String content) {
		Matcher matcher = _incorrectLineBreakPattern.matcher(content);

		if (matcher.find()) {
			addMessage(
				fileName, "There should be a line break after '}'",
				getLineCount(content, matcher.start(1)));
		}
	}

	private String _fixEmptyJavaSourceTag(String content) {
		Matcher matcher = _emptyJavaSourceTagPattern.matcher(content);

		if (matcher.find()) {
			return StringUtil.replace(
				content, matcher.group(), StringPool.BLANK);
		}

		return content;
	}

	private String _fixIncorrectClosingTag(String content) {
		Matcher matcher = _incorrectClosingTagPattern.matcher(content);

		if (matcher.find()) {
			return StringUtil.replaceFirst(
				content, " />\n", "\n" + matcher.group(1) + "/>\n",
				matcher.end(1));
		}

		return content;
	}

	private final Pattern _chainingPattern = Pattern.compile(
		"\\WgetClass\\(\\)\\.");
	private final Pattern _emptyJavaSourceTagPattern = Pattern.compile(
		"\n\t*<%\n+\t*%>\n");
	private final Pattern _incorrectClosingTagPattern = Pattern.compile(
		"\n(\t*)\t((?!<\\w).)* />\n");
	private final Pattern _incorrectLineBreakPattern = Pattern.compile(
		"[\n\t]\\} ?(catch|else|finally) ");

}