/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of the Liferay Enterprise
 * Subscription License ("License"). You may not use this file except in
 * compliance with the License. You can obtain a copy of the License by
 * contacting Liferay, Inc. See the License for the specific language governing
 * permissions and limitations under the License, including but not limited to
 * distribution rights of the Software.
 *
 *
 *
 */

package com.liferay.source.formatter.checkstyle.checks;

import com.liferay.portal.kernel.util.ArrayUtil;
import com.liferay.portal.kernel.util.StringUtil;
import com.liferay.source.formatter.checkstyle.util.DetailASTUtil;

import com.puppycrawl.tools.checkstyle.api.DetailAST;
import com.puppycrawl.tools.checkstyle.api.TokenTypes;

/**
 * @author Hugo Huijser
 */
public class JSONNamingCheck extends BaseCheck {

	@Override
	public int[] getDefaultTokens() {
		return new int[] {
			TokenTypes.METHOD_DEF, TokenTypes.PARAMETER_DEF,
			TokenTypes.VARIABLE_DEF
		};
	}

	@Override
	protected void doVisitToken(DetailAST detailAST) {
		String typeName = DetailASTUtil.getTypeName(detailAST);

		if (typeName.equals("boolean") || typeName.equals("void")) {
			return;
		}

		String name = _getName(detailAST);
		String tokenTypeName = _getTokenTypeName(detailAST);

		_checkName(
			name, typeName, tokenTypeName, "String", "JSON", "Json",
			detailAST.getLineNo(), _TOKEN_TYPE_NAMES);
		_checkName(
			name, typeName, tokenTypeName, "JSONArray", "JSONArray",
			"JsonArray", detailAST.getLineNo(), _TOKEN_TYPE_NAMES);
		_checkName(
			name, typeName, tokenTypeName, "JSONObject", "JSONObject",
			"JsonObject", detailAST.getLineNo(), _TOKEN_TYPE_NAMES);

		_checkName(
			name, tokenTypeName, "JSON", "JSONString", detailAST.getLineNo(),
			new String[] {_TOKEN_TYPE_NAME_VARIABLE});
		_checkName(
			name, tokenTypeName, "JSON", "JsonString", detailAST.getLineNo(),
			new String[] {_TOKEN_TYPE_NAME_VARIABLE});
	}

	private void _checkName(
		String name, String tokenTypeName, String validNameEnding,
		String incorrectNameEnding, int lineNo, String[] checkTokenTypeNames) {

		String lowerCaseName = StringUtil.toLowerCase(name);

		if (!lowerCaseName.endsWith(
				StringUtil.toLowerCase(incorrectNameEnding))) {

			return;
		}

		if (name.endsWith(incorrectNameEnding) &&
			ArrayUtil.contains(checkTokenTypeNames, tokenTypeName)) {

			log(
				lineNo, _MSG_RENAME_VARIABLE,
				StringUtil.toLowerCase(tokenTypeName), name,
				StringUtil.replaceLast(
					name, incorrectNameEnding, validNameEnding));
		}
	}

	private void _checkName(
		String name, String typeName, String tokenTypeName, String type,
		String reservedNameEnding, String incorrectNameEnding, int lineNo,
		String[] checkTokenTypeNames) {

		String lowerCaseName = StringUtil.toLowerCase(name);

		if (!lowerCaseName.endsWith(
				StringUtil.toLowerCase(incorrectNameEnding))) {

			return;
		}

		if (!typeName.equals(type) && !typeName.endsWith("." + type)) {
			String lowerCaseTypeName = StringUtil.toLowerCase(typeName);

			if (!lowerCaseTypeName.endsWith(StringUtil.toLowerCase(type))) {
				log(
					lineNo, _MSG_RESERVED_VARIABLE_NAME, tokenTypeName,
					reservedNameEnding, type);
			}

			return;
		}

		_checkName(
			name, tokenTypeName, reservedNameEnding, incorrectNameEnding,
			lineNo, checkTokenTypeNames);
	}

	private String _getName(DetailAST detailAST) {
		DetailAST nameAST = detailAST.findFirstToken(TokenTypes.IDENT);

		return nameAST.getText();
	}

	private String _getTokenTypeName(DetailAST detailAST) {
		if (detailAST.getType() == TokenTypes.METHOD_DEF) {
			return _TOKEN_TYPE_NAME_METHOD;
		}

		if (detailAST.getType() == TokenTypes.PARAMETER_DEF) {
			return _TOKEN_TYPE_NAME_PARAMETER;
		}

		return _TOKEN_TYPE_NAME_VARIABLE;
	}

	private static final String _MSG_RENAME_VARIABLE = "variable.rename";

	private static final String _MSG_RESERVED_VARIABLE_NAME =
		"variable.name.reserved";

	private static final String _TOKEN_TYPE_NAME_METHOD = "Method";

	private static final String _TOKEN_TYPE_NAME_PARAMETER = "Parameter";

	private static final String _TOKEN_TYPE_NAME_VARIABLE = "Variable";

	private static final String[] _TOKEN_TYPE_NAMES = {
		_TOKEN_TYPE_NAME_METHOD, _TOKEN_TYPE_NAME_PARAMETER,
		_TOKEN_TYPE_NAME_VARIABLE
	};

}